/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Patrick Sager
 */

/**
 * API Status: **Private**
 * @module nmodule/ldap/rc/BasicKrb5Conf
 */
define(['jquery',
  'underscore',
  'Promise',
  'baja!',
  'lex!ldap',
  'nmodule/webEditors/rc/fe/baja/util/compUtils',
  'nmodule/webEditors/rc/fe/CompositeEditor',
  'nmodule/webEditors/rc/fe/config/CompositeBuilder',
  'bajaux/util/SaveCommand',
  'hbs!nmodule/ldap/rc/template/BasicKrb5ConfEditor-structure',
  'css!nmodule/ldap/rc/ldap'], function (
  $,
  _,
  Promise,
  baja,
  lexs,
  compUtils,
  CompositeEditor,
  CompositeBuilder,
  SaveCommand,
  tplBasicKrb5ConfEditorStructure) {

  'use strict';

  var second,
    millisecond,
    lex = lexs[0],
    canWriteSlot = compUtils.canWriteSlot;

  /**
   * An editor for the krb5.conf file. Shows the file's configurable properties in a property sheet
   * style table and writes the results to the file in the correct format.
   *
   * @class
   * @extends module:bajaux/Widget
   * @alias module:nmodule/ldap/rc/BasicKrb5Conf
   */
  var BasicKrb5ConfEditor = function BasicKrb5ConfEditor(params) {
    CompositeEditor.call(this, {moduleName: 'ldap', keyName: 'BasicKrb5ConfEditor'});
  };

  //extend and set up prototype chain
  BasicKrb5ConfEditor.prototype = Object.create(CompositeEditor.prototype);
  BasicKrb5ConfEditor.prototype.constructor = BasicKrb5ConfEditor;

  function getProperties() {
    return {
      'forwardable': {type: Boolean, default: false, value: false},
      'kdc_timeout': {type: baja.Integer, default: baja.Integer.make(30), value: baja.Integer.make(30),
        displayUnits: second, saveUnits: millisecond},
      'max_retries': {type: baja.Integer, default: baja.Integer.make(3), value: baja.Integer.make(3)}
    };
  }

  /**
   * Read the krb5.conf file using rpc call
   *
   * @inner
   * @private
   *
   * @return {Promise} A promise that resolves with the file's contents.
   */
  function readKrb5Conf() {
    return baja.rpc("type:ldap:KerberosConfig", "readKrb5Conf");
  }

  /**
   * Write the krb5.conf file using rpc call
   *
   * @inner
   * @private
   *
   * @param {String} contents The contents of the file.
   * @return {Promise} A promise that resolves when the file has been written.
   */
  function writeKrb5Conf(contents) {
    return baja.rpc("type:ldap:KerberosConfig", "writeKrb5Conf", contents);
  }

  /**
   * Parse a string value into a baja value of the specified type
   *
   * @private
   * @param {string} value string representation of the value
   * @param {function} type the type of the value.
   * @returns {*} the baja representation of the value
   */
  function parseValue(value, type) {
    if (type === baja.Integer) {
      if (! /^(\+|\-)?((0x[\da-fA-F]+)|(\d+))$/.test(value)) {
        throw new Error("Integer required");
      }
      if (value.indexOf('0x') === 0) {
        return baja.Integer.make(parseInt(value, 16));
      } else {
        return baja.Integer.make(parseInt(value, 10));
      }
    } else if (type === Boolean) {
      if (value === "true") {
        return true;
      } else if (value === "false") {
        return false;
      } else {
        throw new Error("boolean required");
      }
    }
  }

  /**
   * Creates the table with rows for child editors
   *
   * @param {jQuery} dom
   * @returns {promise}
   */
  BasicKrb5ConfEditor.prototype.doInitialize = function (dom) {
    var that = this;
    var args = arguments;

    return baja.UnitDatabase.get()
      .then(function (db) {
        second = db.getUnit('second');
        millisecond = db.getUnit('millisecond');
        that.configurableProperties = getProperties();

        var rows = _.map(that.getBuilder().getKeys(), function(key) {
          return {
            key: key,
            name: that.getBuilder().getDisplayNameFor(key)
          };
        });
        dom.html(tplBasicKrb5ConfEditorStructure({
          rows: rows
        })).addClass('BasicKrb5ConfEditor ux-fullscreen');
        return CompositeEditor.prototype.doInitialize.apply(that, args);
      });
  };

  /**
   * Reads the krb5.conf file and loads the configurable configurableProperties into the editors
   *
   * @param {baja.Complex} value, only used for permissions checks.
   * @returns {promise}
   */
  BasicKrb5ConfEditor.prototype.doLoad = function (value) {
      var that = this,
        readonly = value.getParent() ? !canWriteSlot(value.getParent(), value.getName())
          : !value.getPermissions().hasOperatorWrite();
      if (!readonly) {
        this.getCommandGroup().add(new SaveCommand());
      }
      return Promise.join(readKrb5Conf().then(function (contents) {
          that.parseKrb5Conf(contents);
          return CompositeEditor.prototype.doLoad.apply(that, arguments);
      }), this.setReadonly(readonly));
  };

  /**
   * Returns the CompositeBuilder for this editor
   *
   * @returns {CompositeBuilder}
   */
  BasicKrb5ConfEditor.prototype.makeBuilder = function () {
    var that = this;
    var builder = new CompositeBuilder();

    builder.getDomFor = function (key) {
      var dom = $('.BasicKrb5ConfEditor-table-container > table > tbody > .key-' + key + ' > .col-value', that.jq());
      dom.on('bajaux:modify', function () {
        $('.BasicKrb5ConfEditor-table-container > table > tbody > .key-' + key, that.jq()).toggleClass('modified', true);
      });
      return dom;
    };

    builder.getKeys = function () {
      return _.keys(that.configurableProperties);
    };

    builder.getDisplayNameFor = function (key) {
      return lex.get('ldap.simpleKrb5ConfEditor.' + key);
    };

    builder.getValueFor = function (key) {
      return that.configurableProperties[key].value;
    };

    builder.getConfigFor = function (key) {
      return {
        properties: {units: that.configurableProperties[key].displayUnits},
        formFactor: 'mini'
      };
    };

    return builder;
  };

  /**
   * Save child editors and write their values to the krb5.conf file.
   *
   * @returns {Promise}
   */
  BasicKrb5ConfEditor.prototype.doSave = function () {
    var that = this;
    return Promise.all(_.map(this.configurableProperties, function (prop, key) {
      var ed = that.getBuilder().getEditorFor(key);
      return ed.read().then(function (value) {
        prop.value = value;
        $('.BasicKrb5ConfEditor-table-container > table > tbody > .key-' + key, that.jq()).toggleClass('modified', false);
      });
    })).then(function () {
      return that.saveFile();
    });
  };

  /**
   * Save the editors' values to the krb5.conf file.
   *
   * @private
   * @returns {Promise}
   */
  BasicKrb5ConfEditor.prototype.saveFile = function() {
    var that = this,
      written = [],
      inLibdef = false,
      lastLibdefLine = -1; // index of the last non-blank line in libdefaults section

    for (var i = 0; i < this.lines.length; i++) {
      var line = this.lines[i];
      var trimmed = line.trim();
      if (inLibdef) {
        if (_.first(trimmed) === '[' && _.last(trimmed) === ']') {
          inLibdef = false;
          break;
        }
        if (_.first(line) === '#' || !trimmed) {
          continue;
        }
        lastLibdefLine = i;
        var equalIndex = line.indexOf('=');
        if (equalIndex !== -1) {
          var key = line.substring(0, equalIndex).trim();
          var value = line.substring(equalIndex + 1, line.length).trim();
          var prop = this.configurableProperties[key];

          if (prop) {
            var bvalue = parseValue(value, prop.type);
            if (prop.displayUnits && prop.saveUnits) {
              bvalue = bvalue.make(prop.saveUnits.convertTo(prop.displayUnits, bvalue));
            }
            if (!bvalue.equals(prop.value)) {
              // preserve original whitespace
              var saveValue = prop.value;
              if (prop.displayUnits && prop.saveUnits) {
                saveValue = saveValue.make(prop.displayUnits.convertTo(prop.saveUnits, saveValue));
              }
              this.lines[i] = line.substring(0, equalIndex + 1) + line.substring(equalIndex +1, line.length).replace(value, saveValue);
            }
            written.push(key);
          }
        }
      }
      else if (trimmed === "[libdefaults]") {
        inLibdef = true;
        lastLibdefLine = i;
      }
      // we only care about libdefaults, skip over everything else
    }

    var unWritten = _.filter(_.difference(_.keys(this.configurableProperties), written), function (key) {
      var prop = that.configurableProperties[key];
      return ! prop.default.equals(prop.value);
    });

    if (unWritten.length > 0 && lastLibdefLine === -1) {
      if (this.lines.length === 1 && this.lines[0].trim().length === 0) {
        this.lines[0] = '[libdefaults]';
      } else {
        this.lines.splice(0, 0, '[libdefaults]', '');
      }
      lastLibdefLine = 0;
    }

    for (var j = 0; j < unWritten.length; j++) {
      var unwrittenKey = unWritten[j],
        unwrittenProp = this.configurableProperties[unWritten[j]],
        unwrittenSaveValue = unwrittenProp.value;

      if (unwrittenProp.displayUnits && unwrittenProp.saveUnits) {
        unwrittenSaveValue = unwrittenSaveValue.make(unwrittenProp.displayUnits.convertTo(unwrittenProp.saveUnits, unwrittenSaveValue));
      }
      this.lines.splice(lastLibdefLine + j + 1, 0, unwrittenKey + ' = ' + unwrittenSaveValue);
    }

    return writeKrb5Conf(this.lines.join("\n"));
  };

  /**
   * Parse the contents of a krb5.conf file. Store each line in the this.lines array and store the
   * value of any supported properties in the configurableProperties object
   *
   * @private
   * @param {String} content the contents of the file.
   */
  BasicKrb5ConfEditor.prototype.parseKrb5Conf = function(content) {
    this.lines = content.split("\n");
    var inLibdef = false;

    for (var i = 0; i < this.lines.length; i++) {
      var line = this.lines[i],
        trimmed = line.trim();

      if (inLibdef === true) {
        if (trimmed.charAt(0) === '[' && trimmed.charAt(trimmed.length - 1) === ']') {
          inLibdef = false;
          break;
        }
        var equalIndex = line.indexOf('=');
        if (equalIndex !== -1) {
          var key = line.substring(0, equalIndex).trim(),
            value = line.substring(equalIndex + 1, line.length).trim(),
            prop = this.configurableProperties[key];

          if (prop !== undefined) {
            var bvalue = parseValue(value, prop.type);

            if (prop.displayUnits && prop.saveUnits) {
              bvalue = bvalue.make(prop.saveUnits.convertTo(prop.displayUnits, bvalue));
            }
            prop.value = bvalue;
          }
        }
      }
      else if (line.trim() === "[libdefaults]") {
        inLibdef = true;
      }
      // we only care about libdefaults, skip over everything else
    }
  };

  return BasicKrb5ConfEditor;
});
