/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Patrick Sager
 */

/**
 * API Status: **Private**
 * @module nmodule/ldap/rc/AdvancedKrb5Conf
 */
define(['bajaux/Widget',
    'jquery',
    'Promise',
    'baja!',
    'nmodule/webEditors/rc/fe/baja/util/compUtils',
    'nmodule/webEditors/rc/wb/file/TextFileEditor',
    'css!nmodule/ldap/rc/ldap'], function (
    Widget,
    $,
    Promise,
    baja,
    compUtils,
    TextFileEditor) {

  'use strict';

  var canWriteSlot = compUtils.canWriteSlot;

  /**
   * An editor for the krb5.conf file. Shows a plain text editor for the file.
   *
   * @class
   * @extends module:webEditors/wb/file/TextFileEditor'
   * @alias module:nmodule/ldap/rc/AdvancedKrb5Conf
   */
  var AdvancedKrb5ConfEditor = function AdvancedKrb5ConfEditor() {
    TextFileEditor.apply(this, arguments);
  };

  //extend and set up prototype chain
  AdvancedKrb5ConfEditor.prototype = Object.create(TextFileEditor.prototype);
  AdvancedKrb5ConfEditor.prototype.constructor = AdvancedKrb5ConfEditor;

  /**
   * Read the krb5.conf file using rpc call
   *
   * @inner
   * @private
   *
   * @return {Promise} A promise that resolves with the file's contents.
   */
  function readKrb5Conf() {
    return baja.rpc("type:ldap:KerberosConfig", "readKrb5Conf");
  }


  /**
   * Write the krb5.conf file using rpc call
   *
   * @inner
   * @private
   *
   * @param {String} contents The contents of the file.
   * @return {Promise} A promise that resolves when the file has been written.
   */
  function writeKrb5Conf(contents) {
    return baja.rpc("type:ldap:KerberosConfig", "writeKrb5Conf", contents);
  }

  /**
   * Reads the krb5.conf file and loads the text editor with the contents.
   *
   * @param value
   * @returns {promise}
   */
  AdvancedKrb5ConfEditor.prototype.doLoad = function (value) {
    var that = this,
      readonly = value.getParent() ? !canWriteSlot(value.getParent(), value.getName())
        : !value.getPermissions().hasOperatorWrite();

    if (! readonly) {
      that.$addSaveCmd();
    }

    that.$initEditor("text");
    that.$setEditorReadOnly(readonly);

    // If something in the editor changes then mark the widget as modified.
    that.$onEditorChange(function () {
      that.setModified(true);
    });

    return Promise.join(readKrb5Conf()
      .then(function (text) {
        that.$setEditorText(text);
      }), that.setReadonly(readonly));
  };

  /**
   * Gets the text from the editor and saves it to the krb5.conf file.
   *
   * @returns {Promise}
   */
  AdvancedKrb5ConfEditor.prototype.doSave = function () {
    var that = this;

    return !that.$editor || this.isReadonly() ? Promise.resolve()
      : writeKrb5Conf(that.$getEditorText());
  };

  return AdvancedKrb5ConfEditor;
});
