/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Ashutosh Chaturvedi
 */

/* eslint-env browser */

/**
 * API Status: **Private**
 * @module nmodule/cloudLink/rc/history/mgr/CloudHistoryExportUxManager
 */
define(['baja!', 'jquery', 'Promise', 'underscore', 'lex!cloudLink', 'nmodule/webEditors/rc/fe/fe', 'nmodule/cloudLink/rc/history/model/CloudHistoryExportLearnModel', 'nmodule/cloudLink/rc/history/model/CloudHistoryExportMgrModel', 'nmodule/cloudLink/rc/history/util/CloudHistoryExportAssignCommand', 'nmodule/cloudLink/rc/history/util/CloudHistoryExportUnassignCommand', 'nmodule/cloudLink/rc/history/util/CloudHistoryExportExecuteCommand', 'nmodule/cloudLink/rc/history/util/HistoryIdAwareTreeNode', 'nmodule/webEditors/rc/wb/mgr/Manager', 'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber', 'nmodule/webEditors/rc/wb/mgr/commands/NewCommand', 'nmodule/webEditors/rc/wb/mgr/commands/EditCommand', 'nmodule/webEditors/rc/wb/commands/DeleteCommand', 'nmodule/webEditors/rc/wb/mgr/commands/AddCommand', 'nmodule/webEditors/rc/wb/mgr/commands/CancelDiscoverCommand', 'nmodule/webEditors/rc/wb/mgr/commands/LearnModeCommand', 'nmodule/webEditors/rc/wb/mgr/commands/DiscoverCommand', 'nmodule/webEditors/rc/wb/mgr/commands/MatchCommand', 'bajaux/commands/CommandGroup', 'nmodule/webEditors/rc/wb/mixin/ContextMenuSupport', 'nmodule/webEditors/rc/wb/mgr/MgrLearn', 'bajaux/mixin/subscriberMixIn', 'nmodule/webEditors/rc/wb/mgr/mgrUtils', 'nmodule/webEditors/rc/wb/mixin/AttachableSupport', 'bajaux/events', 'hbs!nmodule/cloudLink/rc/history/mgr/HistoryConfigTemplate', 'css!nmodule/cloudLink/rc/history/mgr/HistoryConfigStyle'], function (baja, $, Promise, _, lexicons, fe, CloudHistoryExportLearnModel, CloudHistoryExportMgrModel, CloudHistoryExportAssignCommand, CloudHistoryExportUnassignCommand, CloudHistoryExportExecuteCommand, HistoryIdAwareTreeNode, Manager, DepthSubscriber, NewCommand, EditCommand, DeleteCommand, AddCommand, CancelDiscoverCommand, LearnModeCommand, DiscoverCommand, MatchCommand, CommandGroup, addContextMenuSupport, addLearnSupport, subscribable, mgrUtils, addAttachableSupport, events, historyConfigTemplate) {
  'use strict';
  /**
   * CloudHistoryExportUxManager: The Manager view on the CloudHistoryExport
   * Container added under the histories channel on the CloudLink.
   * The view provide ways to add/modify/delete CloudExportConfig attached to
   * the histories.
   * @private
   * @class
   * @extends module:nmodule/webEditors/rc/wb/mgr/Manager
   * @alias module:nmodule/cloudLink/rc/history/mgr/CloudHistoryExportUxManager
   */

  var CloudHistoryExportUxManager = function CloudHistoryExportUxManager(params) {
    this.configIdsMap = [];
    this.autoExportConfig = null;
    this.idsExcludedByManager = [];
    this.idsToOmit = [];
    Manager.call(this, {
      params: params,
      defaults: widgetDefaults()
    }); // These constants must be identical to the constants in the BCloudHistoryAutoExportConfig Java class

    this.NO_EXCLUSIONS = 0;
    this.EXCLUDED_BY_TAG = 1;
    this.EXCLUDED_BY_MANAGER = 2;
    this.EXCLUDED_NO_CLOUDID = 4;
    subscribable(this);
    this.$subscriber = new DepthSubscriber(3);
    addContextMenuSupport(this);
    addLearnSupport(this);
    addAttachableSupport(this);
    var that = this;
    this.$subscriber.attach('changed', function (property) {
      if (property.getName() === 'enabled' && this.getType().is('cloudLink:CloudHistoryExportConfig')) {
        that.notifyExecuteCommand();
      }
    });
  };

  CloudHistoryExportUxManager.prototype = Object.create(Manager.prototype);
  CloudHistoryExportUxManager.prototype.constructor = CloudHistoryExportUxManager;

  var widgetDefaults = function widgetDefaults() {
    return {
      moduleName: 'cloudLink',
      keyName: 'CloudHistoryExportUxManager',
      properties: {
        rootCssClass: '-t-CloudHistoryExportUxManager'
      }
    };
  };

  var HISTORY_ALL_DEVICES_NAV_NAME = 'history:///'; ////////////////////////////////////////////////////////////////
  // Main Table
  ////////////////////////////////////////////////////////////////

  /**
   * Makes the model for the main database table.
   *
   * @param {baja.Component} component - a CloudHistoryExportContainer
   * @returns {Promise.<module:nmodule/cloudLink/rc/history/model/CloudHistoryExportMgrModel>}
   */

  CloudHistoryExportUxManager.prototype.makeModel = function (component) {
    return CloudHistoryExportMgrModel.make(component);
  };
  /**
   * Subscribe to the Cloud History Export Container and Load the cloud history
   * config into the widget. This will set up a handler to watch the main table
   * selection, and enable or disable the Commands based on the presence of
   * at least one selected config.
   * @param {baja.Component} component - a CloudHistoryExportContainer
   * @returns {Promise}
   */


  CloudHistoryExportUxManager.prototype.doLoad = function (component) {
    var that = this;
    return Manager.prototype.doLoad.apply(that, arguments).then(function () {
      return updateHistoryIdsToNewFormat(that);
    }).then(function (updateHistoryIdFormatPromises) {
      return Promise.all(updateHistoryIdFormatPromises);
    }).then(function () {
      var mainSelection = mgrUtils.getMainTableSelection(that),
          editCmd = that.$getEditCommand(),
          learnModeCmd = that.$getLearnModeCommand();
      mainSelection.on('changed', function () {
        editCmd.setEnabled(!mainSelection.isEmpty());
        that.notifyExecuteCommand();
        that.notifyAssignCommand();
        that.notifyUnassignCommand();
      });
      learnModeCmd.on(events.command.SELECTION_EVENT, function () {
        var state = !learnModeCmd.isSelected();
        that.notifyAssignCommand(state);
        that.notifyUnassignCommand(state);
      });
    });
  };

  function updateHistoryIdsToNewFormat(mgr) {
    var updateHistoryIdFormatPromises = [];
    var rows = mgr.getModel().getRows();

    for (var rowIdx = 0; rowIdx < rows.length; rowIdx++) {
      var row = rows[rowIdx].getSubject();

      if (baja.hasType(row, 'cloudLink:CloudHistoryAutoExportConfig')) {
        _.each(row.getExcludedHistoryIds().getSlots().toValueMap(), function (val, key) {
          if (baja.hasType(val, 'history:HistoryId')) {
            updateHistoryIdFormatPromises.push(row.getExcludedHistoryIds().remove(key).then(function () {
              return row.getExcludedHistoryIds().add({
                slot: baja.SlotPath.escape(val.encodeToString()),
                value: baja.$('baja:Integer', mgr.EXCLUDED_BY_MANAGER),
                flags: baja.Flags.READONLY
              });
            }));
          }
        });

        break;
      }
    }

    return updateHistoryIdFormatPromises;
  }
  /**
   * Notify the assign command of a selection change, so it can decide whether
   * to enable itself based on what is currently selected.
   *
   * @param {Boolean} learnModeSelectionState boolean value, selection state of the learn mode
   */


  CloudHistoryExportUxManager.prototype.notifyAssignCommand = function (learnModeSelectionState) {
    var assignCmd = this.$getAssignCommand();

    if (assignCmd) {
      assignCmd.tableSelectionChanged(learnModeSelectionState);
    }
  };
  /**
   * Notify the unassign command of a selection change, so it can decide whether
   * to enable itself based on what is currently selected.
   * It also gets called after assign command execution.
   *
   * @param {Boolean} learnModeSelectionState boolean value, selection state of the learn mode
   */


  CloudHistoryExportUxManager.prototype.notifyUnassignCommand = function (learnModeSelectionState) {
    var unassignCmd = this.$getUnassignCommand();

    if (unassignCmd) {
      unassignCmd.tableSelectionChanged(learnModeSelectionState);
    }
  };
  /**
   * Notify the execute command of a selection change, so it can decide whether
   * to enable itself based on what is currently selected.
   */


  CloudHistoryExportUxManager.prototype.notifyExecuteCommand = function () {
    var executeCmd = this.$getExecuteCommand();

    if (executeCmd) {
      executeCmd.updateExecuteButton();
    }
  };
  /**
   * This function populates an array with keys as the CloudHistoryExportConfig
   * and the corresponding mapped HistoryIds. The main database table is traversed
   * to populate the array. It also stores the historyIds that have been excluded
   * either by tag or by the manager.
   *
   * @param {module:nmodule/cloudLink/rc/history/mgr/CloudHistoryExportUxManager} manager
   */


  function populateCloudHistoryExportConfigVsHistoryIdsMap(manager) {
    manager.configIdsMap = [];
    manager.autoExportConfig = null;
    manager.idsExcludedByManager = [];
    manager.idsToOmit = [];
    var rows = manager.getModel().getRows();

    for (var rowIdx = 0; rowIdx < rows.length; rowIdx++) {
      var row = rows[rowIdx].getSubject();

      if (baja.hasType(row, "cloudLink:CloudHistoryAutoExportConfig")) {
        manager.autoExportConfig = row;
        manager.idsExcludedByManager = _.map(_.pick(row.getExcludedHistoryIds().getSlots().is('baja:Integer').toValueMap(), function (val) {
          return manager.isExcludedForReason(val, manager.EXCLUDED_BY_MANAGER);
        }), function (val, key) {
          return baja.SlotPath.unescape(key);
        });
        manager.idsToOmit = _.map(_.pick(row.getExcludedHistoryIds().getSlots().is('baja:Integer').toValueMap(), function (val) {
          return manager.isExcludedForReason(val, manager.EXCLUDED_BY_TAG) || manager.isExcludedForReason(val, manager.EXCLUDED_NO_CLOUDID);
        }), function (val, key) {
          return baja.SlotPath.unescape(key);
        });
      } else {
        manager.configIdsMap.push({
          config: row,
          historyIds: row.getHistoryIds().getSlots().is('history:HistoryId').toValueArray().map(function (historyId) {
            return historyId.encodeToString();
          })
        });
      }
    }
  }
  /**
   * This function takes a historyId and searches for the CloudHistoryExportConfig it is mapped to, if any.
   * It uses the `configIdsMap` populated in {@link #populateCloudHistoryExportConfigVsHistoryIdsMap}
   *
   * @param {String} historyId The historyId.
   * @returns {String} The CloudHistoryExportConfig mapped to the historyId, if any.
   */


  CloudHistoryExportUxManager.prototype.getCloudHistoryExportConfigFromHistoryId = function (historyId) {
    for (var i = 0; i < this.configIdsMap.length; i++) {
      var configRow = this.configIdsMap[i],
          config = configRow.config,
          historyIds = configRow.historyIds;

      if (historyIds.includes(historyId)) {
        return config;
      }
    }
  };
  /**
   * This function takes a historyId and checks if it has been excluded by the manager.
   *
   * @param {String} historyId The historyId.
   * @returns {String} The CloudHistoryAutoExportConfig if the historyId is in the excludedHistoryIds vector.
   */


  CloudHistoryExportUxManager.prototype.getCloudHistoryAutoExportConfigFromHistoryId = function (historyId) {
    if (this.idsExcludedByManager.includes(historyId)) {
      return this.autoExportConfig;
    }
  }; ////////////////////////////////////////////////////////////////
  // Learn Table
  ////////////////////////////////////////////////////////////////

  /**
   * Client side Ord resolution to find all the histories in the history space.
   * Each of the history point is converted into a HistoryIdAwareTreeNode.
   *
   * @returns {Promise.<HistoryIdAwareTreeNode[]>} TreeNode with history id
   */


  function makeHistoryNavTreeNode(mgr) {
    return baja.Ord.make("history:").get().then(function (historySpace) {
      return historySpace.getNavChildren(); //force the history space to get navEvents on its children
    }).then(function (children) {
      var allDeviceHistoryFolder = _.find(children, function (node) {
        return node.getNavName() === HISTORY_ALL_DEVICES_NAV_NAME;
      }); //if allDeviceHistoryFolder exists, make sure to call `getNavChildren()` so we can get the navEvents for it


      return allDeviceHistoryFolder && allDeviceHistoryFolder.getNavChildren() || children;
    }).then(function (kids) {
      return kids.map(function (node) {
        return new HistoryIdAwareTreeNode(node, mgr.idsToOmit);
      });
    });
  }
  /**
   * Reload the learn table from the manager's discovered configs.
   *
   * @param {module:nmodule/cloudLink/rc/history/mgr/CloudHistoryExportUxManager} mgr - the
   * Cloud History Export Manager
   * @returns {Promise}
   */


  function reloadLearnTable(mgr) {
    var model = mgr.getLearnModel(),
        currentCount = model.getRows().length,
        root = model.getRootNode();
    return makeHistoryNavTreeNode(mgr).then(function (nodes) {
      return root.getKids().then(function (kids) {
        return Promise.all(kids.map(function (kid) {
          return root.remove(kid);
        }));
      }).then(function () {
        return Promise.all(nodes.map(function (node) {
          return root.add(node);
        }));
      }).then(function () {
        var promise = currentCount ? model.removeRows(0, currentCount) : Promise.resolve();
        return promise.then(function () {
          return model.insertRows(nodes, 0);
        });
      });
    });
  }

  CloudHistoryExportUxManager.prototype.makeLearnModel = function () {
    return CloudHistoryExportLearnModel.make(this);
  }; ////////////////////////////////////////////////////////////////
  // Discovery
  ////////////////////////////////////////////////////////////////

  /**
   * Start the discovery job. This will get the discovered
   * items from the Ord resolution at the client side and reload the table.
   *
   * @private
   */


  CloudHistoryExportUxManager.prototype.doDiscover = function () {
    var that = this,
        learnSelection = mgrUtils.getLearnTableSelection(that); // make assign command aware of any learn model value selection

    learnSelection.on('changed', function () {
      that.notifyUnassignCommand();
      that.notifyAssignCommand();
    });
    populateCloudHistoryExportConfigVsHistoryIdsMap(that);
    return reloadLearnTable(that);
  };
  /**
   * Returns true if the provided exclusion value includes the provided exclusion reason.
   * @param exclusion The exclusion value to test.
   * @param exclusionReason The exclusion reason.
   * @returns true if the provided exclusion value includes the provided exclusion reason,
   * false otherwise
   */


  CloudHistoryExportUxManager.prototype.isExcludedForReason = function (exclusion, exclusionReason) {
    return isValidExclusionReason(this, exclusion) && isValidExclusionReason(this, exclusionReason) && (exclusion & exclusionReason) !== this.NO_EXCLUSIONS;
  };
  /**
   * Returns true if the provided exclusion value includes only the provided exclusion reason.
   * @param exclusion The exclusion value to test.
   * @param exclusionReason The exclusion reason.
   * @return true if the provided exclusion value includes only the provided exclusion reason,
   * false otherwise
   */


  CloudHistoryExportUxManager.prototype.isExcludedOnlyForReason = function (exclusion, exclusionReason) {
    return isValidExclusionReason(this, exclusion) && isValidExclusionReason(this, exclusionReason) && exclusion === exclusionReason;
  };
  /**
   * Return the result of including the provided exclusion reason in the provided exclusion value.
   * @param exclusion The exclusion value.
   * @param exclusionReasonToAdd The exclusion reason.
   * @return The result of including the exclusion reason in the excluded value.
   */


  CloudHistoryExportUxManager.prototype.addReasonForExclusion = function (exclusion, exclusionReasonToAdd) {
    if (!isValidExclusionReason(this, exclusion)) {
      exclusion = this.NO_EXCLUSIONS;
    }

    if (!isValidExclusionReason(this, exclusionReasonToAdd)) {
      exclusionReasonToAdd = this.NO_EXCLUSIONS;
    }

    return exclusion | exclusionReasonToAdd;
  };
  /**
   * Return the result of removing the provided exclusion reason from the provided exclusion value.
   * @param exclusion The exclusion value.
   * @param exclusionReasonToRemove The exclusion reason.
   * @return The result of removing the exclusion reason from the excluded value.
   */


  CloudHistoryExportUxManager.prototype.removeReasonForExclusion = function (exclusion, exclusionReasonToRemove) {
    if (!isValidExclusionReason(this, exclusion)) {
      exclusion = this.NO_EXCLUSIONS;
    }

    if (!isValidExclusionReason(this, exclusionReasonToRemove)) {
      exclusionReasonToRemove = this.NO_EXCLUSIONS;
    }

    return exclusion & ~exclusionReasonToRemove;
  };

  function isValidExclusionReason(mgr, exclusionReason) {
    return exclusionReason > 0 && exclusionReason <= (mgr.EXCLUDED_BY_MANAGER | mgr.EXCLUDED_BY_TAG | mgr.EXCLUDED_NO_CLOUDID);
  } ////////////////////////////////////////////////////////////////
  // Menu and Commands
  ////////////////////////////////////////////////////////////////


  CloudHistoryExportUxManager.prototype.makeCommands = function () {
    var cmds = [],
        newCommand = new NewCommand(this),
        editCommand = new EditCommand(this),
        assignCommand = new CloudHistoryExportAssignCommand(this),
        unassignCommand = new CloudHistoryExportUnassignCommand(this),
        executeCommand = new CloudHistoryExportExecuteCommand(this);
    cmds.push(newCommand);
    cmds.push(editCommand);
    cmds = cmds.concat(this.makeDiscoveryCommands());
    cmds.push(assignCommand);
    cmds.push(unassignCommand);
    cmds.push(executeCommand);
    return _.filter(cmds, function (cmd) {
      return !(cmd instanceof CancelDiscoverCommand || cmd instanceof AddCommand || cmd instanceof MatchCommand);
    });
  };
  /**
   * Returns a reference to the Edit command
   * @private
   * @returns {Command}
   */


  CloudHistoryExportUxManager.prototype.$getEditCommand = function () {
    return mgrUtils.findCommand(this, EditCommand);
  };
  /**
   * Return the 'CloudHistoryExportAssignCommand' instance from the command group.
   * This can be used to update an existing database item from a discovered item.
   *
   * @private
   * @returns {module:nmodule/cloudLink/rc/history/util/CloudHistoryExportAssignCommand}
   */


  CloudHistoryExportUxManager.prototype.$getAssignCommand = function () {
    return mgrUtils.findCommand(this, CloudHistoryExportAssignCommand);
  };
  /**
   * Return the 'CloudHistoryExportUnassignCommand' instance from the command group.
   * This can be used to update an existing database item from a discovered item.
   *
   * @private
   * @returns {module:nmodule/cloudLink/rc/history/util/CloudHistoryExportUnassignCommand}
   */


  CloudHistoryExportUxManager.prototype.$getUnassignCommand = function () {
    return mgrUtils.findCommand(this, CloudHistoryExportUnassignCommand);
  };
  /**
   * Return the 'CloudHistoryExportExecuteCommand' instance from the command group.
   * This can be used to update an existing database item from a discovered item.
   *
   * @private
   * @returns {module:nmodule/cloudLink/rc/history/util/CloudHistoryExportExecuteCommand}
   */


  CloudHistoryExportUxManager.prototype.$getExecuteCommand = function () {
    return mgrUtils.findCommand(this, CloudHistoryExportExecuteCommand);
  };
  /**
   * Return the 'LearnModeCommand' instance from the command group.
   * This is used to show/hide discovery pane related commands.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/mgr/commands/LearnModeCommand}
   */


  CloudHistoryExportUxManager.prototype.$getLearnModeCommand = function () {
    return mgrUtils.findCommand(this, LearnModeCommand);
  };

  CloudHistoryExportUxManager.prototype.setAssignCommandEnabled = function (enabled) {
    this.$getAssignCommand().setEnabled(enabled);
  };

  CloudHistoryExportUxManager.prototype.setUnassignCommandEnabled = function (enabled) {
    this.$getUnassignCommand().setEnabled(enabled);
  };
  /**
   * Initial DOM setup
   *
   * @param {jQuery} dom the DOM element into which to load this View
   * @return {Promise}
   */


  CloudHistoryExportUxManager.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on('dblclick', '.mainTable tr', function () {
      that.$getEditCommand().invoke();
    });
    dom.on('dragstart', false);
    var lex = lexicons[0]; // Call superclass' doInitialize, which allows MgrLearn to set up dom for this Widget

    return Manager.prototype.doInitialize.apply(that, arguments).then(function () {
      // Add in dom elements for historyConfig slots
      dom.prepend(historyConfigTemplate({
        displayStateName: lex.get('CloudHistoryExportUxManager.displayState.displayName') + ':',
        backfillStatusName: lex.get('CloudHistoryExportUxManager.backfillStatus.displayName') + ':'
      }));
      return baja.Ord.make('service:cloudLink:HistoriesChannel').get().then(function (historyChannel) {
        return historyChannel.lease().then(function () {
          return historyChannel;
        });
      });
    }) // Create field editors for slots. Note that these handle their own subscriptions
    .then(function (historyChannel) {
      return fe.buildFor({
        complex: historyChannel,
        slot: 'displayState',
        dom: dom.find('.display-state-value'),
        formFactor: 'mini',
        readonly: 'true'
      }).then(function () {
        return historyChannel.getChannelConfig();
      }).then(function (channelConfig) {
        return channelConfig.lease().then(function () {
          return channelConfig;
        });
      });
    }).then(function (config) {
      return Promise.all([fe.buildFor({
        complex: config,
        slot: 'backfillStatus',
        dom: dom.find('.backfill-status-value'),
        formFactor: 'mini',
        readonly: true
      })]);
    });
  };

  CloudHistoryExportUxManager.prototype.attach = function (attachable) {
    var _this = this;

    var that = this;
    var changedHistId = {};

    var update = function update() {
      var learnModel = that.getLearnModel();

      if (!learnModel) {
        return;
      }

      populateCloudHistoryExportConfigVsHistoryIdsMap(that);
      learnModel.emit('rowsChanged', learnModel.getRows().filter(function (row) {
        var navNode = row.getSubject();
        var rowId = navNode.getNavOrd().parse().get('history').getBody();
        return changedHistId[rowId];
      }));
      changedHistId = {};
    };

    if (!this.properties().getValue('skipDebounce')) {
      update = _.debounce(update, 1000);
    }

    var updateRow = function updateRow(histId) {
      changedHistId[histId] = true;
      update();
    };

    var removePolicy = function removePolicy(policy) {
      var learnModel = _this.getLearnModel();

      if (!learnModel) {
        return;
      }

      populateCloudHistoryExportConfigVsHistoryIdsMap(that);
      var vector = policy.get('historyIds');
      var rowMap = {};
      vector.getSlots().is('history:HistoryId').eachValue(function (histId) {
        rowMap[histId.encodeToString()] = true;
      });
      var rowArray = [];
      learnModel.getRows().map(function (row) {
        var navNode = row.getSubject();
        var rowId = navNode.getNavOrd().parse().get('history').getBody();

        if (rowMap[rowId]) {
          rowArray.push(row);
        }
      });
      learnModel.emit('rowsChanged', rowArray);
    };

    attachable.attach({
      'autoExport/excludedHistoryIds/added': function autoExportExcludedHistoryIdsAdded(prop) {
        updateRow(baja.SlotPath.unescape(prop.getName()));
      },
      'autoExport/excludedHistoryIds/removed': function autoExportExcludedHistoryIdsRemoved(prop) {
        updateRow(baja.SlotPath.unescape(prop.getName()));
      },
      '.+/historyIds/added': function historyIdsAdded(prop) {
        updateRow(this.get(prop).encodeToString());
      },
      '.+/historyIds/removed': function historyIdsRemoved(prop, oldVal) {
        updateRow(oldVal.encodeToString());
      },
      'removed': function removed(prop, oldVal) {
        removePolicy(oldVal);
      }
    });
  }; ////////////////////////////////////////////////////////////
  //Context Menu Support
  ////////////////////////////////////////////////////////////

  /**
   * Callback from CommandGroupContextMenu. Trigger context menus specifically on table row right clicks
   * @override
   * @returns {string}
   */


  CloudHistoryExportUxManager.prototype.getContextMenuSelector = function () {
    return 'tr';
  };
  /**
   * Callback from ContextMenuSupport to provide a chance to pre-process
   * the right click menu's CommandGroup prior to displaying it
   *
   * @param group default CommandGroup consisting of subject's menu agents
   * @returns {module:bajaux/commands/CommandGroup} CommandGroup updated CommandGroup
   */


  CloudHistoryExportUxManager.prototype.updateMenuCommandGroup = function (group) {
    var editCmd = this.$getEditCommand();

    if (group) {
      group.add(editCmd);
    }

    return group;
  };
  /**
   * Callback from ContextMenuSupport. Returns array of currently selected subjects for context menu
   * @override
   * @returns {Array}
   */


  CloudHistoryExportUxManager.prototype.getSubject = function (elem) {
    return this.getMainTable().getSubject(elem);
  }; //TODO: Uncomment with respective story
  ////////////////////////////////////////////////////////////////
  // State
  ////////////////////////////////////////////////////////////////

  /*  CloudHistoryExportUxManager.prototype.saveStateForOrd = function () {
      var that = this, obj = {};
       if (that.$discoveries) {
        obj.$discoveries = _.map(obj.$discoveries, function (discovery) {
          return baja.bson.encodeValue(discovery);
        });
      }
      return obj;
    };
     CloudHistoryExportUxManager.prototype.restoreStateForOrd = function (obj) {
      if (obj.$discoveries) {
        this.$discoveries = _.map(obj.$discoveries, function (discovery) {
          return baja.bson.decodeValue(discovery);
        });
        return reloadLearnTable(this);
      } else {
        return Promise.resolve();
      }
    };*/


  return CloudHistoryExportUxManager;
});
