/*
 * Decompiled with CFR 0.152.
 */
package com.tridium.niagararemoteclient;

import com.tridium.niagararemoteclient.NiagaraRemoteClient;
import com.tridium.niagararemoteclient.NiagaraRemoteConstants;
import com.tridium.niagararemoteclient.NiagaraRemoteMessage;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.WebSocketAdapter;

public class NiagaraRemoteWebsocketAdapter
extends WebSocketAdapter {
    private static final Logger LOGGER = Logger.getLogger(NiagaraRemoteWebsocketAdapter.class.getName());
    private static final long DEFAULT_PING_INTERVAL = TimeUnit.SECONDS.toMillis(15L);
    private final Map<Integer, Connection> connectionMap = new ConcurrentHashMap<Integer, Connection>();
    private final NiagaraRemoteClient client;
    private final ScheduledExecutorService executor;
    private final long pingInterval;

    NiagaraRemoteWebsocketAdapter(NiagaraRemoteClient client) {
        this(client, DEFAULT_PING_INTERVAL);
    }

    NiagaraRemoteWebsocketAdapter(NiagaraRemoteClient client, long pingInterval) {
        this.client = client;
        this.pingInterval = pingInterval;
        ThreadFactory threadFactory = r -> {
            Thread thread = Executors.defaultThreadFactory().newThread(r);
            thread.setName(NiagaraRemoteConstants.PING_THREAD_NAME);
            return thread;
        };
        this.executor = Executors.newSingleThreadScheduledExecutor(threadFactory);
    }

    public void onWebSocketConnect(Session session) {
        super.onWebSocketConnect(session);
        try {
            byte[] publicKeyBytes = this.client.getCertPublicKeyHashBytes();
            NiagaraRemoteMessage clientHello = new NiagaraRemoteMessage(1, 3, publicKeyBytes);
            this.getRemote().sendBytesByFuture(ByteBuffer.wrap(clientHello.toBytes()));
        }
        catch (Exception ex) {
            LOGGER.log(Level.WARNING, "Error sending client hello.", ex);
        }
        this.executor.scheduleAtFixedRate(new PingRunnable(), this.pingInterval, this.pingInterval, TimeUnit.MILLISECONDS);
    }

    public void onWebSocketClose(int statusCode, String reason) {
        super.onWebSocketClose(statusCode, reason);
        for (Connection connection : this.connectionMap.values()) {
            connection.stop();
        }
        this.executor.shutdownNow();
        this.client.websocketClosed(reason);
    }

    public void onWebSocketError(Throwable cause) {
        LOGGER.log(Level.FINE, "Websocket error", cause);
    }

    public void onWebSocketBinary(byte[] buf, int offset, int len) {
        NiagaraRemoteMessage message = NiagaraRemoteMessage.parseMessage(buf, offset, len);
        switch (message.getMessageType()) {
            case 2: {
                this.handleServerHello(message);
                break;
            }
            case 3: {
                this.handleOpen(message);
                break;
            }
            case 4: {
                this.handleStatus(message);
                break;
            }
            case 5: {
                this.handleData(message);
                break;
            }
            default: {
                LOGGER.severe(String.format("Unrecognized message type: 0x%02x", message.getMessageType()));
            }
        }
    }

    public void onWebSocketText(String message) {
        LOGGER.severe(String.format("Received unexpected websocket text: %s", message));
    }

    private void handleServerHello(NiagaraRemoteMessage message) {
        if (message.getHeader() != 3) {
            LOGGER.severe(String.format("Incompatible server version: 0x%02x", message.getHeader()));
        }
    }

    private void handleOpen(NiagaraRemoteMessage message) {
        int connectionId = message.getHeader();
        byte[] messageData = message.getMessageData();
        int port = (messageData[0] & 0xFF) << 8 | messageData[1] & 0xFF;
        UUID sessionQualifier = null;
        if (messageData.length > 2) {
            if (messageData.length != 18) {
                LOGGER.warning("Invalid length for open message: " + messageData.length);
            } else {
                ByteBuffer buffer = ByteBuffer.wrap(messageData, 2, 16);
                long high = buffer.getLong();
                long low = buffer.getLong();
                sessionQualifier = new UUID(high, low);
            }
        }
        this.client.sessionOpened(sessionQualifier);
        Map<Integer, Integer> portMapping = this.client.getPortMapping();
        if (portMapping != null) {
            Integer mappedPort = portMapping.get(port);
            if (mappedPort == null) {
                LOGGER.warning(String.format("Received open request to unmapped port %d", port));
                this.sendStatusResponse(connectionId, (byte)4);
                return;
            }
            LOGGER.fine(String.format("Mapping requested port %d to local port %d", port, mappedPort));
            port = mappedPort;
        }
        LOGGER.fine(String.format("Opening socket connection on port %d with connection id %d", port, message.getHeader()));
        try {
            Socket socket = new Socket("localhost", port);
            InputStream socketIn = socket.getInputStream();
            OutputStream socketOut = socket.getOutputStream();
            Connection connection = new Connection(connectionId, socket, socketIn, socketOut);
            new Thread((Runnable)connection, NiagaraRemoteConstants.WORKER_THREAD_NAME).start();
            this.connectionMap.put(connectionId, connection);
            this.sendStatusResponse(connectionId, (byte)1);
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Error opening socket.", e);
            this.sendStatusResponse(connectionId, (byte)4);
        }
    }

    private void handleStatus(NiagaraRemoteMessage message) {
        int connectionId = message.getHeader();
        byte status = message.getMessageData()[0];
        Connection connection = this.connectionMap.get(connectionId);
        if (connection != null) {
            switch (status) {
                case 2: {
                    connection.shutdownOutput();
                    break;
                }
                case 3: {
                    connection.stop();
                    break;
                }
                default: {
                    LOGGER.severe(String.format("Received unrecognized status 0x%02x for connection %d", connectionId, status));
                    break;
                }
            }
        } else {
            LOGGER.fine(String.format("Received status request for unknown connection %d", connectionId));
        }
    }

    private void handleData(NiagaraRemoteMessage message) {
        LOGGER.finer(String.format("Handling data for connection id %d", message.getHeader()));
        int connectionId = message.getHeader();
        Connection connection = this.connectionMap.get(connectionId);
        if (connection != null) {
            try {
                LOGGER.finer(() -> String.format("Writing %d bytes to socket", message.getMessageData().length));
                connection.socketOut.write(message.getMessageData());
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "Error writing to socket", e);
                connection.stop();
                this.sendStatusResponse(connectionId, (byte)3);
            }
        } else {
            LOGGER.fine(String.format("Received data message for unknown connection %d", connectionId));
        }
    }

    private void sendStatusResponse(int connectionId, byte status) {
        NiagaraRemoteMessage statusMessage = new NiagaraRemoteMessage(4, connectionId, new byte[]{status});
        try {
            this.getRemote().sendBytesByFuture(ByteBuffer.wrap(statusMessage.toBytes()));
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "Error sending status response.", e);
        }
    }

    class PingRunnable
    implements Runnable {
        PingRunnable() {
        }

        @Override
        public void run() {
            try {
                NiagaraRemoteWebsocketAdapter.this.getRemote().sendPing(ByteBuffer.allocate(0));
                LOGGER.fine("Ping success");
            }
            catch (IOException e) {
                LOGGER.log(Level.SEVERE, "Ping failed", e);
            }
        }
    }

    class Connection
    implements Runnable {
        private final int connectionId;
        private final Socket socket;
        private final InputStream socketIn;
        private final OutputStream socketOut;
        private boolean stopping = false;

        public Connection(int connectionId, Socket socket, InputStream in, OutputStream out) {
            this.connectionId = connectionId;
            this.socket = socket;
            this.socketIn = in;
            this.socketOut = out;
        }

        public synchronized void shutdownOutput() {
            try {
                this.socket.shutdownOutput();
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "Error shutting down socket output.", e);
            }
            if (this.socket.isInputShutdown()) {
                this.stop();
            }
        }

        public synchronized void shutdownInput() {
            this.stopping = true;
            try {
                this.socket.shutdownInput();
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "Error shutting down socket input.", e);
            }
            if (this.socket.isOutputShutdown()) {
                this.stop();
            }
        }

        @Override
        public void run() {
            byte[] buffer = new byte[4096];
            while (!this.socket.isClosed() && !this.socket.isInputShutdown()) {
                try {
                    int length = this.socketIn.read(buffer);
                    if (length > 0) {
                        byte[] messageBytes = new byte[length];
                        System.arraycopy(buffer, 0, messageBytes, 0, length);
                        NiagaraRemoteMessage message = new NiagaraRemoteMessage(5, this.connectionId, messageBytes);
                        LOGGER.fine(() -> String.format("Writing %d bytes to websocket", length));
                        NiagaraRemoteWebsocketAdapter.this.getRemote().sendBytesByFuture(ByteBuffer.wrap(message.toBytes())).get();
                        continue;
                    }
                    if (length != -1) continue;
                    this.shutdownInput();
                    NiagaraRemoteWebsocketAdapter.this.sendStatusResponse(this.connectionId, (byte)2);
                    return;
                }
                catch (IOException | InterruptedException | ExecutionException e) {
                    if (this.stopping) continue;
                    LOGGER.log(Level.WARNING, "Error reading socket data.", e);
                    this.stop();
                    NiagaraRemoteWebsocketAdapter.this.sendStatusResponse(this.connectionId, (byte)3);
                }
            }
        }

        public void stop() {
            this.stopping = true;
            try {
                this.socket.close();
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "Failed to close socket.", e);
            }
            NiagaraRemoteWebsocketAdapter.this.connectionMap.remove(this.connectionId);
        }
    }
}

