/**
 * @copyright  2019 Tridium, Inc. All Rights Reserved.
 * @author Ashutosh Chaturvedi
 */

/**
 * API Status: **Private**
 * @module nmodule/cloudLink/rc/history/util/HistoryIdAwareTreeNode
 */
define(['baja!', 'nmodule/webEditors/rc/wb/tree/BajaNavTreeNode', 'nmodule/history/rc/baja/HistoryId', 'baja!history:HistoryId'], function (baja, BajaNavTreeNode, HistoryId) {
  'use strict';
  /**
   * A utility class to build HistoryIdAwareTreeNode which extends from BajaNavTreeNode
   * and each node contains the HistoryId.
   * @private
   * @class
   * @alias nmodule/cloudLink/rc/history/util/HistoryIdAwareTreeNode
   * @extends module:nmodule/webEditors/rc/wb/tree/BajaNavTreeNode
   * @param {baja.NavContainer} navNode
   * @param {Array} historyIdsToSkip
   */

  var HistoryIdAwareTreeNode = function HistoryIdAwareTreeNode(navNode, historyIdsToSkip) {
    BajaNavTreeNode.apply(this, arguments);
    this.historyIdsToSkip = historyIdsToSkip || [];
  };

  HistoryIdAwareTreeNode.prototype = Object.create(BajaNavTreeNode.prototype);
  HistoryIdAwareTreeNode.prototype.constructor = HistoryIdAwareTreeNode;
  /**
   * Loads the array of child nodes by building them up from the nav node's
   * NavChildren and add historyId to them by parsing the navOrd, skipping
   * any historyIds in {@link #historyIdsToSkip}.
   * @returns {Promise} promise to be resolved with an array of
   * `HistoryIdAwareTreeNode`s created from the nav node's NavChildren
   */

  HistoryIdAwareTreeNode.prototype.$loadKids = function () {
    var that = this,
        thisNavNode = this.value();
    return thisNavNode.getNavChildren().then(function (navNodes) {
      return navNodes.filter(function (navNode) {
        var navOrd = navNode.getNavOrd().toString(),
            historyId = that.historyQueryParse(navOrd);
        return historyId && !that.historyIdsToSkip.includes(historyId.encodeToString());
      }).map(function (navNode) {
        var navOrd = navNode.getNavOrd().toString(),
            historyId = that.historyQueryParse(navOrd);

        if (historyId) {
          navNode.getHistoryId = function () {
            return historyId;
          };
        }

        return new HistoryIdAwareTreeNode(navNode);
      });
    });
  };
  /**
   * Parse the navOrd string to get historyId from it. Return null if no historyId
   * is present.
   *
   * The navOrd with historyId can be in the form of:
   * absolute            := fullAbs | historyGroupingAbs
   *
   * fullAbs             := "/" deviceName "/" historyName
   * historyGroupingAbs  := "//" [folderNames] "//" deviceName "/" historyName
   *
   * @returns {module:nmodule/history/rc/baja/HistoryId} historyId of the ord
   */


  HistoryIdAwareTreeNode.prototype.historyQueryParse = function (ordStr) {
    var historyPath = ordStr.indexOf("|history:");

    if (historyPath >= 0) {
      var path = ordStr.substring(historyPath + 9),
          deviceName,
          historyName,
          deviceIdx,
          historyIdx;

      if (path.indexOf("///") === 0) {// root, ignore
      } else if (path.indexOf("//") === 0) {
        // history under grouping, ignore if folder
        path = path.substring(2);
        deviceIdx = path.indexOf("//");

        if (deviceIdx >= 0) {
          historyIdx = path.lastIndexOf("/");
          deviceName = path.substring(deviceIdx + 2, historyIdx);
          historyName = path.substring(historyIdx + 1);
          return HistoryId.make(deviceName, historyName);
        }
      } else if (path.indexOf("/") === 0) {
        // history under device, ignore if folder
        var _historyIdx = path.lastIndexOf("/");

        if (_historyIdx > 0) {
          deviceIdx = path.indexOf("/");
          deviceName = path.substring(deviceIdx + 1, _historyIdx);
          historyName = path.substring(_historyIdx + 1);
          return HistoryId.make(deviceName, historyName);
        }
      }
    }
  };
  /**
   * HistoryIdAwareTreeNode nodes don't support drag and drop.
   *
   * @returns {boolean} false
   */


  HistoryIdAwareTreeNode.prototype.isDraggable = function () {
    return false;
  };
  /**
   * HistoryIdAwareTreeNode nodes don't support drag and drop.
   *
   * @returns {boolean} false
   */


  HistoryIdAwareTreeNode.prototype.isDropTarget = function () {
    return false;
  };

  return HistoryIdAwareTreeNode;
});
