/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to navbar management in Niagara apps.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, browser: true */
/*global baja, niagara */

(function () {
  "use strict";
  
  niagara.util.require(
    'niagara.util.View'
  );
  
  var util = niagara.util,
      View = util.View,
      
      //exports
      NavModel,
      NavModelNode,
      ComponentSpaceNavModel,
      NavFileNavModel;
  
  
////////////////////////////////////////////////////////////////
//NavModel
////////////////////////////////////////////////////////////////
  

  /**
   * Wraps an object in another object that provides extra functionality needed
   * for navigation. Attempts to derive navigation information from the given
   * object like nav ORD, icon, and description.
   * 
   * @class
   * @name niagara.util.nav.NavModelNode
   * @private
   * @param {String|baja.Ord|baja.Complex|niagara.util.View} obj an object to 
   * be used for navigation
   */
  NavModelNode = function NavModelNode(obj) {
    
    var that = this,
        returnNode,
        len,
        split,
        $navOrd,
        $navDescription,
        $navIcon,
        $navChildren = [],
        $navParent,
        $value = obj;
    
    if (obj instanceof NavModelNode) {
      return obj;
    } else if (obj instanceof baja.Ord || typeof obj === 'string') {
      $navOrd = baja.Ord.make(obj);
      $navDescription = String($navOrd);
      split = $navDescription.split('/');
      $navDescription = split[split.length - 1];
      
    } else if (obj instanceof baja.Complex) {
      $navDescription = util.getComponentName(obj);
      $navOrd = util.ord.deriveOrd(obj);
      $navIcon = obj.getIcon();
      
    } else if (obj instanceof View) {
      $navDescription = obj.getName();
      
      if (obj.value instanceof baja.Complex) {
        $navOrd = util.ord.deriveOrd(obj.value);
        $navIcon = obj.value.getIcon();
      }
      if (!$navOrd && obj.pageId) {
        $navOrd = util.mobile.decodePageId(obj.pageId);
        if ($navOrd.indexOf('ord?') === 0) {
          $navOrd = $navOrd.substring(4);
        }
      }
      
    } else if (obj && typeof obj === 'object') {
      $navOrd = obj.$navOrd;
      $navDescription = obj.$navDescription;
      $navIcon = obj.$navIcon;
      $navChildren = obj.$navChildren;
      $navParent = obj.$navParent;
      $value = obj.$value;
    }
    
    that.$navOrd = $navOrd;
    that.$navDescription = $navDescription || baja.lex('mobile').get('propsheet.station');
    that.$navIcon = $navIcon;
    that.$navChildren = $navChildren || [];
    that.$navParent = $navParent;
    that.$value = $value;
  };
  
  NavModelNode.prototype = {
    /**
     * @name niagara.util.nav.NavModelNode#getNavOrd
     * @function
     * @returns {String} this node's nav ORD
     */
    getNavOrd: function () { return this.$navOrd; },
    /**
     * @name niagara.util.nav.NavModelNode#setNavOrd
     * @function
     * @param {String} navOrd the nav ORD to set
     */
    setNavOrd: function (navOrd) { this.$navOrd = navOrd; },
    /**
     * @name niagara.util.nav.NavModelNode#getNavDescription
     * @function
     * @returns {String} this node's nav description
     */
    getNavDescription: function () { return this.$navDescription; },
    /**
     * @name niagara.util.nav.NavModelNode#setNavDescription
     * @function
     * @param {String} navDescription the nav description to set
     */
    setNavDescription: function (navDescription) { this.$navDescription = navDescription; },
    /**
     * @name niagara.util.nav.NavModelNode#getNavIcon
     * @function
     * @returns {baja.Icon} this node's nav icon
     */
    getNavIcon: function () { return this.$navIcon; },
    /**
     * @name niagara.util.nav.NavModelNode#setNavIcon
     * @function
     * @param {baja.Icon} navIcon the nav icon to set
     */
    setNavIcon: function (navIcon) { this.$navIcon = navIcon; },
    /**
     * @name niagara.util.nav.NavModelNode#getNavChildren
     * @function
     * @returns {Array} an array of this node's nav children (also NavModelNode)
     */
    getNavChildren: function () { return this.$navChildren; },
    /**
     * @name niagara.util.nav.NavModelNode#setNavChildren
     * @function
     * @private
     * @param {Array} navChildren the nav children to set (must be NavModelNode)
     */
    setNavChildren: function (navChildren) { this.$navChildren = navChildren; },
    /**
     * @name niagara.util.nav.NavModelNode#getNavParent
     * @function
     * @returns {niagara.util.nav.NavModelNode} this node's nav parent
     */
    getNavParent: function () { return this.$navParent; },
    /**
     * @name niagara.util.nav.NavModelNode#setNavParent
     * @function
     * @private
     * @param {niagara.util.nav.NavModelNode} parent the parent to set
     */
    setNavParent: function (navParent) { this.$navParent = navParent; },
    /**
     * @name niagara.util.nav.NavModelNode#getValue
     * @function
     * @returns {Object} this nav node's current value (ORD, component, view, etc)
     */
    getValue: function () { return this.$value; },
    /**
     * @name niagara.util.nav.NavModelNode#setValue
     * @function
     * @private
     * @param {Object} value the value to set
     */
    setValue: function (value) { this.$value = value; },
    /**
     * @name niagara.util.nav.NavModelNode#toString
     * @function
     * @returns a String representation of this node (just the nav ORD)
     */
    toString: function () { return String(this.$navOrd); }
  };
  
////////////////////////////////////////////////////////////////
//NavModel
////////////////////////////////////////////////////////////////
  
  /**
   * Models a stack of navigation nodes, intended to back a navbar at the
   * top of the screen. A NavModel is backed by a simple array that is treated
   * like a stack using <code>push()</code> and <code>pop()</code>. It will
   * typically represent a single path through a tree, like a component space
   * or a nav file.
   * 
   * @class
   * @memberOf niagara.util.nav
   */
  NavModel = function NavModel() {
    this.$selectedNode = null;
    this.$nodes = [];
    this.$changeListeners = [];
    this.$cutListeners = [];
  };
  
  /*
   * Give NavModel event capabilities
   */
  baja.event.mixin(NavModel.prototype);
  
  /**
   * Adds a new navigation node. Default behavior is simply to push the new
   * node onto the end of the stack.
   * <p>
   * NavModel also gets a BajaScript event mixin, so you can can bind to the
   * following events:
   * 
   * <ul>
   * <li><code>change</code> - a nav node was added or removed, or the selected
   * node was changed</li>
   * <li><code>cut</code> - a nav node was removed (the removed node is passed
   * as first argument to the event handler</li>
   * </ul>
   * 
   * @name niagara.util.nav.NavModel#add
   * @function
   * @param {Object} node the node to add
   * @returns this
   */
  NavModel.prototype.add = function add(obj) {
    var node = new NavModelNode(obj);
    
    this.pruneOtherBranch(node);
    this.$nodes.push(node);
    
    return this;
  };
  
  /**
   * Removes a node from the stack, along with all descendant nodes.
   * 
   * @name niagara.util.nav.NavModel#cut
   * @function
   * @param {Object} node the node to cut
   * @returns this
   */
  NavModel.prototype.cut = function cut(node) {
    if (!node || this.indexOf(node) < 0) {
      return;
    }
    
    node = new NavModelNode(node);
  
    var that = this,
        next = that.next(node),
        pageDiv;
  
    if (next) {
      that.cut(next);
    }
  
    that.$nodes.pop();
    
    that.fireHandlers('cut', baja.error, this, node);
    return that;
  };
  
  /**
   * Returns the currently selected node.
   * 
   * @name niagara.util.nav.NavModel#getSelectedNode
   * @function
   * @returns {Object} the currently selected node
   */
  NavModel.prototype.getSelectedNode = function getSelectedNode() {
    return this.$selectedNode;
  };
  
  /**
   * Returns the index of the currently selected node.
   * 
   * @name niagara.util.nav.NavModel#getSelectedIndex
   * @function
   * @returns {Number} the currently selected index
   */
  NavModel.prototype.getSelectedIndex = function getSelectedIndex() {
    return this.indexOf(this.getSelectedNode());
  };
  
  /**
   * Sets the currently selected node.
   * 
   * @name niagara.util.nav.NavModel#setSelectedNode
   * @function
   * @param {Object} node the node to set as selected
   * @throws {Error} if the given node does not already exist in this NavModel
   */
  NavModel.prototype.setSelectedNode = function setSelectedNode(node) {
    node = new NavModelNode(node);
    
    if (this.indexOf(node) < 0) {
      this.add(node);
    }

    this.$selectedNode = node;
    this.fireHandlers('change', baja.error, this);
  };
  
  /**
   * Returns all the nodes currently contained in this NavModel.
   * 
   * @name niagara.util.nav.NavModel#getNodes
   * @function
   * @returns {Array} an array of nodes
   */
  NavModel.prototype.getNodes = function getNodes() {
    return this.$nodes;
  };

  /**
   * Returns an array of the ORDs corresponding to all the nodes contained
   * in this NavModel.
   * 
   * @name niagara.util.navModel#getOrds
   * @function
   * @returns {Array} an array of ORDs
   * @see niagara.util.navModel#getNodeOrd
   */
  NavModel.prototype.getOrds = function getOrds() {
    var ords = [],
        that = this;
    that.each(function (node) {
      ords.push(node.getNavOrd());
    });
    return ords;
  };
  
  /**
   * Iterate over the nodes in this NavModel, executing a function on each one.
   * 
   * @name niagara.util.nav.NavModel#each
   * @function
   * @param {Function} func a function to execute on each node - the node will
   * passed as the first argument and the node's index as the second
   * @returns the first value returned from <code>func</code>
   */
  NavModel.prototype.each = function each(func) {
    return baja.iterate(this.$nodes, func);
  };
  
  /**
   * Check to see if this NavModel contains no nodes.
   * 
   * @name niagara.util.nav.NavModel#isEmpty
   * @function
   * @returns {Boolean} true if the NavModel is empty
   */
  NavModel.prototype.isEmpty = function isEmpty() {
    return this.$nodes.length === 0;
  };
  
  /**
   * Returns the first node in this NavModel - often, this might correspond to
   * the station root.
   * 
   * @name niagara.util.nav.NavModel#first
   * @function
   * @returns {Object} the first (root) node
   */
  NavModel.prototype.first = function first() {
    return this.get(0);
  };
  
  /**
   * Returns the last node in this NavModel.
   * 
   * @name niagara.util.nav.NavModel#last
   * @function
   * @returns {Object} the last node
   */
  NavModel.prototype.last = function last() {
    return this.get(this.size() - 1);
  };
  
  /**
   * Returns the index of the given node. The index is determined by examining
   * the node's ORD (using <code>getNodeOrd</code>) and comparing it to the
   * ORDs of the other nodes in this NavModel.
   * 
   * @name niagara.util.nav.NavModel#indexOf
   * @function
   * @param {Object} node the node whose index to find
   * @returns {Number} the index of the given node, or -1 if not found 
   */
  NavModel.prototype.indexOf = function indexOf(node) {
    node = new NavModelNode(node);
    
    var that = this,
        ord = node.getNavOrd(),
        index;
    
    index = that.each(function(node, i) {
      var otherOrd = node.getNavOrd();
      if (util.ord.equivalent(ord, otherOrd)) {
        return i;
      }
    });
    
    return (index === undefined) ? -1 : index;
  };
  
  /**
   * Returns the node at the specified index.
   * 
   * @name niagara.util.nav.NavModel#get
   * @function
   * @param {Number} i the desired node index
   * @returns {Object} the node at the specified index
   */
  NavModel.prototype.get = function get(index) {
    return this.$nodes[index];
  };
  
  
  /**
   * Returns the node next in line (to the right of) the given node. If the
   * given node does not exist in the NavModel, or if there are no nodes to
   * the right of it, undefined will be returned.
   * 
   * @name niagara.util.nav.NavModel#next
   * @function
   * @param {Object} node find the node to the right of this node
   * @returns {Object} the next node in line, or undefined if not found
   */
  NavModel.prototype.next = function next(node) {
    var index = this.indexOf(node);
    return index >= 0 ? this.get(index + 1) : undefined;
  };
  
  /**
   * Returns the node previous (to the left of) the given node. If the given
   * node does not exist in the NavModel, or if there are no nodes to the left
   * of it, undefined will be returned.
   * 
   * @name niagara.util.nav.NavModel#prev
   * @function
   * @param {Object} node find the node to the left of this node
   * @returns {Object} the previous node in line, or undefined if not found
   */
  NavModel.prototype.prev = function prev(node) {
    var index = this.indexOf(node);
    return index >= 0 ? this.get(index - 1) : undefined;
  };
  
  /**
   * Returns the number of nodes in this NavModel.
   * 
   * @name niagara.util.nav.NavModel#size
   * @function
   * @returns {Number} the number of nodes in this NavModel
   */
  NavModel.prototype.size = function size() {
    return this.$nodes.length;
  };
  
  
  /**
   * Prunes out any open views from other branches.
   * <p>
   * When navigating a component hierarchy, only one branch of the tree should
   * have components subscribed at any one time - otherwise, given enough
   * time and clicks, we could have dozens or hundreds of components with
   * subscribed views, eating up memory and CPU cycles. This function can prune
   * away other nodes.
   * <p>
   * For instance, if we start at <code>station:|slot:</code> and navigate down
   * to <code>station:|slot:/Services</code>, we will have two views open and
   * subscribed. If we go back up to the root and then navigate down to
   * <code>station:|slot:/Drivers</code>, the view on <code>Services</code>
   * will be stopped and its corresponding page will be removed from the DOM,
   * since it is on a different branch from the one currently being viewed.
   * 
   * @name niagara.util.nav.NavModel#pruneOtherBranch
   * @function
   * @param {niagara.util.mobile.PageView} view the view we wish to add to the
   * stack
   */
  NavModel.prototype.pruneOtherBranch = function pruneOtherBranch(node) {
    var that = this;
    
    if (that.isEmpty()) {
      return;
    }
    
    if (that.sameBranch(node, that.last())) {
      return;
    }
    
    baja.iterate(1, this.size(), function (i) {
      var otherNode = that.get(i);
      if (!that.sameBranch(node, otherNode)) {
        return that.cut(otherNode);
      }
    });
  };
  
  /**
   * Used in <code>pruneOtherBranch</code>, this function checks to see if
   * two nodes are in the same branch of whatever tree we're navigating. This
   * allows us to prune away any other nodes that are not in the same branch
   * of our nav tree each time we attempt to add a new node to our NavModel.
   * 
   * Checks to see if the two nodes exist in the same branch of the component
   * tree. It simply converts the two nodes to ORDs (using 
   * <code>getNodeOrd</code>), and checks to see if one is a substring of
   * the other, starting at string index 0.
   * 
   * @name niagara.util.nav.NavModel#sameBranch
   * @function
   * @param {Object} node1
   * @param {Object} node2
   * @returns {Boolean} true if the two nodes are on the same branch
   */
  NavModel.prototype.sameBranch = function sameBranch(node1, node2) {
    var that = this,
        ord1 = decodeURIComponent(String(node1.getNavOrd())),
        ord2 = decodeURIComponent(String(node2.getNavOrd()));
    
    return ord1.indexOf(ord2) === 0 || ord2.indexOf(ord1) === 0;
  };

  
  
  
////////////////////////////////////////////////////////////////
//ComponentSpaceNavModel
////////////////////////////////////////////////////////////////
  
  /**
   * A NavModel that supports navigation up and down a simple component
   * tree.
   * 
   * @class
   * @name niagara.util.nav.ComponentSpaceNavModel
   * @extends niagara.util.nav.NavModel
   */
  ComponentSpaceNavModel = function ComponentSpaceNavModel() {
    NavModel.call(this);
  }.$extend(NavModel);
    
  /**
   * Override of the <code>add</code> function that ensures that the length
   * of the node stack is always the same as our depth into the component tree.
   * 
   * <p>If we initialize our NavModel with a component at 
   * <code>station:|slot:/</code> (the station root), then our NavModel length
   * will be 1.
   * 
   * <p>However if we initialize with, say, the component at 
   * <code>station:|slot:/Services/UserService</code>, then our length
   * will start out at 3 - the initialization process goes like this:
   * 
   * <ol>
   * 
   * <li>Set our mounted component at the proper position (position is dictated
   * by depth into the component tree), and populate all nodes previous to it
   * with simple string ORDs (in our example, <code>get(0)</code> will return
   * <code>'station:|slot:/'</code> and <code>get(1)</code> will return
   * <code>'station:|slot:/Services'</code>.This will be state of the NavModel
   * as soon as <code>add</code> returns. (A <code>change</code> MicroEvent
   * event will also be triggered before the function returns.)</li>
   * 
   * <li>For any spots in our NavModel that don't have mounted components 
   * (here, 0 and 1), use a <code>baja.BatchResolve</code> to resolve all the
   * string ORDs to actual mounted components.</li>
   * 
   * <li>Once these mounted components return, set them directly as navigation
   * nodes. Trigger a <code>change</code> event for MicroEvent to allow any
   * navbars that use this NavModel to update themselves with the proper
   * display names / icons that we now have.</li> 
   * 
   * </ol> 
   * 
   * @name niagara.util.nav.ComponentSpaceNavModel#add
   * @function
   * @param {Object} node the node to add
   * @returns this
   */
  ComponentSpaceNavModel.prototype.add = function add(obj) {
    var that = this,
        node = new NavModelNode(obj),
        nodes = that.$nodes,
        toResolve = {},
        toResolveArray = [],
        ord = node.getNavOrd(),
        ordHistory = util.ord.makeOrdHistory(ord);
    
    that.pruneOtherBranch(node);

    /*
     * When we first load a page, unless we're only looking at the station root
     * we will need to retrieve the components to obtain their display names
     * and icons for the navbar. But this is an asynchronous action and we
     * have to show *something* while it's taking place, so we start by simply
     * storing an ORD in the view list, then replace with the component once
     * it is retrieved. (We don't do this for the slot we're adding a view for
     * - there's no need.)
     */
    baja.iterate(0, ordHistory.length - 1, function (i) {
      if (!nodes[i]) {
        var ord = ordHistory[i],
            len = toResolveArray.length;
        
        //spot in our node list is empty - set it to the ORD first...
        nodes[i] = new NavModelNode(ord);
        
        //then make a record saying that we need to include this ORD in our
        //batch resolve
        toResolve['ord' + len] = {
          ord: ord,
          origIndex: i
        };
        toResolveArray.push(ord);
      }
    });
    
    if (toResolveArray.length) {
      //we have ORD sitting in our view stack - so let's replace them with
      //live components
      new baja.BatchResolve(toResolveArray).resolve({
        ok: function () {
          var objs = this.getTargetObjects();
          
          baja.iterate(objs, function (obj, i) {
            //replace the ORD in our view stack with the actual component
            var thing = toResolve['ord' + i];
            nodes[thing.origIndex] = new NavModelNode(obj);
          });
          
          /*
           * Once the stack is fully populated with placeholders, trigger a 
           * change event so the nav bar (if we have one) can redraw itself 
           * with proper display names and icons.
           */
          that.fireHandlers('change', baja.error, this);
        },
        fail: baja.error
      });
    }
    
    nodes[ordHistory.length - 1] = node;
    that.fireHandlers('change', baja.error, this);
    return this;
  };
  
  /**
   * Converts a nav file root node into a flat hierarchy of 
   * <code>NavModelNode</code>s. Calls itself recursively to flatten the
   * node's children as well.
   * 
   * @name niagara.util.nav.flattenNavFileNode
   * @function
   * @private
   * @param {baja.NavContainer} node the nav file root node to flatten
   * @param {Object} callbacks an object containing ok/fail callbacks. The
   * flattened <code>NavModelNode</code> will be passed as the first argument
   * to <code>ok</code>.
   */
  function flattenNavFileNode(node, callbacks) {
    callbacks = util.callbackify(callbacks);
    
    var flattened = new NavModelNode({
      $navIcon: node.getNavIcon(),
      $navDescription:  node.getNavDescription(),
      $navOrd: String(node.getNavOrd()).replace('local:|', '')
    });
    flattened.$fromNavFile = true;
    
    node.getNavChildren(function (kids) {
      if (kids && kids.length) {
        var invocs = [];
        
        baja.iterate(kids, function (kid) {
          invocs.push(function (parent) {
            var that = this;
            flattenNavFileNode(kid, function (flattenedKid) {
              parent.$navChildren.push(flattenedKid);
              flattenedKid.$navParent = parent;
              that.ok(parent);
            });
          });
        });
        util.flow.sequential.apply(this, invocs).invoke(flattened, callbacks);
      } else {
        callbacks.ok(flattened);
      }
    });
  }
  
  /**
   * Examines a nav node, looking for a particular ORD. If the given node has
   * the given ORD, return that node. Otherwise, recursively examine the node's
   * children.
   * 
   * @name niagara.util.nav.getNodeMatchingOrd
   * @function
   * @private
   * @param {niagara.util.nav.NavModelNode} node the node to check for an ORD
   * @param {String|baja.Ord} ord the ORD to look for
   * @returns {niagara.util.nav.NavModelNode} the node matching the given ORD,
   * or null if not found
   */
  function getNodeMatchingOrd(node, ord) {
    if (!node) {
      return null;
    }
    
    if (util.ord.equivalent(node.getNavOrd(), ord)) {
      return node;
    }
    
    return baja.iterate(node.getNavChildren(), function (kid) {
      return getNodeMatchingOrd(kid, ord);
    });
  }
  


  /**
   * A navigation model that takes into account the user's nav file. If any
   * of the nodes in our node stack appear in the user's nav file, then the
   * stack will reflect that nav file structure up until the last node that
   * exists in the nav file. Nodes after that will follow a plain component
   * hierarchy structure.
   * 
   * @class
   * @name niagara.util.nav.NavFileNavModel
   * @extends niagara.util.nav.NavModel
   */
  NavFileNavModel = function NavFileNavModel() {
    var that = this;
    
    NavModel.call(that);
    
    /**
     * A copy of the user's nav file, processed and stored as a plain 
     * NavModelNode. This allows us to query the nav file for ORDs 
     * synchronously (as opposed to using the methods off NavContainer which 
     * are asynchronous).
     * 
     * @field
     * @name niagara.util.nav.NavFileNavModel$flattenedNavFile
     * @private
     */
    that.$flattenedNavFile = undefined;
    
    //teeeeeeechnically this should all be a synchronous operation
    flattenNavFileNode(baja.nav.navfile.getRootNode(), function (flattened) {
      that.$flattenedNavFile = flattened;
    });
  }.$extend(NavModel);
  
  /**
   * Processes an ORD into an ORD history that takes into account the user's
   * nav file. If any of the ORDs exist in the nav file, then the first ORD
   * in the resulting array will match the nav file root, and so on down
   * as far as we can go until we no longer match the nav file structure. ORDs
   * after leaving the nav file will be displayed as a simple component
   * hierarchy.
   * 
   * @name niagara.util.nav.NavFileNavModel#makeNavFileOrdHistory
   * @function
   * @param {String|baja.Ord} ord the ord to make into an ord history
   * @returns {Array} an array of ORDs as strings
   */
  NavFileNavModel.prototype.makeNavFileOrdHistory = function makeNavFileOrdHistory(ord) {
    var that = this,
        ordHistory = util.ord.makeOrdHistory(ord),
        len = ordHistory.length,
        backwardsHistory = [],
        forwardsHistory = [];
    
    //iterate backwards through ord
    baja.iterate(0, len, function (i) {
      var ordChunk = ordHistory[len - 1 - i],
          navFileNode = getNodeMatchingOrd(that.$flattenedNavFile, ordChunk);
      
      //this ord exists in the nav file - so everything from here on up the nav
      //file tree should come directly from the nav file
      if (navFileNode) {
        while (navFileNode) {
          backwardsHistory.push(navFileNode);
          navFileNode = navFileNode.getNavParent();
        }
        return true; //stop iteration - we don't care about the rest of our ORD
      } else {
        //still in component tree - continue on to next iteration, stepping up
        //one level in the hierarchy to see if we match anything in the nav
        //file yet
        backwardsHistory.push(new NavModelNode(ordChunk)); 
      }
    });
    
    //put the ord history in forwards order
    len = backwardsHistory.length;
    baja.iterate(0, len, function (i) {
      forwardsHistory.push(backwardsHistory[len - 1 - i]);
    });
    
    return forwardsHistory;
  };
  
  /**
   * Adds an object, as a node, to our nav model. When adding a new node, the
   * entire nav model may be wiped clean and rebuilt, depending on whether the
   * new node exists in the user's nav file or not.
   * 
   * @name niagara.util.nav.NavFileNavModel#add
   * @function
   * @param {String|baja.Ord|baja.Complex|niagara.util.View} obj an object to 
   * convert to a node and add
   * @returns this
   */
  NavFileNavModel.prototype.add = function add(obj) {
    var node = new NavModelNode(obj),
        that = this,
        toResolve = {},
        toResolveArray = [],
        ord = node.getNavOrd(),
        nodes = that.makeNavFileOrdHistory(ord),
        existingNode;
    

    that.pruneOtherBranch(node);
    
    existingNode = that.$nodes[nodes.length - 1] || nodes[nodes.length - 1];
    
    //be sure not to overwrite the icon/description from a nav file
    //node with the icon/description from a component or view - just reset the
    //value we're looking at
    if (existingNode && existingNode.$fromNavFile &&
        util.ord.equivalent(existingNode.getNavOrd(), node.getNavOrd())) {
      node = existingNode;
      node.setValue(obj);
    }
    
    nodes[nodes.length - 1] = node;
    that.$nodes.splice(0, nodes.length);
    that.$nodes = nodes.concat(that.$nodes);
    


    /*
     * When we first load a page, unless we're only looking at the station root
     * we will need to retrieve the components to obtain their display names
     * and icons for the navbar. But this is an asynchronous action and we
     * have to show *something* while it's taking place, so we start by simply
     * storing an ORD in the view list, then replace with the component once
     * it is retrieved. (We don't do this for the slot we're adding a view for
     * - there's no need.)
     */
    baja.iterate(0, that.$nodes.length, function (i) {
      var nodeValue = that.$nodes[i].getValue(),
          len = toResolveArray.length;
      
      if (nodeValue instanceof baja.Ord || typeof nodeValue === 'string') {
        //then make a record saying that we need to include this ORD in our
        //batch resolve
        toResolve['ord' + len] = {
          ord: nodeValue,
          origIndex: i
        };
        toResolveArray.push(nodeValue);
      }
    });
    
    if (toResolveArray.length) {
      //we have ORD sitting in our view stack - so let's replace them with
      //live components
      new baja.BatchResolve(toResolveArray).resolve({
        ok: function () {
          var objs = this.getTargetObjects();
          
          baja.iterate(objs, function (obj, i) {
            //replace the ORD in our view stack with the actual component
            var thing = toResolve['ord' + i];
            that.$nodes[thing.origIndex] = new NavModelNode(obj);
          });
          
          /*
           * Once the stack is fully populated with placeholders, trigger a 
           * change event so the nav bar (if we have one) can redraw itself 
           * with proper display names and icons.
           */
          that.fireHandlers('change', baja.error, this);
        },
        fail: baja.error
      });
    }
    
    that.fireHandlers('change', baja.error, this);
    return this;
  };
  
  /**
   * Determines if two nav nodess are in the same branch of the tree, taking 
   * into account the structure of the user's nav file.
   * 
   * @name niagara.util.nav.NavFileNavModel#sameBranch
   * @function
   * @param {niagara.util.nav.NavModelNode} node1
   * @param {niagara.util.nav.NavModelNode} node2
   * @returns {Boolean} true if the two nodes are in the same branch of
   * the tree
   */
  NavFileNavModel.prototype.sameBranch = function sameBranch(node1, node2) {
    var that = this,
        ord1 = node1.getNavOrd(),
        ord2 = node2.getNavOrd(),
        ordHistory1 = that.makeNavFileOrdHistory(ord1),
        ordHistory2 = that.makeNavFileOrdHistory(ord2),
        len1 = ordHistory1.length,
        len2 = ordHistory2.length,
        different;
    
    different = baja.iterate(0, Math.min(len1, len2), function (i) {
      var different = !util.ord.equivalent(ordHistory1[i], ordHistory2[i]);
      if (different) {
        return true;
      }
    });
    
    return !different;
  };
  
  util.api('niagara.util.nav', {
    'public': {
      NavModel: NavModel,
      ComponentSpaceNavModel: ComponentSpaceNavModel,
      NavFileNavModel: NavFileNavModel
    },
    'private': {
      flattenNavFileNode: flattenNavFileNode,
      getNodeMatchingOrd: getNodeMatchingOrd
    }
  });
}());
