/**
 * Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * Functions to enable one-finger scrolling of an inline div (div must be
 * contained inside an overflow=auto div). Started off with touchscroll.js from
 * http://www.seabreezecomputers.com/tips/scroll-div.htm and massively reworked.
 * 
 * @author Logan Byam
 * @date   March 14 2012
 */

/*jslint white: true, browser: true, plusplus: true */
/*global niagara, baja, $ */

(function () {
  "use strict";
  
  var util = niagara.util,
      nativeScrollSupport = [
        /android 3/i, //honeycomb already supports div scrolling
        /honeycomb/i,
        /macintosh/i, //honeycomb tablet in desktop mode
        /playbook/i, //blackberry playbook has native support as well
        /iphone os 5/i
      ],
      HORIZ_THRESHOLD = 0,
      BIG_HORIZ_THRESHOLD = 15,
      VERT_THRESHOLD = 0,
      BIG_VERT_THRESHOLD = 15;
  
  /**
   * Returns a usable touch object with pageX/pageY given a touch event
   * direct from the browser.
   * 
   * @memberOf niagara.util.mobile.touchscroll
   * @private
   * @returns {Object}
   */
  function getTouch(event) {
    var touches = (event.originalEvent && event.originalEvent.touches) || [],
        changedTouches = (event.originalEvent && event.originalEvent.changedTouches) || [],
        touch = touches[0] || changedTouches[0] || event;
    return touch;
  }
  
  /**
   * Return true if our browser is touch enabled.
   * 
   * @memberOf niagara.util.mobile.touchscroll
   * @private
   * @returns {Boolean}
   */
  function isTouchDevice() {
    try {
      document.createEvent("TouchEvent");
      return true;
    } catch (e) {
      return false;
    }
  }
  
  /**
   * Return true if our browser already supports div overflow scrolling
   * natively.
   * 
   * @memberOf niagara.util.mobile.touchscroll
   * @private
   * @returns {Boolean}
   */
  function supportsNativeScroll() {
    return baja.iterate(nativeScrollSupport, function (regex) {
      if (navigator.userAgent.match(regex)) {
        return true;
      }
    });
  }

  /**
   * Return true if touch scrolling needs to be applied (if we are on a
   * touch-enabled device and we don't already support div overflow scrolling
   * natively).
   * 
   * @memberOf niagara.util.mobile.touchscroll
   * @private
   * @returns {Boolean}
   */
  function needsTouchScroll() {
    return isTouchDevice() && !supportsNativeScroll();
  }
  
  /**
   * Returns a function that will momentarily show a red "no more scrolling"
   * mark on the given element. The returned function will take one of
   * 'top', 'right', 'bottom', or 'left' to choose which edge the mark appears
   * on.
   * @memberOf niagara.util.mobile.touchscroll
   * @private
   * @param {jQuery} elem the element being scrolled
   * @returns {Function}
   */
  function makeShowBlockadeFunction(elem) {
    
    var css = {
          'position': 'absolute',
          'background-color': 'red',
          'opacity': 1
        },
        divs = {
          top: $('<div/>').css(css),
          bottom: $('<div/>').css(css),
          left: $('<div/>').css(css),
          right: $('<div/>').css(css)
        };
    
    return function (border) {
      var pos,
          nonoDiv = divs[border],
          elemHeight = elem.height(),
          parent = elem.parent(),
          parentHeight = parent.height();
      
      switch (border) {
      case 'top': 
        pos =  { left: 0, top: 0, right: 0, height: 5 };
        break;
      case 'left':
        pos = { left: 0, top: 0, width: 5, height: elemHeight };
        break;
      case 'bottom':
        pos = { right: 0, bottom: parentHeight - elemHeight, left: 0, height: 5 };
        break;
      case 'right':
        pos = { top: 0, right: 0, height: elemHeight, width: 5 };
        break;
      }
      
      pos.opacity = 1;
      
      nonoDiv
        .stop()
        .css(pos)
        .remove()
        .appendTo(elem.parent())
        .animate({
          opacity: 0
        }, 'fast', function () {
          nonoDiv.remove();
        });
    };
  }

  /**
   * @memberOf niagara.util.mobile.touchscroll
   * @param {jQuery} elem the element to be scrolled
   * @param {Object} [options] an object literal with options - if omitted
   * defaults to <code>{ horiz: true, vert: true }</code>
   * @param {Boolean} [options.horiz] true if horizontal scrolling should be
   * allowed
   * @param {Boolean} [options.vert] true if vertical scrolling should be
   * allowed
   * @param {Boolean} [options.showBlockade] true if a red mark should appear
   * when scrolling is no longer possible
   * @param {Boolean} [options.force] set to true to skip the check for whether
   * native scrolling is needed and always apply scrolling (good for testing
   * in desktop browser). Note that this library is geared to touch devices and
   * desktop browsers may hide the bottom edge of the scrolled div under a
   * scroll bar.
   */
  function touchScroll(elem, options) {
    options = options || { horiz: true, vert: true, showBlockade: true };

    if (!options.force && !needsTouchScroll()) { 
      return;
    }
    
    var touchStart,
        scrollLeftStart = 0,
        scrollTopStart = 0,
        horizAllowed = options.horiz,
        vertAllowed = options.vert,
        showBlockade = options.showBlockade ? makeShowBlockadeFunction(elem) : baja.noop,
        isTouch = isTouchDevice(),
        touchstart = isTouch ? 'touchstart' : 'mousedown',
        touchmove = isTouch ? 'touchmove' : 'mousemove',
        touchend = isTouch ? 'touchend' : 'mouseup';

    elem.bind(touchstart, function (event) {
      var touch = getTouch(event);
      touchStart = { x: touch.pageX, y: touch.pageY };
      scrollLeftStart = this.scrollLeft;
      scrollTopStart = this.scrollTop;
      //event.preventDefault(); // Keep this remarked so you can click on buttons and links in the div
    });

    elem.bind(touchmove, function (event) {
      if (touchStart === undefined) {
        return;
      }
      
      var that = this,
          
          touch = getTouch(event),
          dY = touchStart.y - touch.pageY,
          dX = touchStart.x - touch.pageX,
          
          scrollTop = that.scrollTop,
          scrollLeft = that.scrollLeft,
          
          maxScrollTop = that.scrollHeight - that.offsetHeight,
          canScrollUp = scrollTop > 0,
          canScrollDown = scrollTop < maxScrollTop,
          
          maxScrollLeft = that.scrollWidth - that.offsetWidth,
          canScrollLeft = scrollLeft > 0,
          canScrollRight = scrollLeft < maxScrollLeft,
          
          tryingToScrollUp = dY < -VERT_THRESHOLD,
          tryingToScrollDown = dY > VERT_THRESHOLD,
          tryingToScrollBigVert = dY < -BIG_VERT_THRESHOLD ||
            dY > BIG_VERT_THRESHOLD,
          
          scrollingUp = tryingToScrollUp && canScrollUp && vertAllowed,
          scrollingDown = tryingToScrollDown && canScrollDown && vertAllowed,
          
          scrollingVert = scrollingUp || scrollingDown,
            
          tryingToScrollLeft = dX < -HORIZ_THRESHOLD,
          tryingToScrollRight = dX > HORIZ_THRESHOLD,
          tryingToScrollBigHoriz = dX < -BIG_HORIZ_THRESHOLD ||
            dX < BIG_HORIZ_THRESHOLD,
          
          scrollingLeft = tryingToScrollLeft && canScrollLeft && horizAllowed,
          scrollingRight = tryingToScrollRight && canScrollRight && horizAllowed,
          
          scrollingHoriz = scrollingLeft || scrollingRight,
          
          //we are actually going to perform a scroll
          doScroll = scrollingHoriz || scrollingVert;
      
      if (tryingToScrollBigVert && !vertAllowed) {
        //don't eat up a vertical scroll just because it moves left or right 5 px
        return;
      }
      
      if (tryingToScrollBigHoriz && !horizAllowed) {
        //don't eat up a vertical scroll just because it moves left or right 5 px
        return;
      }
      
      if (doScroll) {
        that.scrollTop = scrollTopStart + dY;
        that.scrollLeft = scrollLeftStart + dX;
        //scrolling, so eat event
        event.preventDefault();
      } else if (!tryingToScrollUp && !tryingToScrollDown &&
                 !tryingToScrollLeft && !tryingToScrollRight) {
        //finger is wishy-washy - eat event so browser doesn't get control
        event.preventDefault();
      } else {
        if (tryingToScrollRight && horizAllowed) {
          showBlockade('right');
        }
        if (tryingToScrollLeft && horizAllowed) {
          showBlockade('left');
        }
        if (tryingToScrollUp && vertAllowed) {
          showBlockade('top');
        }
        if (tryingToScrollDown && vertAllowed) {
          showBlockade('bottom');
        }
      }
    });
    
    elem.bind(touchend, function () {
      touchStart = undefined;
    });
  }
  
  /**
   * @namespace
   * @name niagara.util.mobile.touchscroll
   */
  util.api('niagara.util.mobile.touchscroll', {
    touchScroll: touchScroll
  });
}());



//console.log('pageY: ' + pageY + ' pageX: ' + pageX + ' scrollTop: ' + 
//scrollTop + ' scrollLeft: ' + scrollLeft + ' startY: ' + 
//scrollStartPosY + ' startX: ' + scrollStartPosY + ' newStartY: ' +
//newScrollTop + ' newStartX: ' + newScrollLeft);