/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to navbar management in Niagara mobile apps.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true */
/*global $, baja, niagara */

(function () {
  "use strict";
  
  var util = niagara.util;

  util.require(
    'jQuery.mobile',
    'niagara.util.nav'
  );
  
////////////////////////////////////////////////////////////////
//Utility functions
////////////////////////////////////////////////////////////////
  
  function getIconUrls(icon) {
    if (typeof icon === 'string') {
      return [icon];
    } else if (icon instanceof baja.Icon) {
      return icon.getImageUris();
    } else {
      return ['/module/icons/x16/object.png'];
    }
  }
  
////////////////////////////////////////////////////////////////
//NavBar
////////////////////////////////////////////////////////////////
  
  /**
   * This class represents a JQM navbar for navigating through a Bajascript
   * app. The buttons will be labeled with different stops in our nav model
   * so that we can quickly navigate back up/down to any point in our 
   * tree we already have open.
   * 
   * <p>This class does not just have a single div - since we navigate back
   * and forth between property sheets using <code>$.mobile.changePage</code>,
   * the necessary navbar div will need to be dynamically generated and
   * inserted upon every new page view (at least until JQM implement persistent
   * header bars).
   * 
   * <p>When viewing a property sheet in a long parent-child component chain,
   * the first and last navbar buttons will be replaced with "..." buttons.
   * Clicking these buttons will scroll the navbar back and forth (no page
   * reloads or property sheet traversal is involved) so that you can jump
   * back and forth quickly to your desired spot.
   * 
   * <p>If a node has nav children (as in the case of a
   * <code>niagara.util.nav.NavFileNavModel</code>), the button in the nav bar
   * will have a down arrow indicating so. The node itself will be passed to
   * the <code>click</code> handler, so it is up to the app using this nav bar
   * to examine the node for children and handle the situation appropriately.
   * 
   * <p>
   * NavBar gets a Bajascript event mixin, so you can can attach to the
   * following events:
   * 
   * <ul>
   * <li><code>click</code> - a button in the nav bar was clicked. The argument
   * to the handler will be an object literal containing the following
   * properties:
   *   <ul>
   *   <li><code>a</code>: the &lt;a&gt; link element that was clicked</li>
   *   <li><code>linkActive</code>: a boolean value indicating whether or not
   *     the link clicked was already in the highlighted state or not</li>
   *   <li><code>node</code>: the <code>niagara.util.nav.NavModelNode</code>
   *     backing the particular nav button that was clicked</li>
   *   </ul>
   * </li>
   * </ul>
   * 
   * @class
   * @memberOf niagara.util.mobile
   * @param {niagara.util.nav.NavModel} navModel the nav model that will back
   * this nav bar
   */
  function NavBar(navModel) {
    var that = this;
    
    navModel.attach('change', function() {
      that.refresh(navModel.getSelectedNode());
    });
    
    this.navModel = navModel;
    this.currentIndices = {};
    this.LEFT_TEXT = "...";
    this.RIGHT_TEXT = "...";
  }
  
  /*
   * Give NavBar event capabilities
   */
  baja.event.mixin(NavBar.prototype);
    
  /**
   * Makes a nav bar div with <code>data-role="navbar"</code>, ready to be 
   * appended to a JQM header div.
   * 
   * @name niagara.util.mobile.nav.NavBar#makeDiv
   * @function
   * @private
   * @returns {jQuery} a nav bar div
   */
  NavBar.prototype.makeDiv = function makeDiv() {
    var that = this,
        div = $('<div class="nav" data-role="navbar" data-theme="b" />'),
        ul = $('<ul/>').appendTo(div),
        currentIndices = that.currentIndices,
        startIndex = currentIndices.startIndex,
        endIndex = currentIndices.endIndex,
        navModel = that.navModel;
    
    baja.iterate(startIndex, endIndex, function (i) {
      var li = $('<li/>'),
          a = $('<a data-theme="b" />'),
          navNode,
          navChildren;

      if (i === startIndex && i > 0) {
        //scroll left button
        a.addClass("scrollLeft");
        a.text(that.LEFT_TEXT);
      } else if (i === endIndex - 1 && i < navModel.size() - 1) {
        //scroll right button
        a.addClass("scrollRight");
        a.text(that.RIGHT_TEXT);
      } else {
        a.addClass("componentLink");
        navNode = navModel.get(i);
        navChildren = navNode.getNavChildren();
        
        a.data('node', navNode);
        a.text(baja.SlotPath.unescape(navNode.getNavDescription()));
        
        //if we have nav children, show a little down-arrow to indicate so
        if (navChildren.length) {
          a.data('icon', 'arrow-d');
          a.data('iconpos', 'right');
          a.data('children', navChildren);
        }
      }

      a.appendTo(li);
      li.appendTo(ul);
    });
    
    return div;
  };
    
  /**
   * Calculates the necessary indices for the property sheet we are currently
   * viewing. Returns an object literal with the following properties:
   * 
   * <ul>
   * <li><code>nodeIndex</code>: the index of the nav node viewed,
   * in the context of the complete nav model (i.e. 0 = station root and
   * increases the further you go to the right)</li>
   * <li><code>selectedIndex</code>: the index, in the context of the
   * currently visible navbar buttons, of the highlighted button. A value of
   * 0 means we are viewing the node referenced by the button on
   * the far left, so highlight that one. Likewise, a value of 4 for the
   * button on the far right.</li>
   * <li><code>startIndex</code>: the node index of the node
   * to be displayed on the far left of the nav bar. If we are viewing
   * the station root, this will likely be 0.</li>
   * <li><code>endIndex</code>: the node index + 1 of the node
   * to be displayed on the far right of the nav bar. If we have navigated
   * to the furthest point down into the nav model, this will be
   * <code>navModel.size()</code>.</li>
   * </ul>
   * 
   * @name niagara.util.mobile.nav.NavBar#getNavBarIndices
   * @function
   * @private
   * 
   * @param {Number} nodeIndex the index of the property sheet to be viewed
   * @param {Number} [numberOfSheets] the number of property sheets 
   * currently loaded. If omitted, <code>this.navModel.size()</code> will be
   * used.
   * @returns {Object} an object literal with index information
   */
  NavBar.prototype.getNavBarIndices = function getNavBarIndices(nodeIndex, numberOfSheets) {
    numberOfSheets = numberOfSheets || this.navModel.size();
    
    var startIndex = nodeIndex - 2,
        endIndex = nodeIndex + 3,
        delta = 0,
        nav;
    
    if (endIndex > numberOfSheets) {
      delta = endIndex - numberOfSheets;
    } else if (startIndex < 0) {
      delta = startIndex;
    }
    endIndex -= delta;
    startIndex -= delta;
    
    startIndex = Math.max(startIndex, 0);
    endIndex = Math.min(endIndex, numberOfSheets);
    
    return {
      nodeIndex: nodeIndex,
      selectedIndex: nodeIndex - startIndex,
      startIndex: startIndex,
      endIndex: endIndex
    };
  };
    
  /**
   * Set the nav bar scrolling to the given node index.
   * 
   * @name niagara.util.mobile.nav.NavBar#scrollTo
   * @function
   * @private
   * 
   * @param {Number} index the index of the property sheet to scroll to
   */
  NavBar.prototype.scrollTo = function scrollTo(index) {
    this.currentIndices = this.getNavBarIndices(index);
  };
    
  /**
   * To be called when the left-side "..." scroll button is clicked. Scrolls
   * our nav bar two spaces to the left.
   * 
   * @name niagara.util.mobile.nav.NavBar#scrollLeft
   * @function
   * @private
   */
  NavBar.prototype.scrollLeft = function scrollLeft() {
    var index = Math.max(this.currentIndices.startIndex, 0);
    this.scrollTo(index);
  };
    
  /**
   * To be called when the right-side "..." scroll button is clicked. Scrolls
   * our nav bar two spaces to the right.
   * 
   * @name niagara.util.mobile.nav#scrollRight
   * @function
   * @private
   */
  NavBar.prototype.scrollRight = function scrollRight() {
    var index = Math.min(this.currentIndices.endIndex - 1, this.navModel.size());
    this.scrollTo(index);
  };
    
  /**
   * Enables click handlers on our nav bar buttons for scrolling left and
   * right and for linking to other nav nodes.
   * 
   * @name niagara.util.mobile.nav.NavBar#armHandlers
   * @function
   * @private
   * 
   * @param {jQuery} navDiv the nav bar div (output from <code>makeDiv</code> - 
   * must already have been appended to a JQM page)
   */
  NavBar.prototype.armHandlers = function armHandlers(navDiv) {
    var that = this;
    
    navDiv.delegate('a.componentLink', 'click', function () {
      var $this = $(this),
          node = $this.data('node');
      
      that.fireHandlers('click', baja.error, that, {
        a: $this,
        linkActive: $this.hasClass('ui-btn-active'),
        node: node
      });
    });
    
    navDiv.delegate('a.scrollLeft', 'click', function () {
      that.scrollLeft();
      that.replaceNavBar(that.navModel.getSelectedNode().getValue());
    });
    
    navDiv.delegate('a.scrollRight', 'click', function () {
      that.scrollRight();
      that.replaceNavBar(that.navModel.getSelectedNode().getValue());
    });
  };
  
  /**
   * Builds the nav bar div and appends it to the header of the current
   * JQM page. After this method returns, the nav bar will be fully
   * constructed, visible, and clickable.
   *  
   * @name niagara.util.mobile.nav.NavBar#replaceNavBar
   * @function
   * @private
   * 
   * @param {jQuery} page the JQM page in which to insert the navbar
   * @returns {jQuery} the nav bar div, already appended to the current
   * active JQM page
   */
  NavBar.prototype.replaceNavBar = function replaceNavBar(page) {
    if (typeof page.getPage === 'function') {
      page = page.getPage();
    }
    
    var header = page.children('div:jqmData(role=header)'),
        oldNav = header.children('div:jqmData(role=navbar)'),
        nav = this.makeDiv();
    
    if (oldNav.length) {
      oldNav.replaceWith(nav);
    } else {
      header.append(nav);
    }
    nav.navbar();
    this.armHandlers(nav);
    
    //set the icons - can only do this after navbar() is called unfortunately
    nav.find('a.componentLink').each(function (index) {
      var a = $(this),
          node = a.data('node'),
          urls = getIconUrls(node.getNavIcon()),
          
          //the &nbsp; in the span is necessary to get the icon to bottom-align
          //don't ask me why
          //i don't know
          div = $('<span class="navbar-icon-container ui-btn-text">&nbsp;</span>')
            .prependTo(a.children('span'));
      
      baja.iterate(urls, function (url) {
        div.append($('<img/>').attr('src', url));
      });
      
      a.find('.ui-icon').addClass('ui-disabled');
    });
    
    return nav;
  };
    
  /**
   * Refreshes the nav bar to reflect which node is currently
   * being viewed. This is <code>NavBar</code>'s one public method which
   * should be called just after navigating to a different page.
   * 
   * @name niagara.util.mobile.nav.NavBar#refresh
   * @function
   * @param {jQuery} page the JQM page in which the navbar lives
   */
  NavBar.prototype.refresh = function refresh(page) {
    page = (page && page.length) ? page : this.$page;
    
    if (!page || !page.length) {
      return;
    }
    
    this.$page = page;
    
    this.scrollTo(this.navModel.getSelectedIndex());
    
    var nav = this.replaceNavBar(page),
        links = nav.find('ul li a'),
        selectedLink;
    
    if (links.length > 1) {
      //selected button should have its 'i have nav children' icon enabled -
      //all others should have it grayed out
      selectedLink = links.eq(this.currentIndices.selectedIndex);
      selectedLink.addClass('ui-btn-active');
      selectedLink.find('.ui-icon').removeClass('ui-disabled');
    }
  };
  
  util.api('niagara.util.mobile.nav', {
    NavBar: NavBar
  });
}());
