/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Niagara Mobile Utils - functions relating to the 
 * "available commands" button and associated dialog in our JQM page headers
 *
 * @author Gareth Johnson
 */

// JsLint options (see http://www.jslint.com )
/*jslint white: true, vars: true */

/*global $, baja, niagara, location, window*/ 

////////////////////////////////////////////////////////////////
// App Command Management
////////////////////////////////////////////////////////////////

(function mobileCommands() {
  "use strict";
  
  niagara.util.require(
    'niagara.util.flow',
    'niagara.util.mobile.dialogs'
  );
  
  var util = niagara.util,
      lex = null,
      dialogs = util.mobile.dialogs,
      defaultCommands = [],
      commandsButtonHtml = "<a class='commandsButton profileHeader' title='{title}' data-icon='menu' data-role='button' data-iconpos='notext'></a>",
      availableActionsCmd,
      homeCmd,
      logoutCmd,
      Command,
      AsyncCommand;

  function getMobileLex() {
    if (!lex) {
      lex = baja.lex('mobile');
    }
    return lex;
  }
  
  ////////////////////////////////////////////////////////////////
  // Command Button and HTML
  ////////////////////////////////////////////////////////////////
        
  /**
   * Returns HTML to be used for the command button (an
   * <code>&lt;a&gt;</code> tag).
   * 
   * @memberOf niagara.util.mobile.commands
   * 
   * @returns {jQuery} an <code>&lt;a&gt;</code> tag, ready to be appended
   * into a JQM page header
   */
  function getCommandsButton() {
    return $(commandsButtonHtml.patternReplace({
      title: getMobileLex().get('menu')
    }));
  }
  
  /**
   * Shows a listview dialog with commands for the user to select.
   * The entries in the dialog are populated by a command object when the dialog is
   * shown.
   * 
   * 
   * @memberOf niagara.util.mobile.commands
   * @param {Array} commands an array of commands to show
   * @param {String} [title] a title for the command dialog
   */
  function showCommandsDialog(commands, title, callbacks) {
    // Fire a jQuery event just before the commands are shown
    $("a.commandsButton").trigger("onShowCommands", commands);
    
    callbacks = util.callbackify(callbacks);
    
    title = title || getMobileLex().get('commands.title');
  
    if (commands.length === 0) {
      dialogs.ok({
        title: title,
        content: getMobileLex().get('commands.noneAvailable') 
      });
    }
    else {
      var dialogInvocation = dialogs.cancel({
        title: title,
        content: function (targetElement, callbacks) {
          var ul = $('<ul data-role="listview" data-theme="c" />');
          
          baja.iterate(commands, function (command) {
            var li = $('<li/>').appendTo(ul),
                link = $('<a/>').appendTo(li);
            link.text(command.getDisplayName());
            link.click(function () {
              dialogInvocation.invoke(function (callbacks) {
                if (command instanceof AsyncCommand) {
                  command.invoke(callbacks);
                } else {
                  command.invoke();
                  callbacks.ok();
                }
              });
            });
          });
          
          targetElement.html(ul);
          ul.listview();
          callbacks.ok();
        },
        callbacks: callbacks
      });
    }
  }

  ////////////////////////////////////////////////////////////////
  // Default Commands
  ////////////////////////////////////////////////////////////////
    
  /**
   * Add a Command to the end of the default list.
   *
   * @memberOf niagara.util.mobile.commands
   * @param {Command} cmd a command that will appear when the commands button is clicked.
   *                      Please note, to add multiple commands in one go, specify multiple
   *                      command arguments for this function.     
   */
  function addDefaultCommand() {
    var args = Array.prototype.slice.call(arguments);
    
    baja.iterate(args, function(c) {
      // Add the command (providing it isn't already present)
      if (!defaultCommands.contains(c)) {
        defaultCommands.push(c);
      }
    });
  }
  
  /**
   * Add a Command to the top of the default list.
   *
   * @memberOf niagara.util.mobile.commands
   * @param {Command} cmd a command that will appear when the commands button is clicked.
   *                      Please note, to add multiple commands in one go, specify multiple
   *                      command arguments for this function.     
   */
  function prependDefaultCommand() {
    var args = Array.prototype.slice.call(arguments);
    
    baja.iterate(args, function(c) {
      // Add the command (providing it isn't already present)
      if (!defaultCommands.contains(c)) {
        if (defaultCommands.length) {
          defaultCommands.splice(0, 0, c);
        } else {
          defaultCommands.push(c);
        }
      }
    });
  }
  
  /**
   * Remove a Command from the default list.
   *
   * @memberOf niagara.util.mobile.commands
   * @param {Command|Array} cmd the command Object to remove.
   *                            Please note, to add multiple commands in one go, specify multiple
   *                            command arguments for this function.     
   */
  function removeDefaultCommand(cmd) {
    var args = Array.prototype.slice.call(arguments);
    
    if (args.length > 1) {
      baja.iterate(args, function(c) {
        removeDefaultCommand(c);
      });
    }
    else {
      baja.iterate(defaultCommands, function(defC, i) {
        if (defC === cmd) {
          defaultCommands.splice(i, 1);
          return true;
        }
      });
    }
  }
  
  /**
   * Return an array of the default Commands.
   * <p>
   * Please note, this returns a defensive copy of the default commands array.
   *
   * @memberOf niagara.util.mobile.commands
   * @returns {Array}
   */
  function getDefaultCommands() {
    return defaultCommands.slice(0);
  }
  
  /**
   * Set our default commands array to the given array of Command objects.
   *
   * @param cmds - Array of niagara.util.mobile.commands.Command objects.
   *  
   * @memberOf niagara.util.mobile.commands
   */
  function setDefaultCommands(cmds){
    defaultCommands = cmds;
  }
  
  ////////////////////////////////////////////////////////////////
  // Command
  ////////////////////////////////////////////////////////////////
  
  /**
   * @class Command
   * <p>
   * A Command represents an item in the Commands Menu and can be invoked by the user.
   *
   * @name niagara.util.mobile.commands.Command
   *
   * @param {String} displayName the display name for the Command. The display name text will be lazily passed   
   *                             through a baja.Format for lexicon translation.
   * @param {Function|baja.Ord|Array} value used when the command is invoked. Functions will
   *                                        be invoked, an Array will attempt to show another command
   *                                        dialog and baja.Ord will attempt a hyperlink. If the value
   *                                        is a function, and this Command is intended to be invoked
   *                                        from a dialog, then the function may return false to indicate
   *                                        that the dialog should <i>not</i> be closed after the command
   *                                        is invoked - otherwise the dialog will automatically close
   *                                        after invocation.
   */
  Command = function(displayName, value) {
    this.$displayName = displayName;
    this.$value = value;
  };
    
  /**
   * Return the Command's display name.
   *
   * @memberof niagara.util.mobile.commands.Command#
   * @returns {String}
   */   
  Command.prototype.getDisplayName = function() {
    // Run the display name through a baja.Format filter
    if (this.$formatDisplayName === undefined) {
      this.$formatDisplayName = baja.Format.format(this.$displayName);
    }
    return this.$formatDisplayName;
  };
  
  /**
   * Return the Command's value.
   * 
   * @memberof niagara.util.mobile.commands.Command#
   * @returns {Function|baja.Ord|Array}
   */
  Command.prototype.getValue = function() {
    return this.$value;
  };
    
  /**
   * Invoke the Command. If the value of the Command is a function, then 
   * any arguments passed to <code>invoke()</code> will also be passed straight
   * through to the function.
   * 
   * @memberOf niagara.util.mobile.commands.Command#
   */
  Command.prototype.invoke = function() {
    var value = this.$value,
        args;
    
    if ($.isArray(value)) {
      // If this is an array then assume it's an array of commands to show
      showCommandsDialog(value);
    } 
    else if (typeof value === 'function') {
      // If this is a function then just invoke it with this command as 'this'
      args = Array.prototype.slice.call(arguments);
      return value.apply(this, args);
    } 
    else if (value instanceof baja.Ord || typeof value === 'string') {
      // If an ORD (this also covers full http links) then attempt to hyperlink to it
      util.mobile.linkToOrd(value);
      // Since JQM will perform its own hyperlink, we don't want the dialog
      // to try to close itself
      return false;
    }
  };
  
  /**
   * A Command for executing an asynchronous function. 
   * 
   * @name niagara.util.mobile.commands.AsyncCommand
   * @class
   * @extends niagara.util.mobile.commands.Command
   * @param {String} displayName
   * @param {Function} value the command value must be a function that takes
   * a single parameter - an object containing ok/fail callbacks
   */
  AsyncCommand = function (displayName, value) {
    baja.strictArg(value, Function);
    Command.call(this, displayName, value);
  };
  AsyncCommand.prototype = new Command();
  
  /**
   * Executes this command's asynchronous function.
   * @name niagara.util.mobile.commands.AsyncCommand#invoke
   * @function
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  AsyncCommand.prototype.invoke = function (callbacks) {
    callbacks = util.callbackify(callbacks);
    this.$value(callbacks);
  };
  
  ////////////////////////////////////////////////////////////////
  // Event Handling
  ////////////////////////////////////////////////////////////////
  
  function showCommandsHandler() {  
    // Get the default list of commands
    var cmds = getDefaultCommands(),
        handler = util.mobile.pages && util.mobile.pages.getCurrentHandler(); 
    
    // If the pages framework is being used then see if the current handler defines a list of commands
    if ( handler && typeof handler.getCommands === "function") {
    
      // Ask for the commands (while passing in the original default list)
      cmds = util.mobile.pages.getCurrentHandler().getCommands({
        commands: cmds,
        page: util.mobile.pages.getCurrent()
      });
    }
  
    // Show the commands dialog whenever this button is clicked
    showCommandsDialog(cmds);
    
    // Return false to stop event bubbling
    return false;
  }
  
  // Called when the DOM is ready...
  $(function() {
    $("a.commandsButton").live("click", showCommandsHandler);
  });
  
  ////////////////////////////////////////////////////////////////
  // Logout Command
  ////////////////////////////////////////////////////////////////
  
  // Automatically add the logout command
  logoutCmd = new Command("%lexicon(mobile:commands.logout)%", function() {
    var url = location.protocol + "//" + location.host + "/logout";
    window.location.assign(url);
  });
  
  if (niagara.view.profile.showLogoutCmd) {
    prependDefaultCommand(logoutCmd);
  }

  ////////////////////////////////////////////////////////////////
  // Home Command
  ////////////////////////////////////////////////////////////////
  
  // Automatically add the home command if we can
  homeCmd = new Command("%lexicon(mobile:commands.home)%", function() {
    // Link to user home
    util.mobile.linkToOrd(baja.getUserHome());
  });
  
  if (niagara.view.profile.showHome) {
    prependDefaultCommand(homeCmd);
  }
    
  ////////////////////////////////////////////////////////////////
  // Available Actions
  ////////////////////////////////////////////////////////////////
  
  // A Command to show available actions on a component.
  availableActionsCmd = new Command("%lexicon(mobile:actions)%", function (component) {
    var commands = [];
    
    component.getSlots(function (slot) {
      return util.slot.isFireable(slot);
    }).each(function (slot) {
      commands.push(new AsyncCommand(component.getDisplayName(slot), function (callbacks) {
        dialogs.action({
          component: component,
          slot: slot,
          facets: component.get('facets') || {},
          callbacks: callbacks
        });
      }));
    });

    showCommandsDialog(commands, this.getDisplayName());
  });
  
  /**
   * Return a Command to show available actions on a component.
   * 
   * @memberOf niagara.util.mobile.commands
   * @returns {Command}
   */
  function getAvailableActionsCmd() {
    return availableActionsCmd;
  }
  
  /**
   * Return a Command to return to the current user's home node.
   * 
   * @memberOf niagara.util.mobile.commands
   * @returns {Command}
   */
  function getHomeCmd() {
    return homeCmd;
  }
  
  /**
   * Return a Command to log out the current user.
   * 
   * @memberOf niagara.util.mobile.commands
   * @returns {Command}
   */
  function getLogoutCmd() {
    return logoutCmd;
  }
  
  ////////////////////////////////////////////////////////////////
  // Export
  ////////////////////////////////////////////////////////////////
  
  /**
   * @namespace
   * @name niagara.util.mobile.commands
   */
  util.api("niagara.util.mobile.commands", {
    "public": {
      Command: Command,
      AsyncCommand: AsyncCommand,
      
      getAvailableActionsCmd: getAvailableActionsCmd,      
      getCommandsButton: getCommandsButton,
      getHomeCmd: getHomeCmd,
      getLogoutCmd: getLogoutCmd,
      showCommandsDialog: showCommandsDialog,
      addDefaultCommand: addDefaultCommand,
      prependDefaultCommand: prependDefaultCommand,
      removeDefaultCommand: removeDefaultCommand,
      getDefaultCommands: getDefaultCommands,
      setDefaultCommands: setDefaultCommands,
      showCommandsHandler: showCommandsHandler
    }
  });
}());