/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview General utility functions for Niagara web apps.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, browser: true, bitwise: true, plusplus: true,
    nomen: true, regexp: true */
/*global niagara, $, baja, BaseBajaObj */

(function () {
  "use strict";
  
  var iterate = baja.iterate,
      lexMap = {};
  
  /**
   * Do nothing, and return <code>arguments[0]</code>.
   * @memberOf niagara.util
   */
  function noop(thing) {
    return thing;
  }
  
  /**
   * Returns a function that will only execute a certain period of time after
   * the last time it was called. Styled after underscore.js's debounce 
   * function.
   * 
   * @memberOf niagara.util
   * @param {Function} func the function to execute
   * @param {Number} interval the amount of time to wait (in millis) after the
   * last function call before it will actually be invoked.
   */
  function debounce(func, interval) {
    var ticket;
    
    return function () {
      var that = this,
          args = Array.prototype.slice.call(arguments);
      
      if (ticket && !ticket.isExpired()) {
        ticket.cancel();
      }
      
      ticket = baja.clock.schedule(function () {
        ticket = null;
        func.apply(that, args);
      }, interval);
    };
  }

  /**
   * Binds the execution scope of a function. Any additional arguments past
   * <code>func</code> and <code>scope</code> will be curried to the returned
   * function.
   * 
   * @memberOf niagara.util
   * 
   * @param {Function} func the function to bind
   * @param {Object} scope the execution scope to apply the function to
   * @returns {Function} a newly bound (and possibly curried) function
   */
  function bind(func, scope) {
    var curryArgs = Array.prototype.slice.call(arguments, 2);
    return function () {
      var args = Array.prototype.slice.call(arguments);
      return func.apply(scope, curryArgs.concat(args));
    };
  }
  
  /**
   * Ensure that a callback object literal exists and has ok and fail callbacks
   * - default them to <code>baja.ok</code> and <code>baja.fail</code>,
   * respectively, if they do not exist.
   * 
   * @memberOf niagara.util
   * @private
   * @param {Object} [callbacks] an object containing ok/fail callbacks
   * (the object may be missing one or both callbacks or may itself be
   * undefined)
   * @param {Function} [defaultOk] the ok handler to use if none provided
   * (defaults to baja.ok)
   * @param {Function} [defaultFail] the fail handler to use if none provided
   * (defaults to baja.fail)
   */
  function callbackify(callbacks, defaultOk, defaultFail) {
    callbacks = baja.objectify(callbacks, 'ok');
    callbacks.ok = callbacks.ok || defaultOk || baja.ok;
    callbacks.fail = callbacks.fail || defaultFail || baja.fail;
    return callbacks;
  }
    
  /**
   * Performs indexOf behavior on an array, in the case that
   * Array.prototype.indexOf does not exist (i.e. IE)
   * 
   * @memberOf niagara.util
   * 
   * @param {Array} array
   * @param element
   * @param {Function} [equalsFunc] a function that takes two parameters:
   * the element from the given array, and the element being sought. Returns
   * true if the array element matches the sought-after element. If omitted,
   * the standard === will be used.
   * @returns the index of the element within the array, or -1 if it does not
   * exist within the array
   */
  function indexOf(array, element, equalsFunc) {
    var i;
    
    equalsFunc = equalsFunc || function (a, b) { return a === b; };
    
    if (array.indexOf && !equalsFunc) {
      return array.indexOf(element);
    } else {
      for (i = 0; i < array.length; i++) {
        if (equalsFunc(array[i], element)) {
          return i;
        }
      }
      return -1;
    }
  }
  
  /**
   * Returns true if the given array contains the given element.
   * 
   * @memberOf niagara.util
   * 
   * @param {Array} array
   * @param element
   * @param {Function} [equalsFunc]
   * @see niagara.util.indexOf
   */
  function contains(array, element, equalsFunc) {
    return indexOf(array, element, equalsFunc) >= 0;
  }
  
  /**
   * Defines a namespace.
   * 
   * <p>The arguments to this function are either a dot-separated string,
   * or an object ("scope") followed by a dot-separated string. If no
   * scope is provided, <code>window</code> will be used.
   * 
   * <p>The objects specified by the given names will be created, one as a
   * parent of the next. No objects will be overwritten - they will only
   * be created if they do not already exist.
   * 
   * <p>Example:
   * <code>
   * <p>var myNamespace = util.namespace('parent.child.myNamespace');
   * <p>var subNamespace = util.namespace(myNamespace, 'subNamespace'); //myNamespace is not overwritten
   * <p>var subNamespace2 = util.namespace(myNamespace, 'subNamespace2'); //subNamespace is not overwritten
   * </code>
   * 
   * @memberOf niagara.util
   */
  function namespace(ns, names) {
    if (typeof ns !== 'object') {
      names = ns;
      ns = window;
    }
    
    iterate(names.split('.'), function (name, i) {
      if (!ns[name]) {
        ns[name] = new BaseBajaObj();
      }
      ns = ns[name];
    });
    return ns;
  }
  
  /**
   * Verify the existence of one or more namespaces. Each is given in the
   * usual dot-separated form, e.g. <code>'niagara.util.schedule'</code>. If
   * any of the namespaces have not been installed already, an exception
   * will be thrown.
   * 
   * @memberOf niagara.util
   * 
   * @throws {Error} if any required namespaces do not exist
   */
  function require() {
    iterate(Array.prototype.slice.call(arguments), function (namespace) {
      baja.strictArg(namespace, String);
      
      var scope = window;
      
      iterate(namespace.split('.'), function (name) {
        if (!scope[name]) {
          throw new Error("Requirement " + namespace + " not satisfied.");
        }
        
        scope = scope[name];
      });
    });
  }
  
  /**
   * Defines an external API. This function accepts sets of both public and
   * private properties. Public properties will be assigned directly to the
   * given object, while private properties will be assigned <i>only</i>
   * if <code>niagara.debug</code> is set to true.
   * 
   * <p>The <code>iface</code> parameter is an object literal. If <code>iface
   * </code> has either <code>'public'</code> or <code>'private'</code>
   * properties, then these properties will be used as public and private.
   * Otherwise, the entire object literal will be assumed to be public.
   * 
   * @memberOf niagara.util
   * 
   * @param {Object|String} obj the object to assign properties to. If this
   * is a string, it will be converted to a namespace first.
   * @param {Object} iface an object containing 'public' and 'private'
   * properties. If both of these are missing, the entire object will be
   * assumed public.
   * @param {Object} [iface.public] an object containing public
   * properties
   * @param {Object} [iface.private] an object containing private
   * properties
   * @returns obj
   */
  function api(obj, iface) {
    var pub, priv;
    
    if (typeof obj === 'string') {
      obj = namespace(obj);
    }
    
    if (iface['public'] || iface['private']) {
      pub = iface['public'] || {};
      priv = iface['private'] || {};
    } else {
      pub = iface || {};
    }
    
    $.extend(obj, pub);
    if (niagara.debug) {
      $.extend(obj, priv);
    }
    
    return obj;
  }
  
  /**
   * Returns an object with <code>get()</code> and <code>getModuleName()</code>
   * functions (like a regular lexicon object), but will not actually make the
   * network call to retrieve the lexicon until first time the 
   * <code>get()</code> function is actually called. This allows you to create
   * a lexicon object in your usual var statement and not have to worry about
   * network calls being made until you actually need to retrieve a lexicon
   * string. May save you an extra network call on app startup.
   * 
   * @memberOf niagara.util
   * @param {String} name the module name to retrieve the lexicon from
   * @returns {Object} an object that can be used in place of a regular
   * <code>baja.lex()</code> lexicon
   */
  function lazyLex(name) {
    var lex = lexMap[name];
    
    if (!lex) {
      lex = (function () {
        var bajaLex;
        
        function getBajaLex() {
          if (!bajaLex) {
            bajaLex = baja.lex(name);
          }
          return bajaLex;
        }
        
        return {
          get: function (obj) {
            return getBajaLex().get(obj);
          },
          getModuleName: function () {
            return name;
          }
        };
      }());
      
      lexMap[name] = lex;
    }
    
    return lex;
  }
  
  /**
   * @namespace
   * @name niagara.util
   */
  api('niagara.util', {
    api: api,
    bind: bind,
    callbackify: callbackify,
    contains: contains,
    debounce: debounce,
    indexOf: indexOf,
    iterate: iterate,
    lazyLex: lazyLex,
    namespace: namespace,
    noop: noop,
    require: require
  });
  
  
  (function customSelectorsFunctions() {
    /**
     * jQuery custom selector that returns the next-to-last element in a 
     * stack of elements. If the stack has only one element in it, that
     * element will be returned. Install using the standard 
     * <code>$.expr[':'].nextToLast = nextToLast</code>
     *  
     * @memberOf niagara.util.customSelectors
     */
    function nextToLast(obj, index, meta, stack) {
      return stack.length === 1 || index === stack.length - 2;
    }
    
    /**
     * jQuery custom selector that returns all elements whose boundaries
     * (calculated using <code>.offset()</code>, not <code>.position()</code>)
     * enclose the given coordinates. Install using the standard
     * <code>$.expr[':'].enclosingPoint = enclosingPoint</code>
     * 
     * @memberOf niagara.util.customSelectors
     */
    function enclosingPoint(obj, index, meta, stack) {
      var elem = $(obj),
          pos = elem.offset(),
          params = meta[3].split(','),
          x = parseInt(params[0], 10),
          y = parseInt(params[1], 10);
      return y >= (pos.top) && y < (pos.top + elem.height()) &&
             x >= (pos.left) && x < (pos.left + elem.width());
    }
    
    /**
     * Installs all custom selectors defined in this namespace.
     * 
     * @memberOf niagara.util.customSelectors
     * 
     * @param {jQuery} target$ the target jQuery object on which to install
     * the custom selectors
     */
    function installAll(target$) {
      $.extend(target$.expr[':'], {
        nextToLast: nextToLast,
        enclosingPoint: enclosingPoint
      });
    }
    
    /**
     * @namespace
     * @name niagara.util.customSelectors
     */
    api('niagara.util.customSelectors', {
      enclosingPoint: enclosingPoint,
      installAll: installAll,
      nextToLast: nextToLast
    });
  }());
  

  (function timeFunctions() {
        /** 
         * Number of milliseconds in one day 
         * @name niagara.util.time.MILLIS_IN_DAY 
         */
    var MILLIS_IN_DAY =      86400000,
        /**
         * Number of milliseconds in one hour
         * @name niagara.util.time.MILLIS_IN_HOUR
         */
        MILLIS_IN_HOUR =      3600000,
        /**
         * Number of milliseconds in half an hour
         * @name niagara.util.time.MILLIS_IN_HALF_HOUR
         */
        MILLIS_IN_HALF_HOUR = 1800000,
        /**
         * Number of milliseconds in one minute
         * @name niagara.util.time.MILLIS_IN_MINUTE
         */
        MILLIS_IN_MINUTE =      60000,
        /**
         * Number of milliseconds in one second
         * @name niagara.util.time.MILLIS_IN_SECOND
         */
        MILLIS_IN_SECOND =       1000;
    
    
    /**
     * Normalizes milliseconds past midnight into a baja.Time. Passing in a
     * negative number or more than MILLIS_IN_DAY will not throw an error like
     * baja.Time.make() will.
     * 
     * @memberOf niagara.util.time
     * @function
     * 
     * @param {Number} millis milliseconds past midnight
     * @return baja.Time corresponding time of day
     */
    function makeTime(millis) {
      millis = (millis + MILLIS_IN_DAY) % MILLIS_IN_DAY;
      return baja.Time.make(millis);
    }
    
    /**
     * For a Javascript <code>Date</code>, returns the number of milliseconds
     * past midnight (UTC). For a <code>baja.Time</code>, returns 
     * <code>getTimeOfDayMillis()</code>.
     * 
     * @memberOf niagara.util.time
     * @param {Date|baja.Time} time
     * @returns {Number} milliseconds past midnight
     */
    function millisOfDay(date) {
      if (date instanceof Date) {
        return date.getTime() % MILLIS_IN_DAY;
      } else if (date instanceof baja.Time) {
        return date.getTimeOfDayMillis();
      } else if (typeof date === 'number') {
        return makeTime(date).getTimeOfDayMillis();
      }
    }
    
    /**
     * @memberOf niagara.util.time
     * @param {Date|baja.Time} time
     * @returns {Number} the number of milliseconds past the hour.
     */
    function millisOfHour(date) {
      if (date instanceof Date) {
        return date.getTime() % MILLIS_IN_HOUR;
      } else if (date instanceof baja.Time) {
        return date.getTimeOfDayMillis() % MILLIS_IN_HOUR;
      } else if (typeof date === 'number') {
        return makeTime(date).getTimeOfDayMillis() % MILLIS_IN_HOUR;
      }
    }
    
    /**
     * Rounds the given <code>Date</code> or <code>baja.Time</code> to the
     * nearest half hour.
     * 
     * @memberOf niagara.util.time
     * @param {Date|baja.Time} time
     * @returns {Date|baja.Time} the input time, rounded to the nearest half
     * hour
     */
    function roundToHalfHour(date) {
      var moh = millisOfDay(date),
          rounded = Math.round(moh / MILLIS_IN_HALF_HOUR) * MILLIS_IN_HALF_HOUR;
      
      if (date instanceof Date) {
        return new Date(rounded);
      } else if (date instanceof baja.Time) {
        return baja.Time.make({relTime: baja.RelTime.make(rounded)});
      } else if (typeof date === 'number') {
        return rounded;
      }
    }

    function zeroPad(num) {
      return (num < 10 ? '0' : '') + num;
    }
    
    function toTimeString(hours, minutes, seconds) {
      var arr = [zeroPad(hours), zeroPad(minutes), zeroPad(seconds || 0)];
      return arr.join(':');
    }
    
    /** 
     * @namespace
     * @name niagara.util.time
     */
    api('niagara.util.time', {
      MILLIS_IN_DAY: MILLIS_IN_DAY,
      MILLIS_IN_HOUR: MILLIS_IN_HOUR,
      MILLIS_IN_HALF_HOUR: MILLIS_IN_HALF_HOUR,
      MILLIS_IN_MINUTE: MILLIS_IN_MINUTE,
      MILLIS_IN_SECOND: MILLIS_IN_SECOND,
      makeTime: makeTime,
      millisOfDay: millisOfDay,
      millisOfHour: millisOfHour,
      roundToHalfHour: roundToHalfHour,
      toTimeString: toTimeString,
      zeroPad: zeroPad
    });
  }());

  

  (function aopFunctions() {
    function getFunction(obj, functionName) {
      return iterate(obj, function (obj) { 
        return obj[functionName]; 
      }, function (obj) {
        return obj.prototype;
      });
    }
    function doWrapFunction(obj, methodName, newFunction, wrapFunctions) {
      if (typeof wrapFunctions !== 'function') {
        throw "missing AOP function wrapper";
      }
      var oldFunction,
          wrappedFunction;
      
      if (typeof obj === 'function' && typeof methodName === 'function') {
        newFunction = obj;
        oldFunction = methodName;
        wrappedFunction = wrapFunctions(newFunction, oldFunction);
        
        //ensure we maintain the old prototype in case we're advising a constructor
        wrappedFunction.prototype = oldFunction.prototype;
        
        return wrappedFunction;
      } else if ((typeof obj === 'object' || typeof obj === 'function') &&
          typeof methodName === 'string' &&
          typeof newFunction === 'function') {
        oldFunction = getFunction(obj, methodName);
        obj[methodName] = doWrapFunction(newFunction, oldFunction, null, wrapFunctions);
      } else {
        throw "AOP functions must be called with argument signature " +
          "[Function, Function] or [Object, String, Function]";
      }
    }
    /**
     * Describes behavior that should occur before a particular method on
     * an object is called.
     * 
     * This method may also be called with only two parameters: 
     * <code>beforeFunc</code> and <code>origFunc</code>. 
     * <code>beforeFunc</code> should accept a single parameter: an array of
     * arguments that would have ordinarily been passed to 
     * <code>origFunc</code>. <code>beforeFunc</code> should perform any
     * processing of these arguments it needs to, including alteration of the
     * arguments themselves, and return a new array. This array will be
     * passed to <code>origFunc</code> using <code>apply()</code>. If
     * <code>beforeFunc</code> returns <code>false</code>, then
     * <code>afterFunc</code> will not be called at all and 
     * <code>undefined</code> will be returned.
     * 
     * @memberOf niagara.util.aop
     * @param {Object} obj an object containing the method to apply the aspect
     * to. If this object is a prototype then every object created with the
     * prototype will receive this new behavior.
     * 
     * @param {String} methodName the name of the method to apply the aspect to
     * 
     * @param {Function} beforeFunction the function to execute before the
     * target function. <code>beforeFunction</code> can accept a single 
     * parameter which is an array containing the arguments passed to the 
     * target function. <code>beforeFunction</code> has the option of 
     * returning a new array of arguments which will then be passed into the
     * target function in place of the original arguments; if nothing is
     * returned, the original arguments array will be used. Please note that
     * in either case, any modifications made to the arguments array within
     * <code>beforeFunction</code> will still pass through into the target
     * function - however they will NOT pass through into any AOP functions
     * registered using <code>after</code>. <code>beforeFunction</code> may
     * also return <code>false</code> to cancel the execution of
     * the target function.
     */
    function before(obj, methodName, beforeFunction) {
      return doWrapFunction(obj, methodName, beforeFunction, function (newFunc, oldFunc) {
        return function () {
          var args = Array.prototype.slice.call(arguments),
              newArgs = newFunc.call(this, args);
          if (newArgs !== false) {
            return oldFunc.apply(this, newArgs || args);
          }
        };
      });
    }
    
    /**
     * Describes behavior that should occur after a particular method on
     * an object is called.
     * 
     * This method may also be called with only two parameters: 
     * <code>func1</code> and <code>func2</code>. <code>func2</code> should 
     * take two parameters: <code>args</code>, an array containing the arguments
     * passed to <code>func1</code>, and <code>value</code>, the value returned
     * from <code>func1</code>. <code>after</code> will return a new function
     * that runs <code>func1</code> and <code>func2</code> in sequence, 
     * returning the value returned from <code>func2</code>. If func2 returns
     * <code>undefined</code>, then the original value from func1 will be
     * returned in its stead - so if you wish to replace a defined value with
     * a nullish one, return <code>null</code> from func2.
     * 
     * 
     * @memberOf niagara.util.aop
     * @param {Object} obj an object containing the method to apply the aspect
     * to. If this object is a prototype then every object created with the
     * prototype will receive this new behavior.
     * 
     * @param {String} methodName the name of the method to apply the aspect to
     * 
     * @param {Function} afterFunction the function to execute after the
     * target function. <code>afterFunction</code> can accept two parameters:
     * the first consisting of the arguments array passed to the target
     * function, and the second consisting of the target function's return
     * value. The value returned by <code>afterFunction</code> will replace
     * the one returned by the target function.
     */
    function after(obj, methodName, afterFunction) {
      return doWrapFunction(obj, methodName, afterFunction, function (newFunc, oldFunc) {
        return function () {
          var args = Array.prototype.slice.call(arguments),
              value = oldFunc.apply(this, args),
              newResult = newFunc.call(this, args, value);
          if (newResult === undefined) {
            return value;
          } else {
            return newResult;
          }
        };
      });
    }
    
//    function benchmark(obj) {
//      baja.strictArg(obj, Object);
//      baja.iterate(obj, function (value, propName) {
//        if (typeof value === 'function' && !propName.charAt(0).match(/[A-Z]/)) {
//          obj[propName] = function () {
//            var args = Array.prototype.slice.call(arguments),
//                date = new Date(),
//                result = value.apply(this, args);
//            console.log(propName + ' took ' + (new Date().getTime() - date.getTime()));
//            return result;
//          };
//        }
//      });
//    }
    
    /**
     * Converts a function into a callback passed to another function.
     * Essentially "wraps" one function in another.
     * <p>
     * This function may also be called with two parameters: <code>func1</code>
     * and <code>func2</code>. <code>func1</code> may take an arbitrary set
     * of parameters; <code>func2</code> must take exactly one parameter: a
     * callback function. The return value of <code>toCallback</code> will then
     * be a new function that takes the same set of parameters
     * <code>func1</code> did, but the <code>func1</code> call will be
     * converted into a parameterless function passed into <code>func2</code>.
     * It is then up to <code>func2</code> to decide whether or not to call
     * this callback.
     * <p>
     * This function is useful for converting a synchronous function into an
     * asynchronous one; for example, <code>func2</code> could show an OK/Cancel
     * dialog and only call its callback if the user clicks OK - essentially
     * giving the user an override option to cancel the <code>func1</code> call
     * if he/she chooses. Or <code>func2</code> could make an AJAX request and
     * call <code>func1</code> upon completion. Note that if <code>func1</code>
     * expects a value to be synchronously returned, said return value will be
     * lost.
     * 
     * @memberOf niagara.util.aop
     * 
     * @param {Object} obj an object containing the method to apply the aspect
     * to. If this object is a prototype then every object created with the
     * prototype will receive this new behavior.
     * 
     * @param {String} methodName the name of the method to apply the aspect to
     * 
     * @param {Function} callingFunction the function that will call the
     * target function as a callback. <code>callingFunction</code> must accept
     * exactly one parameter: a callback function. It is up to
     * <code>callingFunction</code> to decide when and whether to call this
     * callback.
     */
    function beforeAsync(obj, methodName, callingFunction) {
      return doWrapFunction(obj, methodName, callingFunction, function (newFunc, oldFunc) {
        return function () {
          var args = Array.prototype.slice.call(arguments);
          
          return newFunc.call(this, args, bind(function (newArgs) {
            return oldFunc.apply(this, newArgs || args);
          }, this));
        };
      });
    }

    
    /**
     * Convenience method allowing you to add before/after AOP advice to
     * multiple functions on an object in one statement.
     * 
     * @memberOf niagara.util.aop
     * 
     * @param {Object} obj an object requiring behavior advice
     * @param {Object} advice an object containing functions to applied to
     * <code>obj</code> using <code>before()</code> and <code>after</code>
     * @param {Object} [advice.before] a mapping from method name to
     * <code>before()</code> advice functions
     * @param {Object} [advice.after] a mapping from method name to
     * <code>after()</code> advice functions
     * @param {Object} [advice.toCallback] a mapping from method name to
     * <code>toCallback()</code> advice functions

     * 
     * @returns {Object} the newly advised object
     */
    function advise(obj, advice) {
      baja.strictArg(obj, Object);
      
      advice = baja.objectify(advice);
      
      var befores = advice.before || {},
          afters = advice.after || {},
          callbacks = advice.beforeAsync || {},
          beforeCallbacks = advice.beforeCallback || {};
      iterate(befores, function (func, funcName) {
        before(obj, funcName, func);
      });
      iterate(afters, function (func, funcName) {
        after(obj, funcName, func);
      });
      iterate(callbacks, function (func, funcName) {
        beforeAsync(obj, funcName, func);
      });
      
      return obj;
    }
    
    /**
     * Performs <code>advise</code> on the given function's prototype. Very
     * useful for allowing one constructor to extend another.
     * 
     * @memberOf niagara.util.aop
     * 
     * @param {Function} func a function whose prototype to advise 
     * @param {Object} advice an object containing AOP before/after functions
     * (see <code>advise()</code>)
     * @returns {Function} the newly advised function
     */
    function advisePrototype(func, advice) {
      baja.strictArg(func, Function);
      advise(func.prototype, advice);
      return func;
    }



    /** 
     * @namespace
     * @name niagara.util.aop
     */
    api('niagara.util.aop', {
      advise: advise,
      advisePrototype: advisePrototype,
      after: after,
      before: before,
      beforeAsync: beforeAsync
//      benchmark: benchmark
    });
  }());
  

  (function ordFunctions() {
    /**
     * If the given ord ends with a slash, remove it - unless the ord ends with
     * a colon (e.g. 'station:|slot:') which can cause some problems with 
     * ord resolution. In this case, ADD a slash.
     * 
     * @private
     * @inner
     * @memberOf niagara.util.ord
     * @param {String|baja.Ord} ord
     * @returns {String} the ord with its trailing '/' removed, or a trailing
     * ':' changed to ':/'
     */
    function chopLastSlash(ord) {
      if (!ord) {
        return "";
      }
      
      ord = String(ord);
      var len = ord.length,
          lastChar = ord.substring(len - 1, len),
          nextToLastChar = ord.substring(len - 2, len - 1);

      if (lastChar === '/' && nextToLastChar !== ':') {
        ord = ord.substring(0, len - 1);
      } else if (lastChar === ':') {
        ord = ord + '/';
      }
      
      return ord;
    }
    
    /**
     * Convenience method for retrieving a component from a given ORD. 
     * 
     * @memberOf niagara.util.ord
     * @param {baja.Ord|String} ord the ord to resolve
     * @param {Object} callbacks an object containing ok/fail callbacks
     * @param {Function} callbacks.ok a function that will handle the retrieved
     * component
     * @param {Boolean} [lease] whether or not the retrieved component should
     * be leased (defaults to true).
     */
    function get(ord, callbacks, lease) {
      ord = baja.Ord.make(ord);

      if (lease === undefined) {
        lease = true;
      }
      
      callbacks = baja.objectify(callbacks, 'ok');
      
      ord.get({
        ok: callbacks.ok,
        fail: callbacks.fail || baja.error,
        lease: lease
      });
    }
    
    /**
     * Converts an ORD into an array of all its sub-ORDs, beginning at the root. 
     * For example, the ORD <code>'a/b/c'</code> would convert to 
     * <code>['a', 'a/b', 'a/b/c']</code>.
     * 
     * @memberOf niagara.util.ord
     * @param ord {String|baja.Ord} ord the ORD to convert to sub-ORDs
     * @returns {Array} an array of all the given ORD's sub-ORDs
     */
    function makeOrdHistory(ord) {
      ord = String(ord);
      
      //ensure there is no trailing slash as this will add one too many 
      //matches to the split below. chopLastSlash will manage whether to
      //include one or not
      if (ord.match(/\/$/)) {
        ord = ord.substring(0, ord.length - 1);
      }
      
      var split = ord.split('/'),
          i = 0,
          history = [];
      for (i = 0; i < split.length; i++) {
        history.push(chopLastSlash(split.slice(0, i + 1).join('/')));
      }
      return history;
    }
    
    /**
     * Checks two ORDs for equivalence. ORDs are considered equivalent if 
     * their string representations, omitting any trailing slashes, are equal.
     * 
     * @memberOf niagara.util.ord
     * @param {baja.Ord|String} ord1
     * @param {baja.Ord|String} ord2
     */
    function equivalent(ord1, ord2) {
      if (!ord1 || !ord2) {
        return false;
      }
      
      return baja.Ord.make(ord1).relativizeToSession().equals(baja.Ord.make(ord2).relativizeToSession());
    }
    
    /**
     * Calculates the ORD for the given <code>Complex</code> object. If the
     * object is a <code>baja.Component</code>, <code>getNavOrd()</code> will
     * be returned. If the object is otherwise a <code>baja.Complex</code>
     * (e.g. in case of a <code>Struct</code>), the ORD will be constructed
     * by appending property names from the nearest <code>baja.Component</code>
     * parent.
     * 
     * @memberOf niagara.util.ord
     * @param {baja.Complex} complex the object whose ORD we wish to know
     * @returns {String} a String representation of the ORD
     */
    function deriveOrd(complex) {
      if (complex) {
        if (complex instanceof baja.Ord || typeof complex === 'string') {
          return complex.toString();
        }
        
        if (complex.getType().isComponent() && complex.isMounted()) {
          return complex.getNavOrd().relativizeToSession().toString(); 
        }
        
        if (complex.getType().isComplex()) {
          var parentOrd = deriveOrd(complex.getParent());
          if (parentOrd) {
            return parentOrd + '/' + complex.getPropertyInParent(); 
          }
        }
      }
      return undefined;
    }
    

    /**
     * @namespace
     * @name niagara.util.ord
     */
    api('niagara.util.ord', {
      deriveOrd: deriveOrd,
      equivalent: equivalent,
      get: get,
      makeOrdHistory: makeOrdHistory
    });
  }());
  
  (function slotFunctions() {
    
    function arrayShift(array, srcIndex, destIndex) {
      if (srcIndex === destIndex ||
          srcIndex < 0 || destIndex < 0 ||
          srcIndex >= array.length || destIndex >= array.length) {
        return false;
      }
      
      var removed = array.splice(srcIndex, 1);
      array.splice(destIndex, 0, removed[0]);
      return true;
    }
    
    function doMoveSlot(component, prop, obj, doSwap) {
      var slotArray = component.getSlots().dynamic().toArray(),
          index = indexOf(slotArray, component.getSlot(prop));
      if (doSwap(slotArray, index)) {
        component.reorder($.extend(obj, {
          dynamicProperties: slotArray
        }));
      }
    }
    
    /**
     * Reorders the component's dynamic slots so that the given property
     * moves up one in the list.
     * 
     * @memberOf niagara.util.slot
     * @param {baja.Component} component the component whose slots to reorder
     * @param {baja.Slot} slot the slot to move 
     * @param {Object} [obj] an object containing callbacks etc (see
     * <code>baja.Component.reorder()</code>
     */
    function moveUp(component, slot, obj) {
      doMoveSlot(component, slot, obj, function (array, index) {
        return arrayShift(array, index, index - 1);
      });
    }
    
    /**
     * Reorders the component's dynamic slots so that the given property
     * moves down one in the list.
     * 
     * @memberOf niagara.util.slot
     * @param {baja.Component} component the component whose slots to reorder
     * @param {baja.Slot} slot the slot to move 
     * @param {Object} [obj] an object containing callbacks etc (see
     * <code>baja.Component.reorder()</code>
     */
    function moveDown(component, slot, obj) {
      doMoveSlot(component, slot, obj, function (array, index) {
        return arrayShift(array, index, index + 1);
      });
    }
    
    /**
     * Reorders the component's dynamic slots so that the given property
     * moves to the top of the list.
     * 
     * @memberOf niagara.util.slot
     * @param {baja.Component} component the component whose slots to reorder
     * @param {baja.Slot} slot the slot to move 
     * @param {Object} [obj] an object containing callbacks etc (see
     * <code>baja.Component.reorder()</code>
     */
    function moveToTop(component, slot, obj) {
      doMoveSlot(component, slot, obj, function (array, index) {
        return arrayShift(array, index, 0);
      });
    }
    
    /**
     * Reorders the component's dynamic slots so that the given property
     * moves to the bottom of the list.
     * 
     * @memberOf niagara.util.slot
     * @param {baja.Component} component the component whose slots to reorder
     * @param {baja.Slot} slot the slot to move 
     * @param {Object} [obj] an object containing callbacks etc (see
     * <code>baja.Component.reorder()</code>
     */
    function moveToBottom(component, slot, obj) {
      doMoveSlot(component, slot, obj, function (array, index) {
        return arrayShift(array, index, array.length - 1);
      });
    }
    
    /**
     * Check to see if we need to pop up a confirmation dialog before firing
     * this action.
     * 
     * @memberOf niagara.util.slot
     * 
     * @param {baja.Action} action the action we want to fire
     * @return true if we require confirmation before firing this action
     */
    function isConfirmRequired(action) {
      var flags = action.getFlags();
      if (baja.Flags.CONFIRM_REQUIRED & flags) {
        return true;
      }
      
      return false;
    }
    
    /**
     * Check to see if a slot has the READONLY flag set.
     * 
     * @memberOf niagara.util.slot
     * 
     * @param {baja.Slot} slot
     * @return true if the READONLY flag is set
     */
    function isReadonly(slot) {
      if (slot.getType().is('baja:Password')) {
        return true;
      }
      
      return baja.Flags.READONLY & slot.getFlags();
    }
    
    /**
     * Check to see if this slot should be hidden from display on the property
     * sheet.
     * 
     * @memberOf niagara.util.slot
     * 
     * @param {baja.Slot} slot property to check for hidden status
     * @returns {Boolean} true if property should be hidden
     */
    function isHidden(slot) {
      var flags = slot.getFlags();
      
      if (baja.Flags.HIDDEN & flags) {
        return true;
      } else if (slot.isProperty() && slot.getType().is('baja:Link')) {
        return true;
      } else if (slot.getName() === 'wsAnnotation' || 
                 slot.getName() === 'displayNames') {
        return true;
      }
      
      return false;
    }
    
    /**
     * Check to see if clicking this property in the property sheet should link
     * to a field editor. To be editable, a property must not have the READONLY
     * flag set and it must have a field editor registered for its type.
     * 
     * @memberOf niagara.util.slot
     * 
     * @param {baja.Property} prop property to check for editability
     * @returns {Boolean} true if we should link to a field editor
     */
    function isEditable(prop) {
      var fe = niagara.fieldEditors;

      if (prop.getType().is('baja:Password')) {
        return false;
      } else if (isReadonly(prop)) {
        return false;
      } else if (isHidden(prop)) {
        return false;
      } else if (fe && fe.isRegistered(prop)) {
        return true;
      }
      
      return false;
    }
    
    /**
     * Check to see if an action can be fired (if the slot is an action, and
     * the slot is not hidden).
     * 
     * @memberOf niagara.util.slot
     * @param {baja.Slot} slot
     * @returns {Boolean} if the slot is an action that can be fired
     */
    function isFireable(slot) {
      return slot.isAction() && !isHidden(slot);
    }
    
    /**
     * Attempt to pull Facets off of a Complex. First attempt to pull facets
     * directly from the Complex using <code>baja.Complex#getFacets</code>.
     * If the Complex has a frozen slot of type Facets, pull those off and
     * merge those in - in the case of duplicate Facets, facets from the
     * frozen slot will win out.
     * 
     * @memberOf niagara.util.slot
     * @private
     * @param {baja.Complex} complex
     * @return {baja.Facets}
     */
    function getComplexFacets(complex) {
      var facets = baja.Facets.DEFAULT,
          complexFacets,
          frozenSlotFacets;
      if (complex && complex.getType().isComplex()) {
        frozenSlotFacets = complex.getSlots()
          .properties()
          .frozen()
          .is('baja:Facets')
          .firstValue();
        
        if (typeof complex.getFacets === 'function') {
          complexFacets = complex.getFacets();
          if (complexFacets) {
            facets = baja.Facets.make(facets, complexFacets);
          }
        }
        
        if (frozenSlotFacets) {
          facets = baja.Facets.make(facets, frozenSlotFacets);
        }
      }
      
      return facets;
    }
    
    /**
     * Finds the nearest ancestor to the given Complex that is of type
     * Component (this may be the input Complex itself).
     * 
     * @memberOf niagara.util.slot
     * @private
     * @param {baja.Complex} comp
     */
    function getComponentAncestor(comp) {
      var type = comp && comp.getType();
      if (type) {
        if (type.isComponent()) {
          return comp;
        } else if (type.isComplex()) {
          return getComponentAncestor(comp.getParent());
        }
      } else {
        return undefined;
      }
    }
    
    /**
     * Attempts to retrieve a usable Facets object from a particular Baja
     * value. Station-side, there is a significant amount of specialized
     * facets-retrieving logic (often in the form of overrides of
     * <code>getSlotFacets()</code> - since this code isn't available to
     * Bajascript, we'll have to replicate the most relevant bits as best we
     * can in a way that makes sense for our mobile apps.
     * 
     * @memberOf niagara.util.slot
     * @param {baja.Value} value a value to examine for facet information
     * @param {baja.Slot|String} [slot] a slot to examine for facet information
     * @returns {baja.Facets} any facet information that could be retrieved
     * from the given value and/or slot - or <code>baja.Facets.DEFAULT<code> if
     * no facets were found
     */
    function getFacets(value, slot) {
      var facets = baja.Facets.DEFAULT,
          valueFacets,
          type = value && value.getType(),
          ancestor,
          ancestorFacets,
          ancestorType,
          slotName,
          slotFacets;
      
      // did we pass in a Slot as well as a value? If so, see if this value
      // is a Complex and the slot references a Complex as well - if so, grab
      // the Complex property referenced by the slot and examine that one.
      if (slot && value.getType().isComplex()) {
        slot = value.getSlot(slot);
        if (slot.isProperty() && slot.getType().isComplex()) {
          value = value.get(slot);
        }
      }

      
      if (type) {
        if (type.isComplex()) {
          // walk up the component tree until we find a Component
          ancestor = getComponentAncestor(value);
          if (ancestor) {
            ancestorType = ancestor.getType();
            ancestorFacets = getComplexFacets(ancestor);
            //merge our Component facets in with our facets so far
            if (ancestorFacets) {
              facets = baja.Facets.make(facets, ancestorFacets);
            }
          }
        }
      }
      
      // now see if the slot itself has facets configured
      if (slot instanceof baja.Slot) {
        slotName = String(slot);
        if (ancestorType && ancestorType.is('control:IWritablePoint')) {
          if ((slot.isFrozen() && slotName.indexOf('in') === 0) ||
              slotName === 'set' ||
              slotName === 'fallback' ||
              slotName === 'override' ||
              slotName === 'emergencyOverride') {
            //these action slots cannot have their own facets configured -
            //we will use the contents of the 'facets' slot instead, which
            //comes from the calls to getComplexFacets above.
            //see BEnumWritable#getSlotFacets()
            return facets;
          }
        }
        // merge slot facets in with our facets so far
        facets = baja.Facets.make(facets, slot.getFacets());
      }
      
      return facets;
    }
    

    
    /**
     * @namespace
     * @name niagara.util.slot
     */
    api('niagara.util.slot', {
      getFacets: getFacets,
      isConfirmRequired: isConfirmRequired,
      isEditable: isEditable,
      isFireable: isFireable,
      isHidden: isHidden,
      isReadonly: isReadonly,
      moveDown: moveDown,
      moveUp: moveUp,
      moveToTop: moveToTop,
      moveToBottom: moveToBottom
    });
  }());
  
  (function mobileFunctions() {
    var NON_HTML_ID_CHARS = /[^\w-_]/g,
        entityMap = {
          "&": "&amp;",
          "<": "&lt;",
          ">": "&gt;",
          '"': '&quot;',
          "'": '&#39;',
          "/": '&#x2F;'
        };
    
    /**
     * Safely escapes non-HTML characters ( &amp; &lt; &gt; &quot; &#39; &#x2F; )
     * in the given string.
     * @memberOf niagara.util.mobile
     * @param {String} string
     * @returns {String}
     */
    function escapeHtml(string) {
      return String(string).replace(/[&<>"'\/]/g, function (s) {
        return entityMap[s];
      });
    }
    
    /**
     * Determines the actual visible height of the content of the current
     * JQM page - minus header and footer height.
     * 
     * @memberOf niagara.util.mobile
     * 
     * @param {jQuery} page the JQM page whose visible content height you want
     * @returns {Number} the height of the visible portion of the
     * given JQM page, in pixels
     */
    function getVisibleHeight(page) {
      var header = page.children(':jqmData(role="header")'),
          footer = page.children(':jqmData(role="footer")'),
          //http://bugs.jquery.com/ticket/6724
          height = window.innerHeight || $(window).height();
      
      if (header.length) {
        height -= header.outerHeight();
      }
      
      if (footer.length) {
        height -= footer.outerHeight();
      }

      return height;
    }
    
    /**
     * Explicitly sets the height of the current JQM page's content div to
     * maximize visible space.
     * 
     * @memberOf niagara.util.mobile
     * @param {jQuery} page the JQM page whose content div you want to maximize
     * @param {jQuery} [divToSet] you can specify a particular div whose
     * height to set - if omitted, defaults to 
     * <code>page.children('jqmData(role=content)')</code>
     */
    function setContentHeight(page, divToSet) {
      var div = divToSet || page.children(':jqmData(role="content")'),
          visibleHeight = getVisibleHeight(page),
          offset = div.outerHeight() - div.height();
      div.height(visibleHeight - offset);
      div.parent().height(Math.max(div.parent().height(), visibleHeight));
      setTimeout(function () {
        div.trigger('updatelayout');
      }, 100);
    }
    
    /**
     * Adds an event handler to a DOM element, but ensures that the handler
     * winds up <i>first</i> in the handler list. This way, if your handler
     * decides to return false or cancel event propagation, the handlers
     * already registered on that element will not fire.
     * 
     * @memberOf niagara.util.mobile
     * @param {jQuery} dom the element on which to bind
     * @param {String} eventName
     * @param {Function} handler
     */
    function prependEventHandler(dom, eventName, handler) {
      dom.bind(eventName, handler);

      var data = $._data(dom[0]),
          events = data.events,
          handlers = events[eventName];
      
      if (handlers.length > 1) {
        handlers.splice(0, 0, handlers.pop());
      }
    }
    
    /**
     * @class
     * @memberOf niagara.util.mobile
     * @private
     */
    function PageLoadingTicket(delay, timeout, timeoutFunc) {
      var that = this;
      
      that.$ticket = setTimeout(function () {
        if (that.$ticket) {
          $.mobile.showPageLoadingMsg();
        }
      }, delay);
      
      if (timeout) {
        that.$timeoutTicket = setTimeout(function () {
          if (that.$ticket) {
            var msg = 'timeout ' + timeout + 'ms reached';
            if (typeof timeoutFunc === 'function') {
              timeoutFunc(msg);
            } else {
              baja.error(msg);
            }
          }
          that.hide();
        }, timeout);
      }
    }
    
    PageLoadingTicket.prototype.hide = function () {
      $.mobile.hidePageLoadingMsg();
      
      var that = this;
      clearTimeout(that.$ticket);
      delete that.$ticket;
      clearTimeout(that.$timeoutTicket);
      delete that.$timeoutTicket;
    };
    
    /**
     * Returns a ticket object that will show a page loading message after a
     * specified delay. Retrieve this ticket object before performing some
     * asynchronous action, then call the ticket's <code>hide</code> method
     * after the action returns - this way, if the async action takes less than
     * the specified time, the loading message won't be shown at all.
     * 
     * <p>You can also specify a maximum amount of time to spin before some
     * error action is taken.
     * 
     * <pre>
     * var ticket = spinnerTicket(1000, 5000, function () {
     *   baja.error("spun for 5 seconds - timing out");
     * });
     * performSomeAsynchronousAction({
     *   ok: function () {
     *     //if the async action takes less than 1 seconds, no loading message
     *     //will show
     *     ticket.hide();
     *   }
     * });
     * </pre>
     * @memberOf niagara.util.mobile
     * @param {Number} delay amount of time (in ms) to wait before showing
     * the loading message
     * @param {Number} [timeout] maximum amount of time the loading message
     * may spin
     * @param {Function} [timeoutFunc] a function to execute if the timeout
     * is reached before <code>hide()</code> is called - an error message
     * describing the timeout will be passed as the first parameter
     * 
     * @returns {niagara.util.mobile.PageLoadingTicket} call <code>hide()</code>
     * on this to hide the page loading message (or prevent it from showing if
     * it is not already shown)
     */
    function spinnerTicket(delay, timeout, timeoutFunc) {
      return new PageLoadingTicket(delay || 1000, timeout, timeoutFunc);
    }
    
    function setListviewInset(ul, inset) {
      ul.toggleClass("ui-listview-inset ui-corner-all ui-shadow", inset);
      
      var lis = ul.children('li'),
          first = lis.first(),
          last = lis.last();

      first.toggleClass('ui-corner-top', inset);
      last.toggleClass('ui-corner-bottom', inset);

      //for split button list, the split link on far right needs rounding too
      first.children('a').last().toggleClass('ui-corner-tr', inset);
      last.children('a').last().toggleClass('ui-corner-br', inset);
    }
    
    /**
     * Converts a JQM listview to inset form (adds margins and rounded 
     * corners).
     * 
     * @memberOf niagara.util.mobile
     * @param {jQuery} ul a JQM <code>ul:jqmData(role=listview)</code> listview
     * element
     */
    function applyListviewInset(ul) {
      setListviewInset(ul, true);
    }
    
    /**
     * Converts a JQM listview to non-inset form (removes margins and rounded 
     * corners).
     * 
     * @memberOf niagara.util.mobile
     * @param {jQuery} ul a JQM <code>ul:jqmData(role=listview)</code> listview
     * element
     */
    function removeListviewInset(ul) {
      setListviewInset(ul, false);
    }
    
    /**
     * Don't allow a navbar button to retain its JQM highlighting / hover status
     * after it is clicked. 
     * 
     * @memberOf niagara.util.mobile
     * @param {jQuery} navbar a navbar div (<code>:jqmData(role=navbar)</code>)
     */
    function preventNavbarHighlight(navbar) {
      navbar.delegate('a', 'click', function () {
        var $this = $(this);
        setTimeout(function () {
          $this.removeClass(function (index, css) {
            var classesToRemove = 'ui-btn-active ' + 
              (css.match(/\bui-btn-hover-\w/g) || []).join(' ');
            return classesToRemove;
          });
        }, 100);
      });
    }
        
    /**
     * Creates an appropriate HTML ID for a property sheet's containing div.
     * All non-alphanumeric characters (except an actual underscore, or a
     * hyphen) will be escaped by an underscore followed by the character's 
     * ASCII code, e.g. <code>|</code> becomes <code>_7C</code>. If the input
     * page id begins with <code>#</code> (i.e. in jQuery selector form), the
     * beginning <code>#</code> will be stripped from the returned page ID
     * (it will be not be encoded as part of it!)
     * 
     * @memberOf niagara.util.mobile
     * 
     * @param {baja.Ord|String} ord the ORD to convert to a page ID
     * @returns {String} a valid HTML5 ID (not in selector form, i.e. does
     * not start with <code>#</code>)
     */
    function encodePageId(slotPath) {
      slotPath = String(slotPath);
      if (slotPath.charAt(0) === '#') {
        slotPath = slotPath.substring(1);
      }
      
      return slotPath.replace(NON_HTML_ID_CHARS, function (s) {
        return '_' + s.charCodeAt(0).toString(16).toUpperCase();
      });
    }
    
    /**
     * Converts a page ID (the output of <code>encodePageId</code> back into
     * a regular string.
     * 
     * @memberOf niagara.util.mobile
     * @param {String|jQuery} str the page id to decode (or a jQuery object,
     * in which case attr('id') will be used)
     * @returns {String} the decoded string
     */
    function decodePageId(str) {
      if (!str) {
        return str;
      }
      
      if (str instanceof $) {
        str = str.attr('id');
      } else {
        str = String(str);
      }
    
      var re = /\_([A-Z0-9][A-Z0-9])/g;

      str = str.replace(re, function (s) {
        return String.fromCharCode(parseInt(s.substring(1, s.length), 16));
      });
      return str;      
    }
    
    /**
     * Converts an <code>options</code> input to one of the other 
     * <code>linkToOrd</code> function into an object with a 
     * <code>viewQuery</code> property. This enables the input of a number
     * of different types to <code>linkToOrd</code> depending on what you have
     * handy:
     * 
     * <pre>
     *   var ord = 'station:|slot:',
     *       id = 'workbench:PropertySheet',
     *       params = { foo: 'bar' };
     *   
     *   linkToOrd(ord); //no view query
     *   linkToOrd(ord, id); //new ViewQuery using the id
     *   linkToOrd(ord, { id: id, params: params } ); //new ViewQuery using id and params
     *   linkToOrd(ord, new baja.ViewQuery(id)); //uses the ViewQuery as is
     * </pre>
     * 
     * <p>This is a private function, <code>linkToOrd</code> will call it
     * directly.
     * 
     * @memberOf niagara.util.mobile
     * @private
     * @param options
     * @returns {Object} an object literal with a <code>viewQuery</code>
     * property to be used in a <code>linkToOrd</code> method
     * (<code>viewQuery</code> will be undefined if not provided)
     */
    function viewQueryify(options) {
      var viewQuery = options && options.viewQuery;
      if (!viewQuery) {
        options = baja.objectify(options, 'viewQuery');
        viewQuery = options.viewQuery;
      }
      
      if (typeof viewQuery === 'string' || (viewQuery && viewQuery.id)) {
        viewQuery = new baja.ViewQuery(viewQuery);
      } else if (!viewQuery && options.id) {
        viewQuery = new baja.ViewQuery(options);
      }
      
      options.viewQuery = viewQuery;
      return options;
    }
    
    /**
     * Performs a link to a new ORD, triggering a page refresh. Used when linking
     * to an ORD to be viewed in a different app. Simply calls
     * <code>window.location.assign</code>.
     * 
     * @memberOf niagara.util.mobile
     * @see niagara.util.mobile.viewQueryify
     * @param {String|baja.Ord} ord the ORD to link to
     * @param {Object} [options] additional options used in the page transition
     * @param {baja.ViewQuery} [options.viewQuery] can specify any view parameters to be
     * encoded into the ORD
     */
    function linkToOrdExternal(ord, options) {
      options = viewQueryify(options);
      
      var o;
      if (options.viewQuery) {
        o = baja.Ord.make({
          base: ord,
          child: options.viewQuery
        });
      }
      else {
        o = baja.Ord.make(ord);
      }
      
      window.location.assign(o.toUri());
    }
    
    /**
     * Performs a link to a new ORD without causing a page refresh to a new app.
     * Used for linking between <code>PageView</code>s within the same app
     * by listening to changes to the ORD in the browser's current URL.
     * 
     * @memberOf niagara.util.mobile
     * @see niagara.util.mobile.viewQueryify
     * @param {String|baja.Ord} ord the ORD to link to
     * @param {Object} [options] additional options used in the page transition -
     * will be passed directly into <code>$.mobile.changePage()</code>
     * @param {baja.ViewQuery} [options.viewQuery] can specify any view parameters to be
     * encoded into the ORD
     */
    function linkToOrdInternal(ord, options) {
      options = viewQueryify(options);
          
      var o;
      if (options.viewQuery) {
        o = baja.Ord.make({
          base: ord,
          child: options.viewQuery
        });
      }
      else {
        o = baja.Ord.make(ord);
      }
      
      $.mobile.changePage(o.toUri(), options);
    }
    
    /**
     * Queries the server to ascertain that the view for the requested ORD is the
     * same as the current view (that is, the requested ORD should be viewed using
     * the same app we are currently in). If it is, it triggers an internal link
     * ($.mobile.changePage). Otherwise, it instructs 
     * the browser to do a full page reload to the requested app.
     * 
     * @memberOf niagara.util.mobile
     * @see niagara.util.mobile.viewQueryify
     * @param {String|baja.Ord} ord the ORD to link to
     * @param {Object} [options] additional options used in the page transition -
     * will be passed directly into <code>$.mobile.changePage()</code>
     * @param {baja.ViewQuery} [options.viewQuery] can specify any view parameters to be
     * encoded into the ORD
     */
    function linkToOrd(ord, options) {
      var ticket = spinnerTicket(1000);
      
      $.ajax({
        url: baja.Ord.make(ord).toUri(),
        type: "POST",
        headers: {
          "niagara-mobile-rpc": "typeSpec"
        },
        success: function (data) {
          ticket.hide();
          if (data.typeSpec === niagara.view.typeSpec) {
            linkToOrdInternal(ord, options);
          } else {
            linkToOrdExternal(ord, options);
          }
        },
        error: function () {
          ticket.hide();
          linkToOrdExternal(ord, options);
        }
      });
    }
    
    
    /**
     * @namespace
     * @name niagara.util.mobile
     */
    api('niagara.util.mobile', {
      applyListviewInset: applyListviewInset,
      decodePageId: decodePageId,
      encodePageId: encodePageId,
      escapeHtml: escapeHtml,
      getVisibleHeight: getVisibleHeight,
      linkToOrd: linkToOrd,
      linkToOrdExternal: linkToOrdExternal,
      linkToOrdInternal: linkToOrdInternal,
      prependEventHandler: prependEventHandler,
      preventNavbarHighlight: preventNavbarHighlight,
      removeListviewInset: removeListviewInset,
      setContentHeight: setContentHeight,
      spinnerTicket: spinnerTicket
    });
  }());
}());
