/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Color utilities.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true */
/*globals niagara, $ */

(function statusGradients() {
  'use strict';
  
  var AARRGGBB_REGEX = /^\#([0-9a-fA-F]{2})([0-9a-fA-F]{2})([0-9a-fA-F]{2})([0-9a-fA-F]{2})?$/,
      RGB_REGEX = /^\#([0-9a-fA-F])([0-9a-fA-F])([0-9a-fA-F])$/,
      CSS_RGB_REGEX = /^rgb\([\s]*?([0-9]+[%]?)[\s]*?,[\s]*?([0-9]+[%]?)[\s]*?,[\s]*?([0-9]+[%]?)[\s]*?\)$/,
      CSS_RGBA_REGEX = /^rgba\([\s]*?([0-9]+[%]?)[\s]*?,[\s]*?([0-9]+[%]?)[\s]*?,[\s]*?([0-9]+[%]?)[\s]*?,[\s]*?([0-9\.]+)[\s]*?\)$/,
      
      /**
       * Mapping from BColor names to CSS/hex equivalents.
       * @name niagara.util.color.COLOR_CONSTANTS
       * @field
       */
      COLOR_CONSTANTS = {
        transparent: "rgba(0, 0, 0, 0)",
        aliceBlue: "#f0f8ff",
        antiqueWhite: "#faebd7",
        aqua: "#00ffff",
        aquamarine: "#7fffd4",
        azure: "#f0ffff",
        beige: "#f5f5dc",
        bisque: "#ffe4c4",
        black: "#000000",
        blanchedAlmond: "#ffebcd",
        blue: "#0000ff",
        blueViolet: "#8a2be2",
        brown: "#a52a2a",
        burlyWood: "#deb887",
        cadetBlue: "#5f9ea0",
        chartreuse: "#7fff00",
        chocolate: "#d2691e",
        coral: "#ff7f50",
        cornflowerBlue: "#6495ed",
        cornsilk: "#fff8dc",
        crimson: "#dc143c",
        cyan: "#00ffff",
        darkBlue: "#00008b",
        darkCyan: "#008b8b",
        darkGoldenrod: "#b8860b",
        darkGray: "#a9a9a9",
        darkGreen: "#006400",
        darkGrey: "#a9a9a9",
        darkKhaki: "#bdb76b",
        darkMagenta: "#8b008b",
        darkOliveGreen: "#556b2f",
        darkOrange: "#ff8c00",
        darkOrchid: "#9932cc",
        darkRed: "#8b0000",
        darkSalmon: "#e9967a",
        darkSeaGreen: "#8fbc8f",
        darkSlateBlue: "#483d8b",
        darkSlateGray: "#2f4f4f",
        darkSlateGrey: "#2f4f4f",
        darkTurquoise: "#00ced1",
        darkViolet: "#9400d3",
        deepPink: "#ff1493",
        deepSkyBlue: "#00bfff",
        dimGray: "#696969",
        dimGrey: "#696969",
        dodgerBlue: "#1e90ff",
        firebrick: "#b22222",
        floralWhite: "#fffaf0",
        forestGreen: "#228b22",
        fuchsia: "#ff00ff",
        gainsboro: "#dcdcdc",
        ghostWhite: "#f8f8ff",
        gold: "#ffd700",
        goldenrod: "#daa520",
        gray: "#808080",
        green: "#008000",
        greenYellow: "#adff2f",
        grey: "#808080",
        honeydew: "#f0fff0",
        hotPink: "#ff69b4",
        indianRed: "#cd5c5c",
        indigo: "#4b0082",
        ivory: "#fffff0",
        khaki: "#f0e68c",
        lavender: "#e6e6fa",
        lavenderBlush: "#fff0f5",
        lawnGreen: "#7cfc00",
        lemonChiffon: "#fffacd",
        lightBlue: "#add8e6",
        lightCoral: "#f08080",
        lightCyan: "#e0ffff",
        lightGoldenrodYellow: "#fafad2",
        lightGray: "#d3d3d3",
        lightGreen: "#90ee90",
        lightGrey: "#d3d3d3",
        lightPink: "#ffb6c1",
        lightSalmon: "#ffa07a",
        lightSeaGreen: "#20b2aa",
        lightSkyBlue: "#87cefa",
        lightSlateGray: "#778899",
        lightSlateGrey: "#778899",
        lightSteelBlue: "#b0c4de",
        lightYellow: "#ffffe0",
        lime: "#00ff00",
        limeGreen: "#32cd32",
        linen: "#faf0e6",
        magenta: "#ff00ff",
        maroon: "#800000",
        mediumAquamarine: "#66cdaa",
        mediumBlue: "#0000cd",
        mediumOrchid: "#ba55d3",
        mediumPurple: "#9370db",
        mediumSeaGreen: "#3cb371",
        mediumSlateBlue: "#7b68ee",
        mediumSpringGreen: "#00fa9a",
        mediumTurquoise: "#48d1cc",
        mediumVioletRed: "#c71585",
        midnightBlue: "#191970",
        mintCream: "#f5fffa",
        mistyRose: "#ffe4e1",
        moccasin: "#ffe4b5",
        navajoWhite: "#ffdead",
        navy: "#000080",
        oldLace: "#fdf5e6",
        olive: "#808000",
        oliveDrab: "#6b8e23",
        orange: "#ffa500",
        orangeRed: "#ff4500",
        orchid: "#da70d6",
        paleGoldenrod: "#eee8aa",
        paleGreen: "#98fb98",
        paleTurquoise: "#afeeee",
        paleVioletRed: "#db7093",
        papayaWhip: "#ffefd5",
        peachPuff: "#ffdab9",
        peru: "#cd853f",
        pink: "#ffc0cb",
        plum: "#dda0dd",
        powderBlue: "#b0e0e6",
        purple: "#800080",
        red: "#ff0000",
        rosyBrown: "#bc8f8f",
        royalBlue: "#4169e1",
        saddleBrown: "#8b4513",
        salmon: "#fa8072",
        sandyBrown: "#f4a460",
        seaGreen: "#2e8b57",
        seaShell: "#fff5ee",
        sienna: "#a0522d",
        silver: "#c0c0c0",
        skyBlue: "#87ceeb",
        slateBlue: "#6a5acd",
        slateGray: "#708090",
        slateGrey: "#708090",
        snow: "#fffafa",
        springGreen: "#00ff7f",
        steelBlue: "#4682b4",
        tan: "#d2b48c",
        teal: "#008080",
        thistle: "#d8bfd8",
        tomato: "#ff6347",
        turquoise: "#40e0d0",
        violet: "#ee82ee",
        wheat: "#f5deb3",
        white: "#ffffff",
        whiteSmoke: "#f5f5f5",
        yellow: "#ffff00",
        yellowGreen: "#9acd32"
      },
      
      parsedCache = {};
  
  /**
   * Ensures that v is between 0 and max.
   * @param {Number} v
   * @param {Number} max
   * @returns {Number}
   */
  function normalize(v, max) {
    return Math.max(0, Math.min(v, max));
  }
  
  /**
   * @ignore
   */
  function toHex(n) { 
    var s = Math.round(n).toString(16);
    return s.length === 1 ? '0' + s : s;
  }
  
  /**
   * Converts RGB numbers to a CSS rgb() or rgba() string.
   * 
   * @name niagara.util.color.toRgbCssString
   * @function
   * @param {Array} rgb an array of 3 numbers, 0 to 255 (or 4 numbers with
   * alpha from 0.0 - 1.0 as last)
   * @returns {String}
   */
  function toRgbCssString(r, g, b, a) {
    r = Math.round(r);
    g = Math.round(g);
    b = Math.round(b);
    
    if (typeof a === 'number') {
      return 'rgba(' + r + ',' + g + ',' + b + ',' + a + ')';
    }

    return 'rgb(' + r + ',' + g + ',' + b + ')';
  }
  
  /**
   * @ignore
   */
  function fromHex(hex) {
    return parseInt(hex, 16);
  }
  
  /**
   * @ignore
   */
  function parseRgbaStr(str) {
    if (str.charAt(str.length - 1) === "%") {
      // Convert percentage to 255 value
      var val =  parseInt(str.substring(0, str.length - 1), 10);
      return Math.round(val * 2.55);
    }

    return parseInt(str, 10);
  }

  /**
   * Calculates SMPTE C Rec. 709 luma weighting.
   * 
   * @name niagara.util.color.luma
   * @function
   * @param r
   * @param g
   * @param b
   * @returns {Number}
   * @see http://stackoverflow.com/questions/635022/calculating-contrasting-colours-in-javascript
   * @see http://en.wikipedia.org/wiki/Luma_%28video%29
   */
  function luma(r, g, b) {
    return (0.2126 * r) + (0.7152 * g) + (0.0722 * b); // SMPTE C, Rec. 709 weightings
  }
  
  /**
   * From http://mjijackson.com/2008/02/rgb-to-hsl-and-rgb-to-hsv-color-model-conversion-algorithms-in-javascript
   * 
   * Converts an RGB color value to HSV. Conversion formula
   * adapted from http://en.wikipedia.org/wiki/HSV_color_space.
   * Assumes r, g, and b are contained in the set [0, 255] and
   * returns h, s, and v in the set [0, 1].
   *
   * @name niagara.util.color.rgbToHsv
   * @function
   * @param   {Number}  r       The red color value
   * @param   {Number}  g       The green color value
   * @param   {Number}  b       The blue color value
   * @return  {Array}           The HSV representation
   */
  function rgbToHsv (r, g, b) {
    r = normalize(r, 255) / 255; 
    g = normalize(g, 255) / 255; 
    b = normalize(b, 255) / 255;
    
    var max = Math.max(r, g, b), 
        min = Math.min(r, g, b),
        h, 
        v = max,
        d = max - min,
        s = max === 0 ? 0 : d / max;

    if (max === min) {
      h = 0; // achromatic
    } else {
      switch (max) {
        case r: h = (g - b) / d + (g < b ? 6 : 0); break;
        case g: h = (b - r) / d + 2; break;
        case b: h = (r - g) / d + 4; break;
      }
      h /= 6;
    }

    return [h, s, v];
  }
  
  /**
   * Converts RGB numbers to an #RRGGBB string.
   * 
   * @name niagara.util.color.toHexCssString
   * @function
   * @param {Number} r 0 to 255
   * @param {Number} g 0 to 255
   * @param {Number} b 0 to 255
   * @returns {String}
   */
  function toHexCssString(r, g, b) {
    return '#' + toHex(r) + toHex(g) + toHex(b);
  }
  
  /**
   * A Color object that does some HSV calculation and CSS parsing on given 
   * RGB values.
   * Will have <code>r, g, b, a, css, h, s, v, hexString</code> properties. 
   * 
   * @class
   * @name niagara.util.color.Color
   * @param {Number} r 0 to 255
   * @param {Number} g 0 to 255
   * @param {Number} b 0 to 255
   * @param {Number} [a] 0 to 255 - defaults to 255
   * @param {String} [css] css representation of this color - defaults to
   * <code>rgb(r, g, b)</code> or <code>rgba(r, g, b, a)</code>
   * @returns {niagara.util.color.Color}
   */
  function Color(r, g, b, a, css) {
    var hsv = rgbToHsv(r, g, b),
        that = this;
    
    if (typeof a !== 'number') {
      a = 255;
    }
    
    that.red = r; 
    that.green = g;
    that.blue = b; 
    that.alpha = a;
    
    if (typeof css === 'string') {
      that.css = css;
    } else {
      that.css = toRgbCssString(r, g, b, a / 255); 
    }
    
    that.hexString = toHexCssString(r, g, b);
    
    that.h = hsv[0];
    that.s = hsv[1]; 
    that.v = hsv[2];
  }
  
  /**
   * From http://mjijackson.com/2008/02/rgb-to-hsl-and-rgb-to-hsv-color-model-conversion-algorithms-in-javascript
   * 
   * Converts an HSV color value to RGB. Conversion formula
   * adapted from http://en.wikipedia.org/wiki/HSV_color_space.
   * Assumes h, s, and v are contained in the set [0, 1] and
   * returns r, g, and b in the set [0, 255]. Returns Color object.
   *
   * @name niagara.util.color.hsvToRgb
   * @function
   * @param   {Number}  h       The hue
   * @param   {Number}  s       The saturation
   * @param   {Number}  v       The value
   * @return  {niagara.util.color.Color} 
   */
  function hsvToRgb(h, s, v) {
    h = normalize(h, 1);
    s = normalize(s, 1);
    v = normalize(v, 1);
    
    var r, g, b,

        i = Math.floor(h * 6),
        f = h * 6 - i,
        p = v * (1 - s),
        q = v * (1 - f * s),
        t = v * (1 - (1 - f) * s);

    switch (i % 6) {
        case 0: r = v; g = t; b = p; break;
        case 1: r = q; g = v; b = p; break;
        case 2: r = p; g = v; b = t; break;
        case 3: r = p; g = q; b = v; break;
        case 4: r = t; g = p; b = v; break;
        case 5: r = v; g = p; b = q; break;
    }

    return new Color(r * 255, g * 255, b * 255);
  }
  
  /**
   * Parse a string in <code>#RRGGBB</code>, <code>#AARRGGBB</code>,
   * <code>rgb(r, g, b)</code>, or <code>rgba(r, g, b, a)</code> form, or an
   * array with <code>[r, g, b, a]</code> values.
   * 
   * @name niagara.util.color.parseRgba
   * @function
   * @param {String|Array} str
   * @return {niagara.util.color.Color}
   */
  function parseRgba(str) {
    
    if (str instanceof Color) {
      return str;
    }
    
    var res,
        constant = COLOR_CONSTANTS[str],
        r = 0,
        g = 0,
        b = 0,
        a = 255,
        r1, r2, r3, r4,
        css,
        result;
    
    if ($.isArray(str)) {
      r = normalize(str[0], 255);
      g = normalize(str[1], 255);
      b = normalize(str[2], 255);
      
      if (typeof str[3] === 'number') {
        a = normalize(str[3], 255);
        return new Color(r, g, b, a);
      }
      
      return new Color(r, g, b);
      
    } 
    
    if (typeof str === 'string') {
      //parse string w/ regex
      
      if (parsedCache[str]) {
        return parsedCache[str];
      }
      
      if (typeof constant === 'string') {
        str = constant;
      }
  
      res = AARRGGBB_REGEX.exec(str);
      if (res) {
        r1 = res[1]; r2 = res[2]; r3 = res[3]; r4 = res[4];
        
        if (r4) { //have alpha
          a = fromHex(r1); 
          r = fromHex(r2); 
          g = fromHex(r3); 
          b = fromHex(r4);
          css = "rgba(" + r + "," + g + "," + b + "," + a + ")";
        } else {
          r = fromHex(r1); 
          g = fromHex(r2); 
          b = fromHex(r3);
          css = str;
        }
      }
      
      if (!res) {
        res = RGB_REGEX.exec(str);
        
        if (res) {
          r1 = res[1];
          r2 = res[2];
          r3 = res[3];
          r = fromHex(r1 + r1);
          g = fromHex(r2 + r2);
          b = fromHex(r3 + r3);
          css = str;
        }
      }
      
      if (!res) {
        res = CSS_RGB_REGEX.exec(str);
        if (res) {
          r = parseRgbaStr(res[1]);
          g = parseRgbaStr(res[2]);
          b = parseRgbaStr(res[3]);
          
          // Re-encode just incase percentages were used in encoding.
          css = "rgb("  + r + "," + g + "," + b + ")";
        }
      }
      
      if (!res) {
        res = CSS_RGBA_REGEX.exec(str);
        if (res) {
          r = parseRgbaStr(res[1]);
          g = parseRgbaStr(res[2]);
          b = parseRgbaStr(res[3]);
          a = parseFloat(res[4]) * 255;
          
          // Re-encode just incase percentages were used in encoding.
          css = "rgba(" + r + "," + g + "," + b + "," + a / 255 + ")";
        }
      }
      
      if (!res) {
        throw new Error("could not parse string: " + str);
      }
      
      parsedCache[str] = result = new Color(r, g, b, a, css);
    } else {
      throw new Error('invalid input to parseRgba: ' + str);
    }
    
    return result;
  }
  
  /**
   * Alpha blends foreground and background colors.
   * @param {niagara.util.color.Color} fore
   * @param {niagara.util.color.Color} back
   * @returns {niagara.util.color.Color} an alpha blended color
   */
  function alphaBlend(fore, back) {
    var r = fore.red,
        g = fore.green,
        b = fore.blue,
        a = fore.alpha / 255,
        bR = back.red,
        bG = back.green,
        bB = back.blue;
    
    // blended = a * f + (1 - a) * b
    // blended = a * (f - b) + b
    r = Math.round(a * (r - bR) + bR);
    g = Math.round(a * (g - bG) + bG);
    b = Math.round(a * (b - bB) + bB);
    
    return new Color(r, g, b);
  }
  
  /**
   * @namespace niagara.util.color
   */
  niagara.util.api('niagara.util.color', {
    COLOR_CONSTANTS: COLOR_CONSTANTS,
    
    Color: Color,
    
    alphaBlend: alphaBlend,
    hsvToRgb: hsvToRgb,
    luma: luma,
    parseRgba: parseRgba,
    rgbToHsv: rgbToHsv,
    toHexCssString: toHexCssString,
    toRgbCssString: toRgbCssString
  });
}());