/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Utility functions for the mobile scheduler app.
 * 
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, browser: true, plusplus: true */
/*global niagara, $, baja */


(function scheduleUtil() {
  "use strict";
  
  var util = niagara.util,
      DAY = util.time.MILLIS_IN_DAY,
      MINUTE = util.time.MILLIS_IN_MINUTE,
      getTimeLabels,
      mobileLex = baja.lex('mobile'),
      getDateboxDefaultOptions = niagara.fieldEditors.mobile.simple.getDateboxDefaultOptions,
      getBajaTimeFormat = niagara.fieldEditors.mobile.simple.getBajaTimeFormat;

  getTimeLabels = (function() {
    var times;
    return function() {
      // Format locale appropriate time labels. This is done lazily
      // so the time format system property will have been loaded from
      // the server.
      if (!times) {
        times = [
          baja.Time.make({ hour: 3 }).toString(),
          baja.Time.make({ hour: 6 }).toString(),
          baja.Time.make({ hour: 9 }).toString(),
          baja.Time.make({ hour: 12 }).toString(),
          baja.Time.make({ hour: 15 }).toString(),
          baja.Time.make({ hour: 18 }).toString(),
          baja.Time.make({ hour: 21 }).toString()
        ];
      }
      return times;
    };
  }());

  /**
   * Converts a pixel distance on a day div into the corresponding time
   * interval. For instance, if pixels === day.$dom.height(), then we're
   * converting the entire length of the day and the result will equal
   * MILLIS_IN_DAY.
   * 
   * @name niagara.util.schedule.pixelsToMillis
   * @function
   * 
   * @param {jQuery} dayDiv the div for the day whose pixels we're converting
   * @param {Number} the number of pixels on the y axis
   */
  function pixelsToMillis(dayDiv, pixels) {
    return (pixels / dayDiv.children('.blocksDisplay').outerHeight()) * 
              util.time.MILLIS_IN_DAY;
  }
  
  /**
   * Converts a number of milliseconds into the corresponding y-axis distance
   * on the day div. For instance, if millis === MILLIS_IN_DAY, then this
   * will return a number equal to day.$dom.height() since we are converting
   * the entire length of the day.
   *
   * @name niagara.util.schedule.millisToPixels
   * @function
   * 
   * @param {jQuery} dayDiv the div for the day whose time interval we're 
   * converting
   * @param {Number} millis the number of milliseconds to convert to pixels
   */
  function millisToPixels(dayDiv, millis) {
    return (millis / util.time.MILLIS_IN_DAY) * 
              dayDiv.children('.blocksDisplay').outerHeight();
  }
  
  /**
   * Converts a number of milliseconds to a percentage of a day. If millis
   * === MILLIS_IN_DAY then this function will return 1.
   * 
   * @name niagara.util.schedule.percentageOfDay
   * @function
   * 
   * @param {Number} millis milliseconds past midnight
   * @returns {Number} the percentage of a total day
   */
  function percentageOfDay(millis) {
    return millis / util.time.MILLIS_IN_DAY;
  }
  
  function addMonths(absTime, months) {
    var date = absTime.getDate(),
        year = date.getYear(),
        month = date.getMonth().getOrdinal() + months;
    if (month === -1) {
      year--;
      month = 11;
    } else if (month === 12) {
      year++;
      month = 0;
    }
    date = baja.Date.DEFAULT.make({
      year: year,
      month: month,
      day: 2
    });
    
    return baja.AbsTime.make({
      date: date,
      time: baja.Time.make(0)
    });
  }
  
//  function printEvent(event) {
//    return "client: " + event.clientX + "," + event.clientY + " " +
//      "layer: " + event.layerX + "," + event.layerY + " " +
//      "offset: " + event.offsetX + "," + event.offsetY + " " +
//      "page: " + event.pageX + "," + event.pageY + " " +
//      "screen: " + event.screenX + "," + event.screenY;
//  }
  
  function appendTimeLabels(targetElement) {
    var labelsDiv = $('<div class="scheduleLabels" />').appendTo(targetElement);

    baja.iterate(getTimeLabels(), function (time, i) {
      var div = $('<div/>').append($('<div class="labelDiv"/>').append($('<label/>').text(time)));
      div.addClass('scheduleLabel scheduleLabel' + i);
      labelsDiv.append(div);
    });
    
    return labelsDiv;
  }
  
  (function dateboxFunctions() {
    var IS_DATEBOX_SELECTOR = ':jqmData(role=datebox)';
    
    /**
     * We've attempted to move back the start time of a schedule block to a time
     * that conflicts with another block - find the earliest we can possibly
     * set it to without conflicts.
     *  
     * @memberOf niagara.util.schedule.datebox
     * @private
     * @param {niagara.fieldEditors.schedule.DayEditor} dayEditor the day editor
     * for the day we're moving blocks around in
     * @returns {baja.Time} the earliest time we can set the start to without
     * conflicts
     */
    function earliestStartMillis(dayEditor) {
      var scheduleBlock = dayEditor.getSelectedBlock(),
          earliestms = 0,
          startms = scheduleBlock.start.getTimeOfDayMillis();
      
      baja.iterate(dayEditor.getBlocks(), function (myBlock, i) {
        if (myBlock !== scheduleBlock) {
          var finishms = myBlock.finish.getTimeOfDayMillis();
          
          if (finishms > 0 &&
              finishms <= startms && 
              (!earliestms || earliestms < finishms)) {
            earliestms = finishms;
          }
        }
      });
      
      return earliestms;
    }
    
    /**
     * We've attempted to move up the finish time of a schedule block to a time
     * that conflicts with another block - find the latest we can possibly
     * set it to without conflicts.
     *  
     * @memberOf niagara.util.schedule.datebox
     * @private
     * @param {niagara.fieldEditors.schedule.DayEditor} dayEditor the day editor
     * for the day we're moving blocks around in
     * @returns {baja.Time} the latest time we can set the finish to without
     * conflicts
     */
    function latestFinishMillis(dayEditor) {
      var scheduleBlock = dayEditor.getSelectedBlock(),
          latestms = DAY,
          finishms = scheduleBlock.finish.getTimeOfDayMillis();
      
      baja.iterate(dayEditor.getBlocks(), function (myBlock, i) {
        if (myBlock !== scheduleBlock) {
          var startms = myBlock.start.getTimeOfDayMillis();
          
          if (startms >= finishms && (!latestms || latestms > startms)) {
            latestms = startms;
          }
        }
      });
      
      return latestms % DAY;
    }
    
    /**
     * Check to see if our time schedule is valid - that is, the start time
     * is before the end time (or the end time is 12 midnight), and both
     * the start and finish times do not conflict with any other schedule
     * blocks in our day.
     * 
     * @memberOf niagara.util.schedule.datebox
     * @param {niagara.fieldEditors.schedule.DayEditor} dayEditor the day
     * editor we're moving blocks around in
     * @param {baja.Time} startTime the current (pre-edit) start time
     * @param {baja.Time} finishTime the current (pre-edit) finish time
     * @param {String} slot the slot we wish to edit ('start' or 'finish')
     * @param {Number} mschange the change in ms we wish to make
     * @throws {Error} if time change is not valid - error text is an
     * appropriate message from the lexicon
     */
    function validateTimeChange(dayEditor, startTime, finishTime, slot, mschange) {
      var scheduleBlock = dayEditor.getSelectedBlock(),
          startms = startTime.getTimeOfDayMillis(),
          finishms = finishTime.getTimeOfDayMillis(),
          errorLexKey;
      
      if (slot === 'start') {
        startms += mschange;
        //start can't cross midnight boundary
        if (startms < 0 || startms >= DAY) {
          errorLexKey = 'schedule.message.crossesMidnight';
        }
      } else {
        finishms = finishms || DAY;
        finishms += mschange;
        //finish can't cross midnight boundary
        if (finishms <= 0 || finishms > DAY) {
          errorLexKey = 'schedule.message.crossesMidnight';
        }
      }
      

      
      //can't overlap with another block
      if (!dayEditor.canMoveBlockTo(scheduleBlock, startms, finishms)) {
        errorLexKey = 'schedule.message.blockOverlap';
      }
      
      //can't have finish before start
      if ((finishms || DAY) <= startms) {
        errorLexKey = 'schedule.message.finishBeforeStart';
      }
      
      if (errorLexKey) {
        throw new Error(mobileLex.get({
          key: errorLexKey,
          args: [ startTime.toString({textPattern: getBajaTimeFormat() || 'HH:mm'}),
                  finishTime.toString({textPattern: getBajaTimeFormat() || 'HH:mm'}) ] 
        }));
      }
    }
    
    /**
     * We've gotten an invalid time from a TimeSchedule editor - this
     * function returns a corrected, valid time. Prevents such errors as
     * finish time before start time, crossing over midnight, and overlapping
     * with another schedule block.
     * 
     * @memberOf niagara.util.schedule.datebox
     * @private
     * @param {niagara.fieldEditors.schedule.DayEditor} dayEditor the field
     * editor for the currently edited day
     * @param {Number} startms the current (pre-edit) start time for the 
     * TimeSchedule (in ms past midnight)
     * @param {Number} finishms the current (pre-edit) finish time for the 
     * TimeSchedule (in ms past midnight)
     * @param {String} slot the slot we are editing on the TimeSchedule
     * ('start' or 'finish')
     * @param {Number} mschange the change, in milliseconds, we are making
     * to the edited slot
     */
    function getCorrectedTime(dayEditor, startms, finishms, slot, mschange) {
      var direction = mschange > 0 ? 'down' : 'up',
          newstartms = startms + (slot === 'start' ? mschange : 0),
          newfinishms = finishms + (slot === 'finish' ? mschange : 0);
      
      newstartms = (newstartms + DAY) % DAY;
      newfinishms = (newfinishms + DAY) % DAY;
      
      switch (slot + ' ' + direction) {
      case 'start down':
        return (newfinishms || DAY) - MINUTE;
      case 'start up':
        return earliestStartMillis(dayEditor);
      case 'finish down':
        if (finishms + mschange > DAY) {
          return 0;
        } else {
          return latestFinishMillis(dayEditor);  
        }
      case 'finish up':
        return newstartms + MINUTE;
      default:
        throw "slot must be one of 'start' or 'finish'";
      }
    }
    
    /**
     * Checks to ensure that the given time schedule (output from the field
     * editor on the editBlock page) is valid - if not, reset the start/finish
     * time on the datebox editor to the earliest/latest time that <i>is</i>
     * valid. Call this method each time the <code>datebox</code> event is
     * triggered from your field editor.
     * 
     * @memberOf niagara.util.schedule.datebox
     * @param {jQuery} input the <code>datebox</code> input element we're
     * changing (either the start or finish slot)
     * @param {niagara.fieldEditors.schedule.DayEditor} dayEditor the 
     * field editor for the currently edited day
     * @param {niagara.fieldEditors.BaseFieldEditor} timeScheduleEditor the 
     * field editor for the <code>schedule:TimeScheduleEditor</code> we're
     * editing
     * @param {Number} mschange the amount we're changing the start/finish
     * time, in milliseconds (e.g. -60000 for "backwards one minute", 
     * 3600000 for "forwards one hour")
     */
    function validateTimeSchedule(input, dayEditor, timeScheduleEditor, mschange) {
      timeScheduleEditor.getSaveData({
        ok: function (timeSchedule) {
          var slot = input.attr('name').replace('_time', ''), //start or finish
              startTime = timeSchedule.getStart(),
              finishTime = timeSchedule.getFinish(),
              startms = startTime.getTimeOfDayMillis(),
              finishms = finishTime.getTimeOfDayMillis(),
              newms,
              newTime;
          
          
          try {
            validateTimeChange(dayEditor, startTime, finishTime, slot, mschange);
            newms = (slot === 'start' ? startms : finishms);
            newms = (newms + mschange + DAY) % DAY;
          } catch (e) {
            newms = getCorrectedTime(dayEditor, startms, finishms, slot, mschange);
          }
          
          newTime = baja.Time.make(newms);
          
          setTimeout(function () {
            input.trigger('datebox', {
              method: 'set',
              value: newTime.toString({textPattern: getBajaTimeFormat() || 'HH:mm'})
            });
          }, 0);
        }
      });
    }
    
    /**
     * Retrieves or creates an datebox input element inside the given element.
     * If the input does not exist, it will be created inside of a div with
     * class <code>calendarContainer</code>. Input will be created in calendar 
     * box mode ('calbox').
     * 
     * @param {jQuery} element element to search for a datebox input, or to
     * create one if it doesn't exist
     * @returns {jQuery} an input element to be dateboxed
     */
    function getDateboxInput(element) {
      if (element.is(IS_DATEBOX_SELECTOR)) {
        return element;
      }
      
      var input = element.find(IS_DATEBOX_SELECTOR),
          options,
          calendarContainer,
          date,
          year,
          month;
      
      if (!input.length) {
        calendarContainer = $('<div class="calendarContainer"/>');
        options = getDateboxDefaultOptions();
        options.mode = 'calbox';
        options.useInline = true;
        options.useInlineHideInput = true;
        options.calHighToday = false;
        options.calHighPicked = false;
        
        input = $('<input class="hidden" type="date" data-role="datebox" />')
          .jqmData('options', options)
          .appendTo(calendarContainer)
          .datebox();
        
        element.html(calendarContainer);
      }
      
      return input;
    }
    
    /**
     * Finds a datebox input inside the given element, pulls the selected
     * date from it, and converts it to an AbsTime. If no datebox element is
     * found, will return the current time.
     * 
     * @param {jQuery} element the element to search for a datebox input
     * @returns {baja.AbsTime}
     */
    function getAbsTime(element) {
      var input = getDateboxInput(element),
          datebox = input.jqmData('datebox'),
          jsDate = datebox ? datebox.theDate : new Date(),
          absTime = baja.AbsTime.make({ 
            jsDate: jsDate
          });
      
      return absTime;
    }
    

    
    /**
     * @namespace
     * @name niagara.util.schedule.datebox
     */
    util.api('niagara.util.schedule.datebox', {
      'public': {
        getAbsTime: getAbsTime,
        getDateboxInput: getDateboxInput,
        validateTimeChange: validateTimeChange,
        validateTimeSchedule: validateTimeSchedule
      },
      'private': {
        earliestStartMillis: earliestStartMillis,
        getCorrectedTime: getCorrectedTime,
        latestFinishMillis: latestFinishMillis
      }
    });
  }());
  
  /**
   * @namespace
   * @name niagara.util.schedule
   */
  util.api('niagara.util.schedule', {
    addMonths: addMonths,
    appendTimeLabels: appendTimeLabels,
    millisToPixels: millisToPixels,
    percentageOfDay: percentageOfDay,
    pixelsToMillis: pixelsToMillis
//    printEvent: printEvent
  });
}());