/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to displaying and updating calendar
 * widgets within the mobile scheduler app.
 * 
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, bitwise: true */
/*global niagara, baja, $ */


(function calendarUI() {
  "use strict";
  
  niagara.util.require(
   'niagara.fieldEditors',
   'niagara.util.flow',
   'jQuery.mobile.datebox'
  );
  
  var util = niagara.util,
      dateboxUtil = util.schedule.datebox,
      dialogs = util.mobile.dialogs,
      fe = niagara.fieldEditors,
      callbackify = util.callbackify,
      
      SPINNER_DELAY = 2000;
  
  /**
   * Creates a Bajascript <code>BVector</code>, holding a 
   * <code>BWeeklySchedule</code> and a <code>BAbsTime</code>. This vector will 
   * be sent to <code>BScheduleServiceSideCallHandler#getHighlightedDates</code>.
   * 
   * @private
   * @memberOf niagara.schedule.ui.calendar
   * @param {baja.Component} schedule a <code>schedule:WeeklySchedule</code> to
   * send
   * @param {baja.AbsTime} absTime the time to send
   */
  function makeVector(schedule, absTime) {
    var vec = baja.$('baja:Vector');
    vec.add({ 
      slot: 'time', 
      value: absTime
    });
    vec.add({ 
      slot: 'schedule', 
      value: schedule
    });
    return vec;
  }
  
  function appendRefBase(schedule) {
    var obj = {
      slot: 'refBase',
      value: baja.Ord.make(niagara.view.ord),
      flags: baja.Flags.HIDDEN | baja.Flags.TRANSIENT
    };

    // Invoked in BWeekOfMonthSchedule.findFirstDayOfWeek()
    if (schedule.has('refBase')) {
      schedule.set(obj);
    } else {
      schedule.add(obj);
    }
  }
  
  /**
   * Converts an AbsTime to a datebox-friendly format ready to load into
   * a calendar.
   * @param {baja.AbsTime} absTime
   * @returns {String}
   */
  function toDateboxFormat(absTime) {
    var date = absTime.getDate(),
        year = date.getYear(),
        month = date.getMonth().getOrdinal() + 1;
    return year + '-' + (month < 10 ? '0':'') + month + '-01';
  }
  
  /**
   * Calls a server side handler method on 
   * <code>ScheduleServerSideCallHandler</code>.
   * @private
   * @param {baja.Component} a <code>schedule:AbstractSchedule</code>
   * @param {baja.AbsTime} AbsTime
   * @param {String} methodName
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  function doSSC(schedule, absTime, methodName, callbacks) {
    callbacks = callbackify(callbacks);
    
    schedule = schedule.newCopy();
    appendRefBase(schedule);
    
    var vec = makeVector(schedule, absTime);
    
    baja.Ord.make(niagara.view.ord).get({
      ok: function (component) {
        component.serverSideCall({
          typeSpec: 'mobile:ScheduleServerSideCallHandler',
          methodName: methodName,
          value: vec,
          ok: function (val) {
            callbacks.ok(JSON.parse(val));
          },
          fail: callbacks.fail
        });        
      },
      fail: callbacks.fail
    });
  }
  
  /**
   * Given a BAbstractSchedule and an AbsTime, will make a server side call
   * to <code>BScheduleServerSideCallHandler#getHighlightedDates</code>
   * and return its output (an array of dates that are in the same month as
   * the AbsTime and are active in the given schedule).
   * 
   * @param {baja.Component} a BAbstractSchedule
   * @param {baja.AbsTime} get dates for this month
   * @param {Object} callbacks an object containing ok/fail callbacks
   * @returns {Array} highlighted dates, passed to the ok callback
   */
  function getActiveDates(schedule, absTime, callbacks) {
    doSSC(schedule, absTime, 'getHighlightedDates', callbacks);
  }

  /**
   * Get a daily summary of events for the given date, using a server side
   * call to <code>BScheduleServerSideCallHandler#getDaySummary</code>,
   * for display on the summary tab.
   * 
   * @param {baja.Component} a BWeeklySchedule
   * @param {baja.AbsTime} the day to show a summary for
   * @param {Object} callbacks an object containing ok/fail callbacks
   * @returns {Array} an array of day summaries
   */
  function getDaySummary(schedule, absTime, callbacks) {
    doSSC(schedule, absTime, 'getDaySummary', callbacks);
  }

  
  /**
   * Displays/updates a JQM Datebox calendar in the target div. It will 
   * retrieve a list of the effective dates and highlight those dates in the
   * datebox calendar.
   * 
   * @memberOf niagara.schedule.ui.calendar
   * @param {jQuery} targetElement the div in which to show the calendar
   * @param {baja.Component} schedule a <code>schedule:AbstractSchedule</code> 
   * to check for effective dates
   * @param {baja.AbsTime} absTime the target date to show a calendar for (will
   * be truncated to month)
   */
  function showEffectiveRangeCalendar(targetElement, schedule, absTime, callbacks) {
    callbacks = callbackify(callbacks);
    
    var ticket = util.mobile.spinnerTicket(SPINNER_DELAY);
    
    getActiveDates(schedule, absTime, {
      ok: function (activeDates) {
        var input = dateboxUtil.getDateboxInput(targetElement);
        
        //highlight active dates - will redraw in trigger('datebox')
        input.jqmData('datebox').options.highDates = activeDates;
        
        input.trigger('datebox', {
          method: 'set',
          value: toDateboxFormat(absTime)
        });
        
        ticket.hide();
        
        callbacks.ok();
      },
      fail: dialogs.error
    });
  }
  
  function makeDaySummaryList(events) {
    var ul = $('<ul class="daySummary" data-role="listview"/>');
    baja.iterate(events, function (eventObj) {
      var li = $('<li/>'),
          title = $('<label class="daySummaryTitle"/>').appendTo(li),
          subtitle = $('<label class="daySummarySubtitle"/>').appendTo(li);
      
      title.text(eventObj.timestamp);
      subtitle.text(eventObj.output + ' (' + eventObj.source + ')');
      li.appendTo(ul);
    });
    return ul;
  }
  
  function showDaySummary(targetElement, schedule, absTime) {
    var ticket = util.mobile.spinnerTicket(SPINNER_DELAY);
    
    getDaySummary(schedule, absTime, {
      ok: function (daySummary) {
        var ul = makeDaySummaryList(daySummary);
        targetElement.html(ul);
        ul.listview();
        
        //scroll down to show the summary output
        $.mobile.silentScroll(targetElement.offset().top);
        ticket.hide();
      },
      fail: dialogs.error
    });
  }
  
  /**
   * Loads field editors for the effective date range of a schedule - will
   * show day, month, and year dropdowns for both start and end effective dates.
   * 
   * @memberOf niagara.schedule.ui.calendar
   * @param {jQuery} page the page in which to show the date range field
   * editor
   * @param {baja.Component} schedule the <code>schedule:WeeklySchedule</code>
   * whose <code>'effective'</code> property we will edit
   * @param {Object} callbacks an object containing ok/fail callbacks
   * @returns {niagara.fieldEditors.BaseFieldEditor} a field editor for a
   * <code>schedule:DateRangeSchedule</code>
   */
  function loadScheduleEditor(page, schedule, callbacks, readonly) {
    callbacks = util.callbackify(callbacks);

    var contentDiv = page.children(':jqmData(role="content")'),
        editorDiv = contentDiv.find('div:jqmData(role="calendareditor")'),
        calendarDiv = contentDiv.find('div:jqmData(role="calendarpreview")');

    fe.makeFor({
      value: schedule,
      readonly: readonly
    }, function (scheduleEditor) {
      scheduleEditor.buildAndLoad(editorDiv.empty(), callbacks);
    });
  }
  
  /**
   * Binds an editor's <code>editorchange</code> event to a calendar div, so
   * that whenever the editor changes, the calendar display will perform the
   * server side call to display the new effective date range.
   * 
   * @memberOf niagara.schedule.ui.calendar
   * @param {niagara.fieldEditors.BaseFieldEditor} editor the editor to bind to
   * a calendar
   * @param {jQuery} calendarDiv the div where the datebox calendar lives
   */
  function bindEditorToCalendar(editor, calendarDiv) {
    var editorDiv = editor.$dom;
    
    editorDiv.unbind('editorchange');
    editorDiv.bind('editorchange', function (event) {
      var input = dateboxUtil.getDateboxInput(calendarDiv),
          theDate = input.jqmData('datebox').theDate;
      
      //TODO: temporary hack until time zones are fixed
      theDate = new Date(theDate.getTime() + util.time.MILLIS_IN_DAY);
      
      editor.getSaveData({
        ok: function (saveData) {
          showEffectiveRangeCalendar(
              calendarDiv, 
              saveData, 
              baja.AbsTime.make({jsDate: theDate}));
        },
        fail: dialogs.error
      });
    });
  }
  
  /**
   * @namespace
   * @name niagara.schedule.ui.calendar
   */
  util.api('niagara.schedule.ui.calendar', {
    'public': {
      bindEditorToCalendar: bindEditorToCalendar,
      loadScheduleEditor: loadScheduleEditor,
      showDaySummary: showDaySummary,
      showEffectiveRangeCalendar: showEffectiveRangeCalendar
    },
    'private': {
      makeVector: makeVector
    }
  });
}());