/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to the ScheduleBlock object that contains a
 * BTimeSchedule component within a Day. 
 * 
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true */
/*global niagara, $, baja */


(function () {
  "use strict";
  
  niagara.util.require(
    'jQuery',
    'niagara.util.time',
    'niagara.util.schedule'
  );
  
  var util = niagara.util,
      timeUtil = util.time,
      scheduleUtil = util.schedule,
      ScheduleColors = niagara.schedule.scheduleGradients.ScheduleColors,
      getBajaTimeFormat = niagara.fieldEditors.mobile.simple.getBajaTimeFormat,
      
      ScheduleBlock;
  
  /**
   * Represents one schedule block in a day object.
   * <p>
   * The constructor can also take a component of type
   * <code>schedule:TimeSchedule</code>, in which case the fields will be
   * populated using its <code>start</code>, <code>finish</code>, and 
   * <code>effectiveValue</code> slots.
   * 
   * @class
   * @memberOf niagara.schedule
   * 
   * @param {baja.Time} start
   * @param {baja.Time} finish
   * @param {baja.Struct} value (<code>baja:StatusValue</code>)
   */
  ScheduleBlock = function ScheduleBlock(start, finish, value) {
    var that = this,
        timeSchedule;
    
    if (start.getType().is('schedule:TimeSchedule')) {
      timeSchedule = start;
      that.start = timeSchedule.get('start');
      that.finish = timeSchedule.get('finish');
      that.value = timeSchedule.get('effectiveValue');
    } else {
      that.start = start;
      that.finish = finish;
      that.value = value;
    }
    
    that.$colors = new ScheduleColors(that.value);
  };
  
  /**
   * Two schedule blocks are equivalent if they both represent the same
   * start, finish, and value.
   * @name niagara.schedule.ScheduleBlock#equivalent
   * @function
   * 
   * @param {niagara.schedule.ScheduleBlock} other the other schedule block
   * @returns {Boolean} true if both blocks represent the same start, finish,
   * and value.
   */
  ScheduleBlock.prototype.equivalent = function (other) {
    return other instanceof ScheduleBlock &&
           other.start.equivalent(this.start) &&
           other.finish.equivalent(this.finish) &&
           other.value.equivalent(this.value);
  };
  
  /**
   * Creates a div to display this schedule block - a simple div with two child
   * "grab" divs to use for resizing. The returned div's 
   * <code>data('scheduleBlock')</code> will be set to <code>this</code>.
   * 
   * @name niagara.schedule.ScheduleBlock#generateDiv
   * @function
   * 
   * @param {jQuery} targetElement the div into which to insert our own div
   * @returns {jQuery} a div representing this schedule block
   */
  ScheduleBlock.prototype.generateDiv = function generateDiv(targetElement) {
    var containerDiv = $('<div class="scheduleBlockContainer" />')
          .append($('<div class="grab top" />'))
          .append($('<div class="grab bottom" />'))
          .appendTo(targetElement),
        blockDiv = $('<div class="scheduleBlock" />')
          .appendTo(containerDiv),
        displayDiv = $('<div class="display" />')
          .appendTo(blockDiv);
    
    displayDiv.append('<span class="valueDisplay"/>&nbsp;<span class="timeDisplay"/>');
    
    containerDiv.data('scheduleBlock', this);
    return containerDiv;
  };
  
  /**
   * Returns the BTimeSchedule component represented by this block.
   * 
   * @name niagara.schedule.ScheduleBlock#getTimeSchedule
   * @function
   * 
   * @returns {baja.Component} the <code>schedule:TimeSchedule</code> component
   * represented by this block
   */
  ScheduleBlock.prototype.createTimeSchedule = function createTimeSchedule() {
    var ev = 'effectiveValue',
        ts = baja.$('schedule:TimeSchedule', {
          start: this.start,
          finish: this.finish
        });
    
    ts.add({
      slot: 'effectiveValue',
      value: this.value.newCopy(),
      cx: {
        displayName: baja.lex('schedule').get('effectiveValue') || 'effectiveValue'
      }
    });
    
    return ts;
  };
  
  /**
   * Checks to see if this block overlaps with a time period defined by the
   * given start and finish times.
   * 
   * @name niagara.schedule.ScheduleBlock#overlaps
   * @function
   * 
   * @param {Number} startms the start time to check (in ms past midnight)
   * @param {Number} finishms the finish time to check (in ms past midnight)
   * @returns {Boolean} true if this block overlaps with the specified time
   * period
   */
  ScheduleBlock.prototype.overlaps = function overlaps(startms, finishms) {
    var s1 = this.start.getTimeOfDayMillis(), 
        f1 = this.finish.getTimeOfDayMillis(),
        s2 = startms,
        f2 = finishms;
    
    if (f1 === 0) {
      f1 = timeUtil.MILLIS_IN_DAY;
    }
    
    if (f2 === 0) {
      f2 = timeUtil.MILLIS_IN_DAY;
    }
    
    if (f1 < s1) {
      f1 += timeUtil.MILLIS_IN_DAY;
    }
    if (f2 < s2) {
      f2 += timeUtil.MILLIS_IN_DAY;
    }
    
    return f1 > s2 && f2 > s1;
  };
  
  /**
   * Shows this block's time range and effective value.
   * 
   * @name niagara.schedule.ScheduleBlock#toString
   * @function
   */
  ScheduleBlock.prototype.toString = function toString() {
    function timeString(time) {
      return time.toString({ textPattern: getBajaTimeFormat() || 'HH:mm' });
    }
    
    var start = this.start,
        finish = this.finish,
        display = "";
    
    if (start) {
      display += timeString(start) + " - ";
    }
    if (finish) {
      display += timeString(finish);
    }
    
    return display;
  };
  
  util.api('niagara.schedule', {
    ScheduleBlock: ScheduleBlock
  });
}());