/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to the Schedule object that represents
 * a BWeeklySchedule component. 
 * 
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true */
/*global baja, $, niagara */

(function () {
  "use strict";
  
  niagara.util.require(
    'niagara',
    'niagara.util',
    'niagara.util.mobile.pages',
    'niagara.util.schedule',
    'niagara.fieldEditors.mobile',
    'niagara.fieldEditors.schedule.DayEditor'
  );

  //imports
  var util = niagara.util,
      callbackify = util.callbackify,
      timeUtil = util.time,
      scheduleUtil = util.schedule,
      fe = niagara.fieldEditors,
      DayEditor = niagara.schedule.DayEditor,
      MobileFieldEditor = fe.mobile.MobileFieldEditor,
      
      //private vars
      editedScheduleBlock,
      editedDay,
      timeScheduleEditor,
      getOrderedDayNames,

      //exports
      Schedule;

  /*
   * Returns an array of Weekday names, ordered according to the first day
   * of the week for the user's locale.
   */
  getOrderedDayNames = (function() {
    var dayNames;
    return function() {
      var firstDay,
          firstDayTag,
          i,
          j = 0;

      if (!dayNames) {
        dayNames = [];
        firstDayTag = baja.lex('baja').get('weekday.firstDayOfWeek');
        if (!firstDayTag) {
          firstDayTag = 'sunday';
        }
        firstDay = baja.$('baja:Weekday').get(firstDayTag);
        dayNames[j++] = firstDayTag;
        for (i = firstDay.getOrdinal() + 1; i <= 6; i++) {
          dayNames[j++] = baja.$('baja:Weekday').get(i).getTag();
        }
        for (i = 0; i < firstDay.getOrdinal(); i++) {
          dayNames[j++] = baja.$('baja:Weekday').get(i).getTag();
        }
      }
      return dayNames;
    };
  }());
  
  /**
   * Gets a display string for a boolean schedule, taking into account
   * the trueText/falseText facets set on the schedule itself.
   * 
   * @memberOf niagara.schedule
   * @private
   * @param {baja.Struct} statusBoolean a baja:StatusBoolean value used in a
   * boolean schedule
   */
  function toStatusBooleanString(weeklySchedule, statusBoolean) {
    var facets = util.slot.getFacets(weeklySchedule),
        bool = statusBoolean.getValue(),
        result;
    
    if (facets) {
      result = facets.get(bool ? 'trueText' : 'falseText');
    }
    
    if (result) {
      return baja.Format.format(result);
    } else {
      return baja.lex('baja').get(bool ? 'true' : 'false');
    }
  }
  
  /**
   * @memberOf niagara.schedule
   * @private
   */
  function toStatusNumericString(weeklySchedule, statusNumeric, callbacks) {
    var facets = util.slot.getFacets(weeklySchedule),
        units = facets && facets.get('units'),
        unitSplit, unitName, unitSymbol, unitDimension,
        precision = (facets && facets.get('precision')) || 0,
        num = statusNumeric.getValue(),
        str = num.toFixed(precision);
    
    if (units) {
      unitSplit = units.encodeToString().split(';');
      unitName = unitSplit[0];
      unitSymbol = unitSplit[1];
      unitDimension = unitSplit[2];
      fe.mobile.units.getQuantityDb(function (db) {
        var units = db.getUnitFromNameAndDimension(unitName, unitDimension);
        if (units) {
          unitSymbol = units.s;
          if (units.p) {
            str = unitSymbol + ' ' + str;
          } else {
            str += ' ' + unitSymbol;
          }
        } else {
          str += ' ' + unitSymbol;
        }
        callbacks.ok(str);
      });
    } else {
      callbacks.ok(str);
    }
  }
  
  /**
   * Gets a display string for an enum schedule.
   * @memberOf niagara.schedule
   * @private
   */
  function getEnumRangeDisplay(ordinal, range) {
    try {
      var lexicon = range.getOptions().get('lexicon'),
          got = range.get(ordinal),
          tag = got.getTag();
      if (lexicon) {
        return baja.lex(lexicon).get({
          key: tag,
          def: tag
        });
      } else {
        return baja.SlotPath.unescape(got.toString());
      } 
    } catch (err) {
      return String(ordinal);     
    }
  }
  
  
  /**
   * A field editor displaying the seven days of a
   * <code>schedule:WeeklySchedule</code> in a calendar-like grid, like the
   * existing schedule editor in Workbench. A Schedule is not directly 
   * user-editable - it is intended to link off to field editors for the
   * individual day schedules, special events, etc.
   * <p>
   * The <code>WeeklySchedule</code> it contains is really intended to be an
   * unmounted snapshot (retrieve using 
   * <code>niagara.schedule.ui.getSnapshot</code>), since there are many 
   * different properties that need to be edited individually and saved all at 
   * once. To facilitate this, <code>doSaveValue</code> sends the entire 
   * <code>WeeklySchedule</code> up to a server side handler for an 
   * all-or-nothing save operation.
   * 
   * @class
   * @name niagara.schedule.Schedule
   * @extends niagara.fieldEditors.mobile.MobileFieldEditor
   */
  Schedule = fe.defineEditor(MobileFieldEditor, {
    postCreate: function postCreate() {
      this.dayEditors = {};
    },
    
    /**
     * Creates a <code>&lt;div&gt;</code> element to display this schedule -
     * seven divs side by side that will hold the week's seven days. (The
     * actual editors for the days themselves are instantiated and built in
     * <code>doLoadValue</code>).
     * 
     * @name niagara.schedule.Schedule#doInitializeDOM
     * @function
     * @private
     * 
     * @param {jQuery} targetElement the div into which we will insert this 
     * schedule's div
     * @param {Object} callbacks an object containing ok/fail callbacks
     */
    doInitializeDOM: function doInitializeDOM(targetElement, callbacks) {
      var scheduleDiv = $('<div class="schedule" />'),
          labelsDiv = scheduleUtil.appendTimeLabels(scheduleDiv),
          weekdaysDiv = $('<div class="weekdays" />').appendTo(scheduleDiv);
    
      scheduleDiv.appendTo(targetElement);
      
      callbacks.ok();
    },
    
    /**
     * Rebuilds all the field editors for the days in the schedule (Sunday to
     * Saturday). Day field editors are all in read-only mode (must navigate
     * to editDay/editSpecialEvent to use a writable day editor).
     * 
     * @name niagara.schedule.Schedule#doLoadValue
     * @function
     * @private
     * 
     * @param {baja.Component} weeklySchedule the 
     * <code>schedule:WeeklySchedule</code> to load
     * @param {Object} callbacks an object containing ok/fail callbacks
     */
    doLoadValue: function doLoadValue(weeklySchedule, callbacks) {
      if (!weeklySchedule.getType().is('schedule:WeeklySchedule')) {
        return callbacks.ok();
      }
      
      var that = this,
          dom = that.$dom,
          parent = dom.parent(),
          week = weeklySchedule.getSchedule().get('week'),
          weekdaysDiv;
      
      dom.remove();
      weekdaysDiv = dom.find('div.weekdays').empty();

      baja.iterate(getOrderedDayNames(), function (dayName) {
        var dailySchedule = week.get(dayName);
        
        fe.makeFor({
          container: dailySchedule,
          slot: 'day',
          readonly: true,
          label: baja.lex('baja').get(dayName + '.short')
        }, function (editor) {
          var dayContainer = $('<div class="dayContainer"/>').appendTo(weekdaysDiv);
          editor.buildAndLoad(dayContainer);
          that.dayEditors[dayName] = editor;
        });
      });
      
      that.defaultValue = weeklySchedule.getDefaultOutput();
    
      dom.appendTo(parent);
      
      callbacks.ok();
    },
    
    /**
     * Does nothing - just passes <code>this.value</code> to
     * <code>callbacks.ok</code>.
     * 
     * @name niagara.schedule.Schedule#getSaveData
     * @function
     * @private
     * 
     * @param {Object} callbacks an object containing ok/fail callbacks
     */
    getSaveData: function getSaveData(callbacks) {
      callbacks.ok(this.value);
    },
    
    /**
     * Passes our edited local instance of the <code>WeeklySchedule</code>
     * up to the server to be saved. It does this via server side
     * call handler <code>mobile:ScheduleServerSideCallHandler#save</code>.
     * 
     * <p>Note: the save() server side call returns a new snapshot of the
     * schedule just as if you had called <code>getSnapshot()</code>. However,
     * there's no way to simply return this snapshot in the callback, as field 
     * editors on Components require edit-by-ref semantics (therefore the 
     * return value is always the input value). After saving the schedule
     * editor, the new snapshot will be available simply as 
     * <code>this.snapshot</code>.
     * 
     * @name niagara.schedule.Schedule#doSaveValue
     * @function
     * @private
     * 
     * @param {baja.Component} valueToSave the edited local instance of
     * <code>WeeklySchedule</code> to save
     * @param {baja.Component} [saveData] the output of 
     * <code>getSaveData</code> (here, the exact same instance as 
     * <code>valueToSave</code> and thus not used)
     * @param {Object} callbacks an object containing ok/fail callbacks
     * 
     * @see niagara.schedule.Schedule#getSaveData
     */
    doSaveValue: function doSaveValue(valueToSave, saveData, callbacks) {
      var that = this;
      baja.Ord.make(niagara.view.ord).get({
        ok: function (component) {
          component.serverSideCall({
            typeSpec: 'mobile:ScheduleServerSideCallHandler',
            methodName: 'save',
            value: valueToSave,
            ok: function (snapshot) {
              that.snapshot = snapshot;
              callbacks.ok(valueToSave);
            },
            fail: callbacks.fail
          });
        },
        fail: callbacks.fail
      });
    }
  });

  Schedule.prototype.refreshWidgets = function () {
    baja.iterate(this.dayEditors, function (editor) {
      editor.refreshWidgets();
    });
  };

  
  /**
   * Wipes out this schedule's collection of <code>DayEditor</code>s from Monday to
   * Friday and replaces them all with the <code>BTimeSchedule</code>s from
   * the given day.
   * 
   * @name niagara.schedule.Schedule#applyMF
   * @function
   * 
   * @param {niagara.schedule.Day} day the day to apply Monday-Friday
   */
  Schedule.prototype.applyMF = function applyMF(dayEditor) {
    var oldDayEditor,
        that = this;

    // Iterate by weekday tag rather than array index - don't assume that a
    // particular day of week is always at a known position in the dayEditors
    // array.

    baja.iterate(getOrderedDayNames(), function (dayName) {
      if (dayName === 'saturday' || dayName === 'sunday') {
        return;
      }

      oldDayEditor = that.dayEditors[dayName];
      
      if (oldDayEditor.value !== dayEditor.value) {
        dayEditor.copyTo(oldDayEditor);
        oldDayEditor.saveValue();
      }
    });
    
    this.setModified(true);
  };
  
  /**
   * Overwrites all time schedule data in one day with that from another day.
   * 
   * @name niagara.schedule.Schedule#overwrite
   * @function
   * 
   * @param {niagara.fieldeditors.schedule.DayEditor} srcDayEditor the day whose 
   * schedule blocks we want to use to overwrite the recipient day
   * @param {niagara.fieldEditors.schedule.DayEditor} destDayEditor the day 
   * whose schedule data will be overwritten
   */
  Schedule.prototype.overwrite = function overwrite(srcDayEditor, dayName) {
    srcDayEditor.copyTo(this.dayEditors[dayName]);
    
    this.setModified(true);
  };
  
  /**
   * Removes all data from all days.
   * 
   * @name niagara.schedule.Schedule#empty
   * @function
   */
  Schedule.prototype.empty = function empty() {
    baja.iterate(this.dayEditors, function (day) {
      day.empty();
      day.saveValue();
    });
    this.setModified(true);
  };
  
  /**
   * Returns a proper string representation of a StatusValue used in a weekly
   * schedule. If the status is null, returns the lexicon value for "null",
   * otherwise prints a proper string, taking into account any relevant
   * display facets set on the schedule itself.
   * 
   * @name niagara.schedule.Schedule#stringify
   * @function
   * @private
   * @param {baja.Struct} statusValue a baja:StatusValue used in a weekly
   * schedule
   * @param {Object} callbacks an object containing ok/fail callbacks - the
   * string will be passed to the object's ok handler
   */
  Schedule.prototype.stringify = function (statusValue, callbacks) {
    callbacks = callbackify(callbacks);
    
    if (statusValue.getStatus() === baja.Status.nullStatus) {
      return callbacks.ok(baja.lex('baja').get('Status.null'));
    }
    
    var type = statusValue.getType(),
        weeklySchedule = this.value,
        value = statusValue.getValue();
    if (type.is('baja:StatusBoolean')) {
      callbacks.ok(toStatusBooleanString(weeklySchedule, statusValue));
    } else if (type.is('baja:StatusString')) {
      callbacks.ok(statusValue.getValue());
    } else if (type.is('baja:StatusNumeric')) {
      toStatusNumericString(weeklySchedule, statusValue, callbacks);
    } else if (type.is('baja:StatusEnum')) {
      callbacks.ok(getEnumRangeDisplay(value.getOrdinal(), weeklySchedule.get('facets').get('range')));
    } else {
      callbacks.ok("N/A");
    }
  };
  
  /**
   * Decides what the default value should be when creating a new schedule
   * block.
   * <p>
   * If the user has not yet created a new schedule block, then use the
   * default output value from the currently edited Schedule.
   * <p>
   * Otherwise, remember the last value the user entered
   * (<code>niagara.schedule.lastEnteredValue</code> - set in
   * <code>niagara.schedule.ui.editSelectedBlock()</code>) and reuse that.
   * 
   * @name niagara.schedule.Schedule#getNewBlockValue
   * @function
   * 
   * @returns {baja.Struct} the <code>baja:StatusValue</code> to use when
   * creating a new block
   */
  Schedule.prototype.getNewBlockValue = function getNewBlockValue() {
    if (niagara.schedule.lastEnteredValue) {
      return niagara.schedule.lastEnteredValue.newCopy();
    }
    
    var weeklySchedule = this.value,
        defaultValue = weeklySchedule.getDefaultOutput().newCopy(),
        value = defaultValue.getValue(),
        facets = util.slot.getFacets(weeklySchedule),
        min, 
        max, 
        range;
    
    defaultValue.setStatus(baja.Status.ok);
    
    if (defaultValue.getType().is('baja:StatusBoolean')) {
      defaultValue.setValue(!value);
    } else if (defaultValue.getType().is('baja:StatusNumeric')) {
      value = 0;
      if (facets) {
        min = facets.get('min');
        max = facets.get('max');
        if (value < min) {
          value = min;
        } else if (value > max) {
          value = max;
        }
        defaultValue.setValue(value);
      }
    } else if (defaultValue.getType().is('baja:StatusEnum')) {
      if (value.getType().is('baja:DynamicEnum')) {
        
        range = facets && facets.get('range');
        if (range) {
          defaultValue.setValue(baja.DynamicEnum.make({
            range: range
          }));
        }
      }
    }
    return defaultValue;
  };
  
  
  fe.register('schedule:WeeklySchedule', Schedule);
  
  /**
   * @namespace
   * @name niagara.schedule
   */
  util.api('niagara.schedule', {
    'public': {
      Schedule: Schedule
    },
    'private': {
      getEnumRangeDisplay: getEnumRangeDisplay,
      toStatusBooleanString: toStatusBooleanString
    }
  });
}());
