/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/*jslint white: true, vars: true, sloppy: true */
/*global niagara, baja, $ */

(function scheduleFieldEditors() {
  
  niagara.util.require(
    'niagara.fieldEditors.mobile.MobileFieldEditor'
  );
  
  var util = niagara.util,
      aop = util.aop,
      fe = niagara.fieldEditors,
      mobileFE = fe.mobile,
      compositeFE = fe.composite,
      dialogs = niagara.util.mobile.dialogs,
      MobileFieldEditor = fe.mobile.MobileFieldEditor,
      simple = {},
      
      scheduleLex = baja.lex('schedule'),
      mobileLex = baja.lex('mobile'),
      anyMonth = scheduleLex.get({
        key: 'month.anyMonth',
        def: 'Any Month'
      }),
      anyWeekday = scheduleLex.get({
        key: 'weekday.anyWeekday',
        def: 'Any Weekday'
      }),
      anyYear = scheduleLex.get({
        key: 'year.anyYear',
        def: 'Any Year'
      }),
      lastDay = scheduleLex.get({
        key: 'dayofmonth.last_day',
        def: 'Last Day' 
      }),
      last7Days = scheduleLex.get({
        key: 'dayofmonth.last7days',
        def: 'Last 7 Days'
      }),
      through = scheduleLex.get({
        key: 'daterange.through',
        def: 'Through'
      }),
      
      ENUMSET_HTML = 
        '<select data-theme="a" name="enumset_{id}" id="enumset_{id}">' +
          '<option value="-1">' + mobileLex.get('loading') + '</option>' +
        '</select>';
    
  (function simpleFunctions() {
    function getSlotFacets(component, slot) {
      if (component && slot && component.has(slot)) {
        return component.getSlot(slot).getFacets();
      }
    }
    
    /**
     * Populates the select dropdown when loading a single-value EnumSet into
     * a field editor (used for DateSchedule, DateRangeSchedule, and
     * WeekAndDaySchedule).
     * @param editor the field editor to load
     * @param od ordinals and display values used to build up the dropdown
     * @param {baja.EnumSet} enumSet the EnumSet being loaded
     * @private
     * @inner
     * @see getOrdinalsAndDisplays
     */
    function populateSelect(editor, od, enumSet) {
      var select = editor.$dom.find('select').empty(),
          ordinals = od.ordinals,
          displays = od.displays,
          selectedValue;
      
      baja.iterate(ordinals, function (ordinal, i) {
        var option = $('<option/>')
                       .val(ordinal)
                       .text(displays[i]);
        select.append(option);
      });
      
      if (enumSet === baja.EnumSet.NULL) {
        //NULL indicates "Any" selection which is always at the top
        selectedValue = ordinals[0];
      } else {
        selectedValue = enumSet.encodeToString();        
      }
      select.val(selectedValue).trigger('change');
    }
    
    /**
     * Get the ordinals and display values from the EnumSet currently loaded
     * in the given field editor. Will return an object with 'ordinals' and
     * 'displays' array properties. You can splice or push additional values
     * onto these before loading them into a checkbox multiselect dialog.
     * 
     * <p>Optionally, pass in an <code>anyDisplay</code> to prepend an 'any'
     * value to the head of the ordinals/displays.
     * 
     * @param editor the field editor to edit
     * @param [anyDisplay] the display of the 'any' value
     * @returns {Object}
     * @private
     * @inner
     */
    function getOrdinalsAndDisplays(editor, anyDisplay) {
      var slotFacets = getSlotFacets(editor.container, editor.slot),
          range = slotFacets && slotFacets.get('range'),
          ordinals = range.getOrdinals(),
          returnOrdinals = [],
          returnDisplays = [];
      
      if (anyDisplay) {
        returnOrdinals.push('any');
        returnDisplays.push(anyDisplay);
      }
      
      baja.iterate(ordinals, function (ordinal) {
        var display = mobileFE.simple.getEnumRangeDisplay(ordinal, range);
        returnOrdinals.push(ordinal);
        returnDisplays.push(display);
      });
      
      return {
        ordinals: returnOrdinals,
        displays: returnDisplays
      };
    }
    
    function makeSingleEnumSet(val, nullVal) {
      if (val === nullVal) {
        return baja.EnumSet.NULL;
      } else {
        return baja.EnumSet.make([val]);
      }
    }
    
    /**
     * Edit the current value of the given EnumSet field editor using a dialog.
     * The dialog will allow you to select members of the EnumSet individually 
     * using checkboxes.
     * 
     * @param editor the field editor we're editing
     * @param od ordinals and display values to load into the checkboxes
     * @param anyVal the value representing 'any' (will be either 'any' or '0',
     * depending on the EnumSet)
     * @private
     * @inner
     * @see getOrdinalsAndDisplays
     */
    function dialogEdit(editor, od, anyVal) {
      var ordinals = od.ordinals,
          displays = od.displays,
          enumSet = editor.value,
          setOrdinals = enumSet.getOrdinals(),
          div = $('<div data-role="fieldcontain"/>'),
          fieldset = $('<fieldset data-role="controlgroup"/>')
                       .appendTo(div);
      
      baja.iterate(ordinals, function (ordinal, i) {
        var display = displays[i],
            id = "checkbox" + ordinal,
            input = $('<input type="checkbox"/>')
                      .attr('id', id)
                      .attr('value', ordinal),
            label = $('<label/>')
                      .attr('for', id)
                      .text(display);
            
        if (util.contains(setOrdinals, ordinal) ||
            (String(ordinal) === anyVal && setOrdinals.length === 0)) {
          input.attr('checked', 'checked');
        }
        
        fieldset.append(input).append(label);
      });
      
      fieldset.delegate('input', 'change', function () {
        var val = $(this).val();
        if (val === anyVal) {
          fieldset.find('input[value!=' + anyVal + ']').prop('checked', false);
        } else {
          fieldset.find('input[value=' + anyVal + ']').prop('checked', false);
        }
        
        if (fieldset.find('input:checked').length === 0) {
          fieldset.find('input[value=' + anyVal + ']').prop('checked', true);
        }
        
        fieldset.find('input').checkboxradio('refresh');
        editor.setModified(true);
      });
      
      dialogs.okCancel({
        content: function (div, callbacks) {
          div.append(fieldset).trigger('create');
          callbacks.ok();
        },
        title: editor.label,
        ok: function (cb) {
          var selectedOrdinals = [],
              enumSet;
          fieldset.find('input:checked').each(function () {
            var val = $(this).val();
            if (val !== anyVal) {
              selectedOrdinals.push(Number(val));
            }
          });
          
          enumSet = baja.EnumSet.make(selectedOrdinals);
          
          editor.loadValue(enumSet, cb);
        }
      });
      
    }

    /** 
     * Default field editor HTML for an EnumSet.
     * @memberOf niagara.fieldEditors.schedule.simple
     */
    function schedulerEnumSetInitialize(targetElement, callbacks) {
      targetElement.append(ENUMSET_HTML.patternReplace({ id: this.name }));
      callbacks.ok();
    }
    
    /**
     * Create a button that when clicked will allow the editor's EnumSet to be
     * edited using a dialog with multiple checkboxes.
     *  
     * @param editor an editor for an EnumSet inside a CustomSchedule
     * @param div the div in which the editor will be loaded
     * @param od ordinals and display values for the EnumSet
     * @param anyVal the value representing 'any' (either 'any' or '0')
     */
    function doMultiInitialize(editor, div, od, anyVal) {
      var a = $('<a data-role="button" data-theme="a"/>');
      
      editor.label = editor.container.getDisplayName();
      
      a.text(editor.label);
      
      a.click(function () {
        dialogEdit(editor, od, anyVal);
      });
      
      a.appendTo(div);
    }
    
    /**
     * Initializes a multiselect editor for a MonthSchedule in a CustomSchedule.
     */
    function schedulerMonthMultiInitialize(targetElement, callbacks) {
      var od = getOrdinalsAndDisplays(this, anyMonth),
          ordinals = od.ordinals;
      
      ordinals.pop(); //remove "12" - no groups
      ordinals.pop(); //remove "13" - no groups
      
      doMultiInitialize(this, targetElement, od, 'any');
      callbacks.ok();
    }
    
    /**
     * Loads an EnumSet into a dropdown for editing a WeekSchedule inside a 
     * WeekAndDaySchedule.
     */
    function schedulerWeekLoad(enumSet, callbacks) {
      var od = getOrdinalsAndDisplays(this);
      
      populateSelect(this, od, enumSet);
      callbacks.ok();
    }
    
    /**
     * Just calls ok callback (load() doesn't need to do anything on a 
     * multi-select editor).
     */
    function nopLoad(value, callbacks) {
      callbacks.ok();
    }
    
    /**
     * Loads an EnumSet into a dropdown for editing a DayOfMonthSchedule inside
     * a DateSchedule, DateRangeSchedule, or WeekAndDaySchedule.
     */
    function schedulerDayLoad(enumSet, callbacks) {
      var od = getOrdinalsAndDisplays(this);
      
      populateSelect(this, od, enumSet);
      callbacks.ok();
    }
    
    /**
     * Initializes a multiselect editor for a DayOfMonthSchedule inside a
     * CustomSchedule.
     */
    function schedulerDayMultiInitialize(targetElement, callbacks) {
      var od = getOrdinalsAndDisplays(this);
      
      od.ordinals.push(32, 33);
      od.displays.push(lastDay, last7Days);
      doMultiInitialize(this, targetElement, od, '0');
      callbacks.ok();
    }
    
    /**
     * Loads a dropdown for a MonthSchedule inside a DateSchedule or
     * DateRangeSchedule.
     */
    function schedulerMonthLoad(enumSet, callbacks) {
      var od = getOrdinalsAndDisplays(this, anyMonth),
          ordinals = od.ordinals,
          displays = od.displays;
      
      ordinals.pop(); //remove "12" - no groups
      ordinals.pop(); //remove "13" - no groups
      
      populateSelect(this, od, enumSet);
      callbacks.ok();
    }

    /**
     * Loads a dropdown for a MonthSchedule inside a WeekAndDaySchedule
     * (includes month groups).
     */
    function schedulerGroupsMonthLoad(enumSet, callbacks) {
      var od = getOrdinalsAndDisplays(this, anyMonth);
      
      populateSelect(this, od, enumSet);
      callbacks.ok();
    }

    /**
     * Loads a dropdown for a WeekdaySchedule inside a WeekAndDaySchedule.
     */
    function schedulerWeekdayLoad(enumSet, callbacks) {
      var od = getOrdinalsAndDisplays(this, anyWeekday);
      
      populateSelect(this, od, enumSet);
      callbacks.ok();
    }
    
    /**
     * Initializes a multiselect editor for a WeekdaySchedule inside a
     * CustomSchedule.
     */
    function schedulerWeekdayMultiInitialize(targetElement, callbacks) {
      var od = getOrdinalsAndDisplays(this, anyWeekday);
      
      doMultiInitialize(this, targetElement, od, 'any');
      callbacks.ok();
    }
    
    /**
     * Initializes a multiselect editor for a WeekSchedule inside a
     * CustomSchedule.
     */
    function schedulerWeekMultiInitialize(targetElement, callbacks) {
      var od = getOrdinalsAndDisplays(this);
      doMultiInitialize(this, targetElement, od, '0');
      callbacks.ok();
    }
    

    /**
     * Just returns this.value (editing is already done on
     * multiselect editors).
     */
    function nopRead(obj) {
      obj.ok(this.value);
    }
    
    /** Get the value of the editor's dropdown */
    function getSelectValue(editor) {
      return editor.$dom.find('select').val();
    }
    
    /**
     * Reads an EnumSet with a single ordinal, for a MonthSchedule inside a
     * DateSchedule, DateRangeSchedule, or WeekAndDaySchedule. 
     */
    function schedulerMonthRead(obj) {
      var val = getSelectValue(this);
      obj.ok(makeSingleEnumSet(val, 'any'));
    }
    
    /**
     * Reads an EnumSet with a single ordinal, for a DayOfMonthSchedule inside a
     * DateSchedule or DateRangeSchedule.
     */
    function schedulerDayRead(obj) {
      var val = getSelectValue(this);
      obj.ok(makeSingleEnumSet(val, '0'));
    }
    
    /**
     * Reads an EnumSet with a single ordinal, for a WeekdaySchedule inside a
     * WeekAndDaySchedule.
     */
    function schedulerWeekdayRead(obj) {
      var val = getSelectValue(this);
      obj.ok(makeSingleEnumSet(val, 'any'));
    }
    
    /**
     * reads an EnumSet with a single ordinal, for a WeekSchedule inside a
     * WeekAndDaySchedule.
     */
    function schedulerWeekOfMonthRead(obj) {
      var val = getSelectValue(this);
      obj.ok(makeSingleEnumSet(val, '0'));
    }
    
    /**
     * Initializes a dropdown for a YearSchedule inside a DateSchedule,
     * DateRangeSchedule, or CustomSchedule.
     */
    function schedulerYearInitialize(targetElement, callbacks) {
      var currentYear = new Date().getFullYear(),
          select = $('<select data-theme="a" name="year" />');
      
      $('<option />')
        .val('any')
        .text(anyYear)
        .appendTo(select);
      
      baja.iterate(currentYear, currentYear + 10, function (year) {
        $('<option />').val(String(year)).text(String(year)).appendTo(select);
      });
      
      select.appendTo(targetElement);
      
      callbacks.ok();
    }
    
    /**
     * Loads a dropdown for a YearSchedule inside a DateSchedule,
     * DateRangeSchedule, or CustomSchedule.
     */
    function schedulerYearLoad(value, callbacks) {
      var yearSchedule = value,
          select = this.$dom.find('select');
      
      if (yearSchedule.getAlwaysEffective()) {
        select.val('any');
      } else {
        select.val(String(yearSchedule.getYear()));
      }
      
      callbacks.ok();
    }
    
    /**
     * Reads a YearSchedule for a DateSchedule, DateRangeSchedule, or
     * CustomSchedule.
     */
    function schedulerYearRead(obj) {
      var yearSchedule = fe.toSaveDataComponent(this.value),
          val = this.$dom.find('select').val();
      
      if (val === 'any') {
        yearSchedule.setAlwaysEffective(true);
      } else {
        yearSchedule.setAlwaysEffective(false);
        yearSchedule.setYear(Number(val));
      }
      
      obj.ok(yearSchedule);
    }
      
    
    (function registerFEs() {
      /** EnumSet for month slot on a MonthSchedule */
      var EnumSetMonthEditor = fe.defineEditor(MobileFieldEditor, {
        doInitializeDOM: schedulerEnumSetInitialize,
        doLoadValue: schedulerMonthLoad,
        getSaveData: schedulerMonthRead
      });
      fe.register('baja:EnumSet', EnumSetMonthEditor, 'enumset-month');

      
      /** EnumSet for month slot on a CustomSchedule */
      var EnumSetMonthMultiEditor = fe.defineEditor(MobileFieldEditor, {
        doInitializeDOM: schedulerMonthMultiInitialize,
        doLoadValue: nopLoad,
        getSaveData: nopRead
      });
      fe.register('baja:EnumSet', EnumSetMonthMultiEditor, 'enumset-month-multi');

      
      /** EnumSet for month slot on a MonthSchedule in a WeekAndDaySchedule, includes groups */
      var EnumSetGroupsMonthEditor = fe.defineEditor(MobileFieldEditor, {
        doInitializeDOM: schedulerEnumSetInitialize,
        doLoadValue: schedulerGroupsMonthLoad,
        getSaveData: schedulerMonthRead
      });
      fe.register('baja:EnumSet', EnumSetGroupsMonthEditor, 'enumset-month-groups');

      
      /** EnumSet for a DayOfMonthSchedule */
      var EnumSetDayEditor = fe.defineEditor(MobileFieldEditor, {
        doInitializeDOM: schedulerEnumSetInitialize,
        doLoadValue: schedulerDayLoad,
        getSaveData: schedulerDayRead
      });
      fe.register('baja:EnumSet', EnumSetDayEditor, 'enumset-day');

      
      /** EnumSet for day slot on a DayOfMonthSchedule in a CustomSchedule */
      var EnumSetCustomDayEditor = fe.defineEditor(MobileFieldEditor, {
        doInitializeDOM: schedulerDayMultiInitialize,
        doLoadValue: nopLoad,
        getSaveData: nopRead
      });
      fe.register('baja:EnumSet', EnumSetCustomDayEditor, 'enumset-day-custom');

      /** EnumSet for week slot on a WeekOfMonthSchedule on a WeekAndDaySchedule */
      var EnumSetWeekEditor = fe.defineEditor(MobileFieldEditor, {
        doInitializeDOM: schedulerEnumSetInitialize,
        doLoadValue: schedulerWeekLoad,
        getSaveData: schedulerWeekOfMonthRead
      });
      fe.register('baja:EnumSet', EnumSetWeekEditor, 'enumset-weekofmonth');
      
      /** EnumSet for week slot on a WeekOfMonthSchedule on a CustomSchedule */
      var EnumSetWeekMultiEditor = fe.defineEditor(MobileFieldEditor, {
        doInitializeDOM: schedulerWeekMultiInitialize,
        doLoadValue: nopLoad,
        getSaveData: nopRead
      });
      fe.register('baja:EnumSet', EnumSetWeekMultiEditor, 'enumset-weekofmonth-multi');


      /** EnumSet for weekday slot on a WeekdaySchedule, except for CustomSchedules */
      var EnumSetWeekdayEditor = fe.defineEditor(MobileFieldEditor, {
        doInitializeDOM: schedulerEnumSetInitialize,
        doLoadValue: schedulerWeekdayLoad,
        getSaveData: schedulerWeekdayRead
      });
      fe.register('baja:EnumSet', EnumSetWeekdayEditor, 'enumset-weekday');
      
      
      /** EnumSet for weekday slot on a WeekdaySchedule in a CustomSchedules */
      var EnumSetWeekdayMultiEditor = fe.defineEditor(MobileFieldEditor, {
        doInitializeDOM: schedulerWeekdayMultiInitialize,
        doLoadValue: nopLoad,
        getSaveData: nopRead
      });
      fe.register('baja:EnumSet', EnumSetWeekdayMultiEditor, 'enumset-weekday-multi');
      

      /** YearSchedule editor, same everywhere */
      var YearScheduleEditor = fe.defineEditor(MobileFieldEditor, {
        doInitializeDOM: schedulerYearInitialize,
        doLoadValue: schedulerYearLoad,
        getSaveData: schedulerYearRead
      });
      fe.register('schedule:YearSchedule', YearScheduleEditor);

      
      
      
      
      
      /** MonthSchedule for DateSchedule, DateRangeSchedule  */
      var MonthScheduleEditor = compositeFE.makeComposite([
        { slot: 'set', key: 'enumset-month' }
      ]);
      fe.register('schedule:MonthSchedule', MonthScheduleEditor);
      

      /** MonthSchedule for CustomSchedule */
      var MonthScheduleMultiEditor = compositeFE.makeComposite([
        { slot: 'set', key: 'enumset-month-multi' }
      ]);
      fe.register('schedule:MonthSchedule', MonthScheduleMultiEditor, 
          'month-multi');
      
      
      /** MonthSchedule for WeekAndDaySchedule */
      var MonthGroupsScheduleEditor = compositeFE.makeComposite([
        { slot: 'set', key: 'enumset-month-groups' }
      ]);
      fe.register('schedule:MonthSchedule', MonthGroupsScheduleEditor, 
          'month-groups');
      
      
      /** DayOfMonthSchedule for DateSchedule, DateRangeSchedule */
      var DayOfMonthScheduleEditor = compositeFE.makeComposite([
        { slot: 'set', key: 'enumset-day' }
      ]);
      fe.register('schedule:DayOfMonthSchedule', DayOfMonthScheduleEditor);
      
      
      /** DayOfMonthSchedule for CustomSchedule */
      var CustomDayOfMonthScheduleEditor = compositeFE.makeComposite([
        { slot: 'set', key: 'enumset-day-custom' }
      ]);
      fe.register('schedule:DayOfMonthSchedule', 
          CustomDayOfMonthScheduleEditor, 'day-custom');
      
      
      /** WeekOfMonthSchedule for WeekAndDaySchedule */
      var WeekOfMonthScheduleEditor = compositeFE.makeComposite([
        { slot: 'set', key: 'enumset-weekofmonth' }
      ]);
      fe.register('schedule:WeekOfMonthSchedule', WeekOfMonthScheduleEditor);
      
      
      /** WeekOfMonthSchedule for CustomSchedule */
      var WeekOfMonthScheduleMultiEditor = compositeFE.makeComposite([
        { slot: 'set', key: 'enumset-weekofmonth-multi' }
      ]);
      fe.register('schedule:WeekOfMonthSchedule', WeekOfMonthScheduleMultiEditor, 
          'week-multi');
      
      
      /** WeekdaySchedule for DateSchedule, WeekAndDaySchedule */
      var WeekdayScheduleEditor = compositeFE.makeComposite([
        { slot: 'set', key: 'enumset-weekday' }
      ]);
      fe.register('schedule:WeekdaySchedule', WeekdayScheduleEditor);
      
      
      /** WeekdaySchedule for CustomSchedule */
      var WeekdayScheduleMultiEditor = compositeFE.makeComposite([
        { slot: 'set', key: 'enumset-weekday-multi' }
      ]);
      fe.register('schedule:WeekdaySchedule', WeekdayScheduleMultiEditor, 'weekday-multi');



      
      
      /** DateSchedule */
      var DateScheduleEditor = compositeFE.makeComposite([
        'weekdaySchedule', 'daySchedule', 'monthSchedule', 'yearSchedule'
      ]);
      
      /** DateSchedule: start/end field of a DateRangeSchedule */
      var DateScheduleNoWeekdaysEditor = compositeFE.makeComposite([
        'daySchedule', 'monthSchedule', 'yearSchedule'
      ]);
      
      /** DateRangeSchedule */
      var DateRangeScheduleEditor = compositeFE.makeComposite([ 
        { slot: 'start', key: 'noweekdays' }, 
        { slot: 'end', key: 'noweekdays' } 
      ], MobileFieldEditor);
      
      /** WeekAndDaySchedule */
      var WeekAndDayScheduleEditor = compositeFE.makeComposite([
        'weekdays', 'weeksOfMonth', { slot: 'months', key: 'month-groups' }  
      ]);
      
      /** TimeSchedule (for schedule blocks in day editor) */
      var TimeScheduleEditor = compositeFE.makeComposite([
        'start', 'finish', 'effectiveValue'
      ]);
      
      /** CustomSchedule */
      var CustomScheduleEditor = compositeFE.makeComposite([
        { slot: 'daysOfMonth', key: 'day-custom' }, 
        { slot: 'months', key: 'month-multi' },
        { slot: 'weekdays', key: 'weekday-multi' },
        { slot: 'weeksOfMonth', key: 'week-multi' }, 
        'year'
      ]);
      
      /** ScheduleReference */
      var ScheduleReferenceEditor = compositeFE.makeComposite([
        {
          slot: 'ref',
          key: 'componentSelector',
          defaultValue: baja.Ord.make('station:|slot:/|bql: ' +
              'select displayName, slotPathOrd from schedule:CalendarSchedule ' +
              'where parent.type != schedule:AbstractSchedule')
        }
      ]);
      
      aop.before(DateScheduleNoWeekdaysEditor.prototype, 'doInitializeDOM', function (args) {
        var targetElement = args[0],
            callbacks = args[1];
        aop.before(callbacks, 'ok', function () {
          targetElement.wrapInner('<fieldset data-role="controlgroup"/>');
        });
      });
      
      aop.before(DateRangeScheduleEditor.prototype, 'doInitializeDOM', function (args) {
        var targetElement = args[0],
            callbacks = args[1];
        aop.before(callbacks, 'ok', function () {
          targetElement
            .children('div.subEditor').eq(0)
            .after($('<div class="through"/>').text(through));
        });
      });
      
      
      
      
      fe.register('schedule:DateSchedule', DateScheduleEditor);
      fe.register('schedule:DateSchedule', DateScheduleNoWeekdaysEditor, 
          'noweekdays');
      fe.register('schedule:DateRangeSchedule', DateRangeScheduleEditor);
      
      fe.register('schedule:WeekAndDaySchedule', WeekAndDayScheduleEditor);
      fe.register("schedule:TimeSchedule", TimeScheduleEditor);
      fe.register("schedule:CustomSchedule", CustomScheduleEditor);
      fe.register("schedule:ScheduleReference", ScheduleReferenceEditor);
    }());
  }());
    
  /**
   * @namespace
   * @name niagara.fieldEditors.schedule
   */
  util.api('niagara.fieldEditors.schedule', {
    simple: simple
  });
}());