//
// Copyright 2010, Tridium, Inc. All Rights Reserved.
//

/**
 * @fileOverview Bindings for the Px App.
 *
 * @author Gareth Johnson
 * @version 0.0.1.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: false, nomen: false, eqeqeq: true, bitwise: true, regexp: true, newcap: true, immed: true, strict: false, indent: 2, vars: true, continue: true */

/*global $, baja, BaseBajaObj, window, niagara*/ 
  
(function pxMobileBindings(baja) {
  // Use ECMAScript 5 Strict Mode
  "use strict";
          
  niagara.util.require(
    "$.mobile",
    "niagara.fieldEditors",
    "niagara.util.flow",
    "niagara.util.mobile.dialogs",
    "niagara.util.mobile.commands",
    "niagara.util.px",
    "niagara.util.px.mobile",
    "niagara.util.mobile.pages"
  );
  
  var fieldEditors = niagara.fieldEditors,
      util = niagara.util,
      dialogs = util.mobile.dialogs,
      commands = util.mobile.commands,
      pxUtil = util.px,
      pxUtilMobile = util.px.mobile,
      pages = util.mobile.pages;
      
////////////////////////////////////////////////////////////////
// Bindings
////////////////////////////////////////////////////////////////    
  
  /**
   * @class Binding.
   * <p>
   * A Component that represents a Niagara 'bajaui:Binding' Type.
   * This Component's ORD is resolved and used to dynamically update a Widget
   * when a Px page is loaded.
   *
   * @extends baja.Component
   */
  var Binding = function() {
    Binding.$super.apply(this, arguments);
    this.$target = null;
  }.$extend(baja.Component).registerType("bajaui:Binding");
  
  /**
   * Return the parent Widget.
   */
  Binding.prototype.getWidget = function() {
    return this.getParent();
  };
  
  /**
   * Return the ORD Target for this binding (null if not bound).
   */
  Binding.prototype.getTarget = function() {
    return this.$target;
  };
    
  /**
   * Called when the binding is first loaded.
   */
  Binding.prototype.load = function() {
    // Widget Degradation
    if (!this.isBound()) {
      var widget = this.getWidget();
    
      var behavior = this.getDegradeBehavior();
      if (behavior.is("disable")) {
        widget.setEnabled(false);
        widget.update(function (propName) {
          return propName === "enabled";
        });
      }
      else if (behavior.is("hide")) {
        widget.setVisible(false);
        widget.update(function (propName) {
          return propName === "visible";
        });
      }
    }
    
    // Call super load handlers
    this.doLoad.apply(this, arguments);
  };
  
  /**
   * Called when the binding is first loaded and is designed to be overriden
   */
  Binding.prototype.doLoad = function() {
  };
    
  /**
   * Called when the binding is updated.
   */
  Binding.prototype.update = function() {  
    var binding = this,
        widget = binding.getWidget();
    
    // Lazily create closure function for hasUpdate only once    
    if (!this.$widgetHasUpdate) {
      this.$widgetHasUpdate = function(propName) {
        var slot = widget.getSlot(propName);
        if (slot === null) {
          return false;
        }
        return binding.isBound() && binding.isOverridden(slot);
      };
    }    
        
    widget.update(this.$widgetHasUpdate);
  };
  
  /**
   * Internal Framework Method
   *
   * @private
   */
  Binding.prototype.$fw = function(x, a, b, c, d) {
    if (x === "updateBinding") {
      this.$target = a; // Update the ORD Target for the binding
      return;
    }    
    return Binding.$super.prototype.$fw.apply(this, arguments);
  };
  
  /**
   * Return the String for a binding.
   *
   * @return {String}
   */
  Binding.prototype.toString = function() {
    // Return the ORD for the binding (this is used when the page is loaded).
    return this.getOrd().toString();
  };
  
  /**
   * Return true if the Binding has a valid ORD Target.
   *
   * @return {Boolean}
   */
  Binding.prototype.isBound = function() {
    return this.$target ? true : false;
  };
  
  /**
   * Return the override value for the given Property.
   *
   * @param {baja.Property} the Property to try and override
   * @return {baja.Value} the overriden value.
   */
  Binding.prototype.getOnWidget = function(prop) {
    return null;
  };
  
  /**
   * Return true if the Property is overridden.
   *
   * @param {baja.Property} prop the Property to test.
   * @return {Boolean} true if overridden.
   */
  Binding.prototype.isOverridden = function(prop) {
    return false;
  };
    
  /**
   * Handle an event from the Widget.
   *
   * @param {String} eventName.
   * @return {Boolean} return true if this binding has handled the event.
   */
  Binding.prototype.handleEvent = function(eventName) {
    return false;
  };
      
  /**
   * @class ValueBinding.
   * <p>
   * A Component that represents a Niagara 'bajaui:ValueBinding' Type.
   *
   * @name ValueBinding
   * @extends Binding
   */
  var ValueBinding = function() {
    ValueBinding.$super.apply(this, arguments);
  }.$extend(Binding).registerType("bajaui:ValueBinding");
  
  /**
   * Return true if the Property is overridden.
   *
   * @param {baja.Property} prop the Property to test.
   * @return {Boolean} return true if the Property is overridden.
   */
  ValueBinding.prototype.isOverridden = function(prop) {
    var override = this.get(prop);
    
    // If we have a converter then the Property is overriden
    return override !== null && override.getType().is("baja:Converter");
  };
  
  /**
   * Return any override display String for the given Property.
   *
   * @param {baja.Property} the Widget's Property to return the overridden value for.
   * @return {String} the overridden value.
   */
  ValueBinding.prototype.getOnWidget = function(prop) {
    var override = this.get(prop);
    
    if (override !== null && override.getType().is("baja:Converter")) {
      return override.convert(this.$target.getObject(),  // from
                              prop.getDefaultValue(),    // to
                              this.$target);             // target
    }
    
    return override;
  };
  
  /**
   * Called when the binding is first loaded.
   */
  ValueBinding.prototype.doLoad = function() {
    if (this.getHyperlink() !== baja.Ord.DEFAULT) {
      this.getWidget().addMouseOver();
    }
  };
    
  /**
   * Handle an event from the Widget.
   *
   * @param {String} eventName.
   * @return {Boolean} return true if this binding has handled the event.
   */
  ValueBinding.prototype.handleEvent = function(eventName) {
    if (eventName !== "click") {
      return false;
    }
    
    // If specified, then let the hyperlink take precedence
    if (this.getHyperlink() !== baja.Ord.DEFAULT) {
      // Hyperlink
      return pxUtilMobile.hyperlink(this.getWidget(), this.getHyperlink());
    }
    
    // Action pop up
    if (this.isBound() && this.getPopupEnabled()) {
    
      var comp = this.$target.getComponent();
      if (comp) {
        var hasActions = false;
        comp.getSlots(function (slot) {
          return niagara.util.slot.isFireable(slot);
        }).each(function (slot) {
          hasActions = true;
          return true;
        });
      
        if (hasActions) {
          // Show available Actions
          commands.getAvailableActionsCmd().invoke(comp);   
          return true;
        }
      }
    }
    
    return false;
  };
  
  /**
   * @class PopupBinding.
   * <p>
   * A Component that represents a Niagara 'kitPx:PopupBinding' Type.
   *
   * @name PopupBinding
   * @extends Binding
   */
  var PopupBinding = function() {
    PopupBinding.$super.apply(this, arguments);
  }.$extend(Binding).registerType("kitPx:PopupBinding");
  
  /**
   * Handle an event from the Widget.
   *
   * @param {String} eventName.
   * @return {Boolean} return true if this binding has handled the event.
   */
  PopupBinding.prototype.handleEvent = function(eventName) {
    if (eventName === "click") {
      var ord = this.getOrd();
      if (ord !== baja.Ord.DEFAULT) {
        return pxUtilMobile.hyperlink(this.getWidget(), ord);
      }
    }
    return false;
  };
  
  /**
   * Called when the binding is first loaded.
   */
  PopupBinding.prototype.doLoad = function() {
    if (this.getOrd() !== baja.Ord.DEFAULT) {
      this.getWidget().addMouseOver();
    }
  };
  
  /**
   * @class BoundLabelBinding.
   * <p>
   * A Component that represents a Niagara 'bajaui:BoundLabelBinding' Type.
   *
   * @name BoundLabelBinding
   * @extends ValueBinding
   */
  var BoundLabelBinding = function() {
    BoundLabelBinding.$super.apply(this, arguments);
  }.$extend(ValueBinding).registerType("kitPx:BoundLabelBinding");
  
  /**
   * Return true if the Property is overridden.
   *
   * @param {baja.Property} prop the Property to test.
   * @return {Boolean} return true if the Property is overridden.
   */
  BoundLabelBinding.prototype.isOverridden = function(prop) {
    // Call super method
    var isOver = BoundLabelBinding.$super.prototype.isOverridden.apply(this, arguments);
    
    if (this.isBound() && !isOver && !this.getStatusEffect().is("none")) {
      var val = this.$target.getObject();  
      
      // Value must implement BIStatus      
      if (val.getType().is("baja:IStatus")) { 
        var nm = prop.getName();
        if (nm === "blink" || nm === "foreground" || nm === "background") {
          isOver = true;
        }
      }
    }
    
    return isOver;
  };
  
  /**
   * Return any override display String for the given Property.
   *
   * @param {baja.Property|String} the Widget's Property to return the overridden value for.
   * @return {String} the overridden value.
   */
  BoundLabelBinding.prototype.getOnWidget = function(prop) {
    var effect = this.getStatusEffect();
    var val = this.$target.getObject();
  
    if (prop && this.isBound() && val.getType().is("baja:IStatus") && !effect.is("none")) {
      var nm = prop.getName();
      var status = baja.Status.getStatusFromIStatus(val);
      
      if (nm === "blink") {
        if (status.isUnackedAlarm() && effect.is("colorAndBlink")) {
          return true;
        }
      }
      else if (nm === "foreground") {
        var fg = pxUtil.getStatusBrush({
          status: status,
          isForeground: true
        });
        
        if (fg !== null) {
          return fg;
        }
      }
      else if (nm === "background") {
        var bg = pxUtil.getStatusBrush({
          status: status,
          isForeground: false
        });
        
        if (bg !== null) {
          return bg;
        }
      }
    }
    
    // Call super method
    return BoundLabelBinding.$super.prototype.getOnWidget.apply(this, arguments);
  };
      
  /**
   * @class SetPointBinding.
   * <p>
   * A Component that represents a Niagara 'kitPx:SetPointBinding' Type.
   *
   * @name SetPointBinding
   * @extends Binding
   */
  var SetPointBinding = function() {
    SetPointBinding.$super.apply(this, arguments);
  }.$extend(ValueBinding).registerType("kitPx:SetPointBinding");
  
  /**
   * Save the Widget Property driving the target
   *
   * @return {baja.Value} the value for the Widget Property (or null if there's no value available);
   */
  SetPointBinding.prototype.saveWidgetProperty = function() {
    var widget = this.getWidget();
    var prop = widget.getSlot(this.getWidgetProperty());
    if (prop === null) {
      return null;
    }     
    return widget.get(prop);
  };
  
  function convertSetPointValue(from, to) {
    // If the same then easy as pie
    if (from.getType().equals(to.getType())) {
      return from;
    }
    
    // ------- Numeric -------
    
    // INumeric -> Numeric
    var x;
    if (from.getType().is("baja:INumeric") && to.getType().isNumber()) {
      x = Number.getNumberFromINumeric(from);
      switch (to.getDataTypeSymbol()) {
      case "i":
        return baja.Integer.make(x);
      case "l":
        return baja.Long.make(x);        
      case "f":
        return baja.Float.make(x);
      case "d":
        return x;
      }
    }
    
    // INumeric -> StatusNumeric
    if (from.getType().is("baja:INumeric") && to.getType().is("baja:StatusNumeric")) {
      var statusNumeric = baja.$("baja:StatusNumeric");
      statusNumeric.setValue(Number.getNumberFromINumeric(from));
      return statusNumeric;
    }
    
    // ------- Boolean -------
    
    // IBoolean -> Boolean
    if (from.getType().is("baja:IBoolean") && to.getType().is("baja:Boolean")) {
      return Boolean.getBooleanFromIBoolean(from);
    }
    
    // IBoolean -> StatusBoolean
    if (from.getType().is("baja:IBoolean") && to.getType().is("baja:StatusBoolean")) {
      var statusBoolean = baja.$("baja:StatusBoolean");
      statusBoolean.setValue(Boolean.getBooleanFromIBoolean(from));
      return statusBoolean;
    }
    
    // ------- Enum -------
    
    // IEnum -> DynamicEnum
    if (from.getType().is("baja:IEnum") && to.getType().is("baja:DynamicEnum")) {
      return baja.Enum.getEnumFromIEnum(from);
    }
    
    // IEnum -> StatusEnum
    if (from.getType().is("baja:IEnum") && to.getType().is("baja:StatusEnum")) {
      var statusEnum = baja.$("baja:StatusEnum");
      statusEnum.setValue(baja.Enum.getEnumFromIEnum(from));
      return statusEnum;
    }
    
    // ------- String -------
    
    // String -> StatusString
    if (from.getType().is("baja:String") && to.getType().is("baja:StatusString")) {
      var statusString = baja.$("baja:StatusString");
      statusString.setValue(from.toString());
      return statusString;
    } 
    
    baja.error("WARNING: No mechanism found to save");
    return null;
  }
  
  function verifySetPointBounds(val, facets) {
    if (!val.getType().isNumber() || facets === null) {
      return;
    }
    
    // Just incase this is a boxed type, get the inner value
    val = val.valueOf();
        
    var min = facets.get("min");
    if (min !== null) {
      min = min.valueOf();
    }
    
    var max = facets.get("max");
    if (max !== null) {
      max = max.valueOf();
    }
    
    // Check the bounds
    if (min !== null && val < min) {
      throw new Error();
    }
    if (max !== null && val > max) {
      throw new Error();
    }
  }
 
  function saveSetPointAction(obj) {
    // TODO: Need to enforce facets here too
  
    var c = obj.binding.$target.getComponent();
    var action = c.getSlot("set");
    if (!action || (action && !action.isAction())) {
      return false;
    }

    if (!action.getParamType().equals(obj.value.getType())) {
      return false;
    }
    
    // Invoke the Action (please note, this performs an asynchronous network call)
    c.invoke({
      slot: action,
      value: obj.value,
      ok: obj.ok,
      fail: obj.fail,
      batch: obj.batch
    });
    
    return true;
  }  
  
  function saveSetPointProperty(obj) {
    // Get the PropertyPath
    var c = obj.binding.$target.getComponent();
    var path = obj.binding.$target.propertyPath;
    if (!path || (path && path.length === 0)) {
      return false;
    }
    
    // Make sure the Property is not readonly
    if ((baja.Flags.READONLY & c.getFlags(path[0])) === baja.Flags.READONLY) {
      return false;
    }
    
    // Find appropriate Property on target
    var parent = c;
    baja.iterate(path, 0, path.length - 1, function (p) {
      parent = parent.get(p);
    });
    
    // Check min/max
    verifySetPointBounds(obj.value, c.getFacets(path[0]));
    
    // Make set (please note, this performs an asynchronous network call)
    parent.set({
      slot: path[path.length - 1],
      value: obj.value,
      ok: obj.ok,
      fail: obj.fail,
      batch: obj.batch
    });
    
    return true;
  }  
  
  /**
   * Save the Set Point Binding
   *
   * @param {Object} obj the Object Literal for the method's arguments.
   * @param obj.value the value that will be saved.
   */
  SetPointBinding.prototype.saveSetPoint = function(obj) {
    obj.ok = obj.ok || baja.ok;
    obj.fail = obj.fail || baja.fail;
  
    if (!this.isBound()) {
      obj.ok();
      return;
    }
    
    var targetVal = this.$target.getObject(), arg;
    
    // If we are bound to a property under a Component,
    // then convert widget -> target and save Property
    if (!targetVal.getType().isComponent()) {
      var newVal = convertSetPointValue(obj.value, targetVal); 
      
      if (newVal === null) {
        obj.ok();
        return;
      }
      
      arg = {
        binding: this, 
        value: newVal, 
        ok: obj.ok, 
        fail: obj.fail,
        batch: obj.batch
      };
      
      // Try to save value as Property
      if (saveSetPointProperty(arg)) {
        return;
      }
    }
    
    arg = {
      binding: this, 
      value: obj.value, 
      ok: obj.ok, 
      fail: obj.fail,
      batch: obj.batch
    };
    
    if (saveSetPointAction(arg)) {
      return;
    }
    
    // TODO: Implement save Action
    obj.fail("Could not save");
  };
  
  /**
   * Handle an event from the Widget.
   *
   * @param {String} eventName.
   * @return {Boolean} return true if this binding has handled the event.
   */
  SetPointBinding.prototype.handleEvent = function (eventName) {
    // Please note, this binding overrides the default click behavior of
    // ValueBinding
    if (this.getWidgetEvent() === "actionPerformed") {
      try {
        var value = this.saveWidgetProperty();
        if (value === null) {
          return;
        }
      
        this.saveSetPoint({
          value: value
        });
      }
      catch (err) {
        dialogs.error(err);
      }
    
      // Save set point value
      return true;
    }
    return false;
  };
      
  /**
   * Called when the binding is updated.
   */
  SetPointBinding.prototype.update = function() {
    // Load the set point value
    var widget = this.getWidget();
    
    if (this.isBound()) {
      // handle SetPointFieldEditor special
      if (widget.getType().is("kitPx:SetPointFieldEditor")) { 
        widget.loadSetPoint(this);
      }
      else {                  
        var prop = widget.getSlot(this.getWidgetProperty());
                
        if (prop !== null) {   
          // Get the current Widget value
          var widgetVal = widget.get(prop);
          
          // Get target value
          var targetVal = this.$target.getObject(); 
          
          // convert target value
          var newVal = convertSetPointValue(targetVal, widgetVal);
          
          // If we have a solution then update the Widget Property
          if (newVal !== null) {
            widget.set({
              slot: prop,
              value: newVal
            });
          }
        }
      }
    }

    // Call super
    SetPointBinding.$super.prototype.update.apply(this, arguments);
  };
  
  /**
   * Return true if the Property is overridden.
   *
   * @param {baja.Property} prop the Property to test.
   * @return {Boolean} return true if the Property is overridden.
   */
  SetPointBinding.prototype.isOverridden = function(prop) {
    if (prop.getName() === this.getWidgetProperty()) {
      return true;
    }
  
    // Call super method
    return SetPointBinding.$super.prototype.isOverridden.apply(this, arguments);
  };
  
  /**
   * @class Increment Set Point Binding.
   * <p>
   * A Component that represents a Niagara 'kitPx:IncrementSetPointBinding' Type.
   *
   * @name IncrementSetPointBinding
   * @extends SetPointBinding
   */
  var IncrementSetPointBinding = function() {
    IncrementSetPointBinding.$super.apply(this, arguments);
  }.$extend(SetPointBinding).registerType("kitPx:IncrementSetPointBinding");
  
  /**
   * Save the Widget Property driving the target
   *
   * @return {baja.Value} the value for the Widget Property (or null if there's no value available);
   */
  IncrementSetPointBinding.prototype.saveWidgetProperty = function() {
    if (!this.isBound()) {
      return null;
    }
    var val = this.$target.getObject();
    if (!val.getType().is("baja:INumeric")) {
      return null;
    }
    return Number.getNumberFromINumeric(val) + this.getIncrement();
  };
  
  /**
   * @class SpectrumBinding.
   * <p>
   * A Component that represents a Niagara 'kitPx:SpectrumBinding' Type.
   *
   * @name SpectrumBinding
   * @extends Binding
   */
  var SpectrumBinding = function() {
    SpectrumBinding.$super.apply(this, arguments);
  }.$extend(Binding).registerType("kitPx:SpectrumBinding");
  
  /**
   * Return true if the Property is overridden.
   *
   * @param {baja.Property} prop the Property to test.
   * @return {Boolean} return true if the Property is overridden.
   */
  SpectrumBinding.prototype.isOverridden = function(prop) {
    if (prop.getName() === this.getWidgetProperty()) {
      return true;
    }
  
    // Call super method
    return SpectrumBinding.$super.prototype.isOverridden.apply(this, arguments);
  };
  
  var solveSpectrumColor = function(value) {
    var red, blue, green, alpha, mRed, bRed, mGreen, bGreen, mBlue, bBlue, mAlpha, bAlpha,
        mid = this.getSetpoint(),
        delta = this.getExtent() / 2.0,
        lowColor = this.getLowColor(),
        midColor = this.getMidColor(),
        highColor = this.getHighColor();

    // solve for the color using a linear equation y = mx + b,
    // the y axis is the value being monitored, and the x
    // axis is the color (red, green, or blue)
    if (value < mid) {
      // handle current over boundary
      if (value < mid - delta) {
        return lowColor;
      }

      mRed = (midColor.getRed() - lowColor.getRed()) / delta;
      bRed = midColor.getRed() - mRed * mid;
      red = mRed * value + bRed;

      mGreen = (midColor.getGreen() - lowColor.getGreen()) / delta;
      bGreen = midColor.getGreen() - mGreen * mid;
      green = mGreen * value + bGreen;

      mBlue = (midColor.getBlue() - lowColor.getBlue()) / delta;
      bBlue = midColor.getBlue() - mBlue * mid;
      blue = mBlue * value + bBlue;

      mAlpha = (midColor.getAlpha() - lowColor.getAlpha()) / delta;
      bAlpha = midColor.getAlpha() - mAlpha * mid;
      alpha = mAlpha * value + bAlpha;
    }
    else {
      // handle current over boundary
      if (value > mid + delta) {
        return highColor;
      }

      mRed = (highColor.getRed() - midColor.getRed()) / delta;
      bRed = midColor.getRed() - mRed * mid;
      red = mRed * value + bRed;

      mGreen = (highColor.getGreen() - midColor.getGreen()) / delta;
      bGreen = midColor.getGreen() - mGreen * mid;
      green = mGreen * value + bGreen;

      mBlue = (highColor.getBlue() - midColor.getBlue()) / delta;
      bBlue = midColor.getBlue() - mBlue * mid;
      blue = mBlue * value + bBlue;

      mAlpha = (highColor.getAlpha() - midColor.getAlpha()) / delta;
      bAlpha = midColor.getAlpha() - mAlpha * mid;
      alpha = mAlpha * value + bAlpha;
    }

    return niagara.px.Color.make({red: red, green: green, blue: blue, alpha: alpha});
  };
  
  /**
   * Called when the binding is updated.
   */
  SpectrumBinding.prototype.update = function() {
    // Load the set point value
    var widget = this.getWidget();
    
    if (!this.isBound()) {
      return;
    }
    
    var prop = widget.getSlot(this.getWidgetProperty());
    if (prop !== null) {
      // Get the current Widget value
      var widgetVal = widget.get(prop);
      
      // Get target value
      var targetVal = this.$target.getObject(); 
      
      // convert target value
      var newVal = Number.getNumberFromINumeric(targetVal);
      
      // Solve the color
      var color = solveSpectrumColor.call(this, newVal);
      
      if (prop.getType().is("gx:Brush")) {
        widget.set({
          slot: prop,
          value: niagara.px.Brush.make({ color: color })
        });
      }
      
      if (prop.getType().is("gx:Color")) {
        widget.set({
          slot: prop,
          value: color
        });
      }
    }
   
    // Call super
    SpectrumBinding.$super.prototype.update.apply(this, arguments);
  };
  
  /**
   * @class SpectrumSetpointBinding.
   * <p>
   * A Component that represents a Niagara 'kitPx:SpectrumSetpointBinding' Type.
   *
   * @name SpectrumSetpointBinding
   * @extends ValueBinding
   */
  var SpectrumSetpointBinding = function() {
    SpectrumSetpointBinding.$super.apply(this, arguments);
  }.$extend(ValueBinding).registerType("kitPx:SpectrumSetpointBinding");
  
  /**
   * Called when the binding is updated.
   */
  SpectrumSetpointBinding.prototype.update = function() {
    if (this.isBound()) {
      var targetVal = this.$target.getObject();
      if (targetVal.getType().is("baja:INumeric")) {
        var newVal = Number.getNumberFromINumeric(targetVal);
        var widget = this.getWidget();
        
        // Find all SpectrumBindings and update the setpoint Property with the corresponding Property
        widget.getSlots(function (slot) {
          return slot.isProperty() && slot.getType().is("kitPx:SpectrumBinding");
        }).each(function (slot) {
          var spectrumBinding = this.get(slot);
          
          // Update the value on the SpectrumBinding
          spectrumBinding.setSetpoint(newVal);
          
          // Call update to ensure this Binding updates the UI with the new value
          spectrumBinding.update();
        });
      }
    }
   
    // Call super
    SpectrumBinding.$super.prototype.update.apply(this, arguments);
  };
      
  /**
   * @class WbFieldEditorBinding.
   * <p>
   * A Component that represents a Niagara 'workbench:WbFieldEditorBinding' Type.
   */
  var WbFieldEditorBinding = function() {
    WbFieldEditorBinding.$super.apply(this, arguments);
  }.$extend(Binding).registerType("workbench:WbFieldEditorBinding");
  
  var doInitializeFieldEditorWorkflow = util.flow.sequential(
    function (cx) {
      fieldEditors.makeFor(cx.editorParams, this);
    },
    function (cx, editor) {
      editor.initializeDOM(cx.targetElement, this);
    }
  );
  
  /**
   * Called when the Widget is loaded.
   */
  WbFieldEditorBinding.prototype.doLoad = function() {
    if (!this.isBound()) {
      return;
    }
            
    var widget = this.getWidget(), 
        that = this, 
        target = this.$target,
        propertyPath = target.propertyPath,
        fe,
        val,
        slot, 
        container,
        parent;
    
    function loadedFieldEditor() {           
      // Set the field editor to readonly if need be
//      widget.$fieldEditor.setReadonly(!widget.isEnabled());
           
      // Update the Widget now the field editor has loaded           
      that.update();
    }
        
    if (widget && widget.getType().is("kitPx:GenericFieldEditor")) {
      
      // If the field editor is already loaded then we don't need to build any of the html again
      if (widget.$fieldEditor) {
        loadedFieldEditor();
        return;
      }
    
      // Find the container and slot
      parent = target.getComponent();      
      baja.iterate(propertyPath, 0, propertyPath.length - 1, function (prop) {
        parent = parent.get(prop);
      });       
      
      slot = propertyPath[propertyPath.length - 1];
      val = parent.get(slot);
            
      doInitializeFieldEditorWorkflow.invoke({
        editorParams: {
          value: val,
          container: parent,
          slot: slot,
          readonly: !widget.isEnabled()
        },
        targetElement: widget.getDomElement()
      }, {
        ok: function (fe) {
          widget.$fieldEditor = fe;

          loadedFieldEditor();
        },
        fail: function (err) {
          baja.error("Failed to create field editor: " + err);
        }
      });
    }
  };
  
  /**
   * Called when the binding is updated.
   */
  WbFieldEditorBinding.prototype.update = function(firstUpdate) {
    // Bail on the first update since doLoad will cause an update instead
    if (firstUpdate) {
      return;
    }
      
    // Load the value into the field editor
    if (this.isBound()) {
      var widget = this.getWidget();
      if (widget && widget.getType().is("kitPx:GenericFieldEditor") && widget.$fieldEditor && !widget.isModified()) {
        widget.$fieldEditor.loadValue(this.$target.getObject());
      }
    }
    
    // Call super
    WbFieldEditorBinding.$super.prototype.update.apply(this, arguments);
  };
    
  /**
   * @class ActionBinding.
   * <p>
   * A Component that represents a Niagara 'kitPx:ActionBinding' Type.
   *
   * @name ActionBinding
   * @extends Binding
   */
  var ActionBinding = function() {
    ActionBinding.$super.apply(this, arguments);
  }.$extend(Binding).registerType("kitPx:ActionBinding");
    
  /**
   * Handle an event from the Widget.
   *
   * @param {String} eventName.
   * @return {Boolean} return true if this binding has handled the event.
   */
  ActionBinding.prototype.handleEvent = function(eventName) {
    var target = this.$target;
    
    if (!this.isBound() || this.getWidgetEvent() !== "actionPerformed") {
      return false;
    }
    
    if (!(target.slot && target.slot.isAction())) {
      throw new Error("ActionBinding not bound to Action");
    }
    
    dialogs.action({
      component: target.getComponent(),
      slot: target.slot,
      parameter: this.getActionArg()
    });
         
    return true;
  };


  /**
   * @class MomentaryToggleBinding.
   * <p>
   * A Component that represents a Niagara 'kitPx:MomentaryToggleBinding' Type.
   *
   * @name MomentaryToggleBinding
   * @extends Binding
   */
  var MomentaryToggleBinding = function() {
    MomentaryToggleBinding.$super.apply(this, arguments);
  }.$extend(Binding).registerType("kitPx:MomentaryToggleBinding");

  /**
   * Sets the boolean state on the binding target based on whether the target is
   * a baja:Boolean, an Action with a baja:Boolean parameter or a baja:StausBoolean
   *
   * @private
   * @param binding
   * @param eventName
   * @returns {Boolean} Returns true if the function handled the event
   */
  function setValueOnTarget(binding, eventName) {

    if(eventName){
      binding.$lastEvent = eventName;
    }

    //dispatch vmouseup events outside of button to handleEvent() for a situation where the
    //button is pressed and released outside of the button bounds
    if (eventName === 'vmousedown') {
      $(document).one('vmouseup', function () {
        binding.handleEvent('vmouseup');
      });
    }

    function isPressed(widget) {
      return eventName === 'vmousedown' || !eventName && binding.$lastEvent === 'vmousedown';
    }

    var target    = binding.$target,
        component = target.getComponent(),
        widget    = binding.getWidget(),
        pressed   = isPressed(widget),
        slot      = target.slot;

    if (slot) {

      if (slot.isProperty()) {
        var typeSpec = slot.getType().getTypeSpec(),
            value;

        if (typeSpec === 'baja:Boolean') {
          component.set({
            slot: slot,
            value: pressed
          });
        }

        else if (typeSpec === 'baja:StatusBoolean') {
          value = component.get(slot);
          value.setStatus(baja.Status.make({
            orig: value.getStatus(),
            bits: baja.Status.NULL,
            state: false
          }));
          value.setValue(pressed)
        }

        else {
          throw new Error('configuration error, property must be Boolean or StatusBoolean');
        }
      }

      else if (slot.isAction()) {

        if (slot.getParamType().getTypeSpec() === 'baja:Boolean') {
          component.invoke({
            slot: slot,
            value: pressed
          });
        }

        else {
          throw new Error('configuration error, action must take boolean parameter');
        }
      }

      else {
        throw new Error('configuration error, slot must be property or action');
      }
    }

    else {

      if (baja.hasType(component, 'control:BooleanWritable')) {
        component.invoke({
          slot: 'set',
          value: pressed
        });
      }

      else {
        throw new Error('configuration error, target must be BooleanWritable');
      }
    }
    
    return true;
  }

  /**
   * Called when the binding is updated.
   */
  MomentaryToggleBinding.prototype.update = function () {
    setValueOnTarget(this);
    
    // Call super
    MomentaryToggleBinding.$super.prototype.update.apply(this, arguments);
  };

  /**
   * Handle an event from the Widget.
   *
   * @param {String} eventName.
   * @return {Boolean} return true if this binding has handled the event.
   */
  MomentaryToggleBinding.prototype.handleEvent = function (eventName) {
    return setValueOnTarget(this, eventName);
  };
  
  
}(baja));


