/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Mobile Px App.
 *
 * @author Gareth Johnson
 * @version 0.0.1.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: false, nomen: false, eqeqeq: true, bitwise: true, regexp: true, newcap: true, immed: true, strict: false, indent: 2, vars: true, continue: true */

/*global $, baja, location, window, history, niagara, document*/ 

(function pxAppView(baja) {
  // Use ECMAScript 5 Strict Mode
  "use strict";
    
  niagara.util.require(
    "$.mobile",
    "niagara.util.mobile.pages",
    "niagara.util",
    "niagara.util.mobile.dialogs",
    "niagara.util.mobile.commands",
    "niagara.util.mobile.views"
  );  
    
  var util = niagara.util,
      callbackify = util.callbackify,
      pages = util.mobile.pages, 
      dialogs = util.mobile.dialogs,
      views = util.mobile.views,
      commands = util.mobile.commands,
      pageStack = [],
      pageStackSize = 5, // The limit to the number of PxPage DOM elements to cache
      lex,
      isPreviousContentMobilePane = true;

////////////////////////////////////////////////////////////////
// Util
//////////////////////////////////////////////////////////////// 
      
  function getMobileLex() {
    if (!lex) {
      lex = baja.lex("mobile");
    }
    return lex;
  }
      
////////////////////////////////////////////////////////////////
// PageStack
////////////////////////////////////////////////////////////////  
  
  /**
   * Add a page DOM element to the internal stack.
   * <p>
   * The page stack keeps track of the loaded page DOM elements. A DOM element 
   * will be removed from the DOM if the stack size is exceeded.
   * 
   * @param pageDom the DOM element for an entire Px page
   */
  function addToPageStack(pageDom) {
    var ord = pageDom.jqmData("pageData").ord,
        i;
    
    // If the pageDom already exists in the stack then remove it
    for (i = 0; i < pageStack.length; ++i) {
      if (pageStack[i].jqmData("pageData").ord === ord) {
        pageStack.splice(i, 1);
        break;
      }
    }
    
    // Push this page onto the stack
    pageStack.push(pageDom);
    
    // If the limit of the stack has been reached then remove the
    // oldest DOM element from the page
    if (pageStack.length > pageStackSize) {
      $(pageStack.shift()).remove();
    }
  }
       
////////////////////////////////////////////////////////////////
// Px Page
////////////////////////////////////////////////////////////////      
  
  var PxPage = (function pxGraphicsPage() {
  
    /**
     * @class PxPage
     *
     * Represents a Px page.
     *
     * @param pageDiv the page DOM element
     * @param bson the BSON encoding of a Px page that we need to load
     * @param {baja.Ord} viewOrd the view ORD
     * @param {String} displayName the display name of the Px view
     */  
    function PxPage(pageDiv, bson, viewOrd, displayName) {
      this.$allBindings = [];    // All the bindings found
      this.$boundBindings = [];  // All the bindings with ORDs to be bound
      this.$ords = [];           // All of the ORDs to be resolved for the page
      this.$byComponent = {};    // A Component lookup map to associate Components to Bindings
      this.$subscriber = new baja.Subscriber();
      this.$contentDom = pageDiv.find(".pxContent");
      this.$headerDom = pageDiv.find(".pxHeader");
      this.$bson = bson;
      this.$started = false;
      this.$viewOrd = viewOrd;
      this.$displayName = displayName;
      this.$rootWidget = null;
    }
    
    /**
     * Called if the Px page fails to load.
     *
     * @param err
     */   
    function pxFailed(err) {
      baja.error(err); 
      dialogs.error(err);    
    }
          
    /**
     * Update the bindings from the result of the batch request.
     *
     * @param pxPage
     * @param {baja.comm.Batch} batch
     */   
    function updateBindings(pxPage, batch) {
      var target, navOrd, comp, i,
          boundBindings = pxPage.$boundBindings,
          byComponent = pxPage.$byComponent,
          allBindings = pxPage.$allBindings;
      
      // Detach all current event handlers
      pxPage.$subscriber.detach();
      
      // Attach the changed handler now that everything is subscribed
      pxPage.$subscriber.attach("changed", function () {
        var navOrd = this.getNavOrd();
        if (pxPage.$byComponent.hasOwnProperty(navOrd)) {
          var b = pxPage.$byComponent[navOrd],
              i;
          for (i = 0; i < b.length; ++i) {
            b[i].update();
          }
        }
      });
      
      // Set the ORD Targets up in the binding Components
      for (i = 0; i < batch.size(); ++i) {
        target = null;
        if (batch.isResolved(i)) {
          target = batch.getTarget(i);
          boundBindings[i].$fw("updateBinding", target);
          
          comp = target.getComponent();
          if (comp && comp.isMounted()) {            
            navOrd = comp.getNavOrd();
            if (!byComponent.hasOwnProperty(navOrd)) {
              byComponent[navOrd] = [boundBindings[i]];
            }
            else {
              byComponent[navOrd].push(boundBindings[i]);
            }
          }
        }
        else {
          baja.error("Failed to resolve: " + boundBindings[i].getOrd().toString());
        }
      }
      
      // Append the root of the Px page to the DOM
      pxPage.$contentDom.append(pxPage.$rootDom);
      
      // Handle the commands button click
      $("#commandsButton").unbind("click");
      $("#commandsButton").click(commands.showCommandsHandler);
      
      // Update all bindings
      for (i = 0; i < allBindings.length; ++i) {
        allBindings[i].load();
        allBindings[i].update(/*firstUpdate*/true);
      }
                                            
      // Update the commands button depending on whether the Px page is modified
      $("a.commandsButton").toggleClass("red", pxPage.$rootWidget.isWidgetTreeModified());
            
      // Update the header with the display name
      pxPage.$headerDom.text(pxPage.$displayName);
      
      // Update the main HTML title text...
      try {
        document.title = pxPage.$displayName;
      }
      catch(ignore) {
        // Believe it or not, IE doesn't seem to like changing the title text so we need to catch any errors here
      }
      
      pxPage.$rootWidget.getDomElement().trigger("pxPageLoaded");
    }
          
    /**
     * Recursively search for all of the bindings.
     *
     * @param pxPage
     * @param comp the Component to search for bindings on.
     */
    function findBindings(pxPage, comp) {
      if (!comp) {
        return;
      }
      
      // Make a note of any bindings we find
      if (comp.getType().is("bajaui:Binding")) {        
        if (comp.getOrd() !== baja.Ord.DEFAULT) {
          // Join base ORD and Binding ORD so BatchResolve will normalize and process a complete absolute ORD
          pxPage.$ords.push(baja.Ord.make({
            base: pxPage.$viewOrd,
            child: comp.getOrd().toString()
          }));
          pxPage.$boundBindings.push(comp);
        }
        pxPage.$allBindings.push(comp);
      }
      
      var map = comp.$map.$map, // Use internal OrderedMap for max efficiency
          p;
      
      // Search all Widgets for bindings      
      for (p in map) {
        if (map.hasOwnProperty(p) && map[p].isProperty() && map[p].getType().isComponent()) {
          findBindings(pxPage, comp.get(map[p]));  
        }
      }
    }
    
    function postLoadWidgetsHasUpdate() {
      return true;
    }
     
    /**
     * Recursively called once all Widgets have been loaded so they can update for the first time.
     *
     * @param pxPage
     * @param widget
     */   
    function postLoadWidgets(pxPage, widget) {    
      // Check we have the correct methods defined on this objects
      if (typeof widget.isModified === "function" && typeof widget.save === "function") {
        pxPage.$rootWidget.addSavableWidget(widget);
      }
      
      // Internally set the root Widget for performance
      widget.$root = pxPage.$rootWidget;
      
      // Only post load and update if there's a valid DOM element available for the Widget
      if (widget.getDomElement()) {    
        // Post load the Widget after the page has been enhanced
        widget.postLoad();
        
        // First update
        widget.update(postLoadWidgetsHasUpdate, /*firstUpdate*/true);
      }
      
      var map = widget.$map.$map, // Use internal OrderedMap for max efficiency
          p;
      
      for (p in map) {
        if (map.hasOwnProperty(p) && map[p].isProperty() && map[p].getType().isComponent() && map[p].getType().is("bajaui:Widget")) {
          postLoadWidgets(pxPage, widget.get(map[p]));  
        }
      }
    }
            
    /**
     * Load the Widgets for the Px page
     *
     * @param pxPage
     */
    function loadWidgets(pxPage) {  
      // Load the Widget tree and set a root container
      var rootWidget = pxPage.$rootWidget = baja.$("bajaui:RootContainer");
      rootWidget.$pxPage = pxPage;
      rootWidget.add({ slot: "content", value: baja.bson.decodeValue(pxPage.$bson, baja.$serverDecodeContext) });
                    
      // Create the room DOM element
      var rootDom = pxPage.$rootDom = $("<div></div>");
         
      // Create DOM structure
      rootWidget.load(rootDom);
      rootWidget.loadChildren();
      
      // Now all the content is added, dynamically update the page
      rootDom.trigger("create"); 
      
      // Post load and update the Widgets
      postLoadWidgets(pxPage, rootWidget);
      
      // Find the bindings
      findBindings(pxPage, pxPage.$rootWidget);
    }
    
    /**
     * Start the Px page.
     * <p>
     * This will subscribe any valid bindings on the Px page.
     */   
    PxPage.prototype.start = function() { 
      var pxPage = this;
    
      // If already started then don't stop
      if (pxPage.$started) {
        return;
      }
      
      pxPage.$started = true;
      
      // TODO: Consider any timing issues between subscription and unsubscription on start and stop.
      
      // Resolve the bindings and Subscribe too all mounted Components
      var batch = new baja.BatchResolve(pxPage.$ords),
          isContentMobilePane = pxPage.$rootWidget.get("content").getType().is("mobile:IMobilePane");
      
      // Keep track of previous state to avoid any unnecessary reflows...      
      if (isPreviousContentMobilePane !== isContentMobilePane) {    
        isPreviousContentMobilePane = isContentMobilePane;
        $(".ui-mobile-viewport,.ui-content").toggleClass("nonRootMobilePane", !isContentMobilePane);
        $("div[data-role='page']").toggleClass("nonRootMobilePanePage", !isContentMobilePane);
      }
            
      batch.resolve({
        subscriber: pxPage.$subscriber,
        ok: function () {
          updateBindings(pxPage, this);    
          
          pxPage.$rootWidget.getDomElement().trigger("pxPageStarted");
        },
        fail: function (err) {
          updateBindings(pxPage, this);
        }
      }); 
    };
    
    /**
     * Stop the Px page.
     * <p>
     * This will unsubscribe any bound bindings on the Px page.
     */
    PxPage.prototype.stop = function() { 
      var pxPage = this;
      
      // Set the header to loading
      pxPage.$headerDom.text(baja.lex("mobile").get("loading"));
          
      if (pxPage.$started) {
        pxPage.$started = false;
        pxPage.$subscriber.detach();
        pxPage.$subscriber.unsubscribeAll({
          ok: function () {
            pxPage.$rootWidget.getDomElement().trigger("pxPageStopped");
          },
          fail: baja.fail
        });
      }

    };
    
    /**
     * Load the Px page.
     * <p>
     * This will decode and load all of the Widgets and Bindings for the Px page.
     * 
     * @param {Function} ok called once the Px page has loaded.
     */
    PxPage.prototype.load = function(callbacks) {
      callbacks = util.callbackify(callbacks, baja.ok, pxFailed);
      var that = this;
      if (that.$loaded) {
        callbacks.ok();
        return;
      }
      
      function importOk() {
        try {
          // Decode the Px BSON embedded into the page
          that.$contentDom.empty();
          loadWidgets(that);
          that.start();
          that.$loaded = true;
          callbacks.ok();
        }
        catch(err) {
          callbacks.fail(err);
        }
      }
    
      // Scan BSON for unknown Types and make a network call if necessary           
      var unknownTypes = baja.bson.scanForUnknownTypes(this.$bson);            
      if (unknownTypes.length > 0) {
        baja.importTypes({
          typeSpecs: unknownTypes, 
          ok: importOk, 
          fail: function (err) {
            pxFailed(err);
          }
        });
      }
      else {
        importOk();
      }  
    };
    
    /**
     * Return the ORD for this view.
     * 
     * @returns {baja.Ord}
     */
    PxPage.prototype.getViewOrd = function() {
      return this.$viewOrd; 
    };
    
    /**
     * Return the display name for the view.
     * 
     * @returns {String}
     */
    PxPage.prototype.getDisplayName = function() {
      return this.$displayName; 
    };
    
    /**
     * Return true if the current Px Page is modified.
     *
     * @returns {Boolean}
     */
    PxPage.isCurrentPxPageModified = function() {
      var pageData = pages.getCurrent().jqmData("pageData"); 
      if (pageData && pageData.pxPage && pageData.pxPage.$rootWidget) {
        return pageData.pxPage.$rootWidget.isWidgetTreeModified();
      }
      else {
        return false;
      }
    };
    
    /**
     * Return the current Px Page (or null if there isn't one available)
     *
     * @returns PxPage (or null if there is none available).
     */
    PxPage.getCurrentPxPage = function() {
      var pageData = pages.getCurrent().jqmData("pageData"); 
      return pageData && pageData.pxPage ? pageData.pxPage : null;
    };
        
    return PxPage;
  }()); // pxGraphicsPage
    
  ////////////////////////////////////////////////////////////////
  // Pages
  ////////////////////////////////////////////////////////////////  
    
  pages.register(function ordMatcher(str) {
    // Handle all ORDs that match this String  
    return str.match(/^ord/);
  },
  (function pxPageHandler() {
    // All of these functions are handlers for the Pages framework
    
    /**
     * If the current PX page is modified, shows a confirmation dialog asking
     * whether to save changes before invoking the action.
     * 
     * @param {Function} action the action we wish to perform - inside this
     * function, <code>this</code> will be bound to the dialog invocation
     * (enabling things like <code>this.redirect(anotherPage);</code>)
     * @param {Object} callbacks an object containing ok/fail callbacks (to
     * be called when the dialog is closed, whether or not the action was
     * performed)
     */
    function confirmAbandonChanges(action, callbacks) {
      callbacks = callbackify(callbacks);
      
      var rootPage, rootWidget;
      
      // Capture any save changes if the user attempts to navigate away from an unsaved Px Page
      if (PxPage.isCurrentPxPageModified()) {
        rootPage = PxPage.getCurrentPxPage();
        rootWidget = rootPage.$rootWidget;
            
        dialogs.confirmAbandonChanges({
          viewName: rootPage.getDisplayName(),
          yes: function (cb) {
            var that = this;
            // Save the Widgets
            rootWidget.saveWidgets(function () {
              action.call(that);
              cb.ok();
            });
          },
          no: function (cb) {
            var that = this;
            
            // Clear all modified changes
            rootWidget.clearWidgetTreeModified();
            
            action.call(that);
            cb.ok();
          },
          callbacks: callbacks
        });
      } else {
        action();
        callbacks.ok();
      }
    }
  
    function decodeUrl(url) {      
      return {
        ord: baja.Ord.make(url.href)
      };
    }

    function encodeUrl(pageData) {
      var ord = pageData.ord || niagara.view.ord,
          url = baja.Ord.make(ord).toUri();
      
      return url;
    }
    
    var pxHtml = "<div data-role='page' id='px' >" +
      "<div data-role='header' data-theme='a'>" +
        "<h1 class='pxHeader profileHeader'>{title}</h1>" +
        "<a class='profileHeader profileHeaderBack' data-icon='arrow-l' data-iconpos='notext'></a>" +
        "<a class='commandsButton profileHeader' id='commandsButton' title='{menuBtn}' data-icon='menu' data-role='button' data-iconpos='notext'></a>" +
      "</div>" +
      "<div class='pxContent' data-role='content'>" +
      "</div>" +
    "</div>";
        
    function createPage(obj) {    
      var pageData = obj.pageData,
          ord = pageData.ord; // TODO: is this legal?
          
      function loadPx(title, bson) {    
        var pageDiv = $(pxHtml.patternReplace({
          title: baja.lex("mobile").get("loading"),
          menuBtn: baja.lex("mobile").get("menu")
        }));
        
        pageDiv.find('.profileHeaderBack')
          .unbind()
          .bind('click', function () {
            confirmAbandonChanges(function () { history.back(); });
          });
        
        // Add the pxPage to the page data      
        pageData.pxPage = new PxPage(pageDiv, bson, ord, title);
        
        obj.ok(pageDiv);
      } 
      
      // If maps to the originally loaded Px Page
      if (ord.toString() === niagara.view.ord) {
        loadPx(niagara.view.px.displayName, niagara.view.px.bson);
      }
      else {
        // Make a Servlet call to the Px App to get the page data
        $.ajax({
          url:  ord.toUri(),
          type: "POST",
          headers: {
            "niagara-mobile-rpc": "px"
          },
          success: function (data) {
            loadPx(data.displayName, data.bson); 
          }
        });
      } 
    }
    
    function pagebeforehide(obj) {
      var pageData = obj.page.jqmData("pageData"),
          nextPageData = obj.nextPage.jqmData("pageData"),
          nextPxPage = nextPageData && nextPageData.pxPage ? nextPageData.pxPage : {};
           
      // Only stop this page if the current page is Px and the next page isn't a dialog
      if (pageData &&
          pageData.pxPage &&
          obj.nextPage.jqmData("role") !== "dialog" &&
          nextPxPage !== pageData.pxPage) {
        pageData.pxPage.stop();
      }
    }
        
    function pagelayout(obj) {
      var pageData = obj.page.jqmData("pageData"),
          pxPage = pageData && pageData.pxPage;
      if (pxPage) {
        // Load the Px page and make the callback once completed
        pageData.pxPage.load(function () {
          pageData.pxPage.start();
        });
      }
    }

    function pageshow(obj) {
      addToPageStack(obj.page);
    }

    function pagebeforechange(obj) {
      var newId = obj.nextPage.attr('id'),
          currentId = obj.page.attr('id');
          
      if (PxPage.isCurrentPxPageModified() && newId !== currentId) {
        obj.event.preventDefault();
        confirmAbandonChanges(function () {
          this.redirect(util.mobile.decodePageId(newId));
        });
      }
    } 


    function getCommands(obj) {
      var cmds = obj.commands,
          homeCmd = commands.getHomeCmd(),
          homeIndex = util.indexOf(cmds, homeCmd),
          logoutCmd = commands.getLogoutCmd(),
          logoutIndex = util.indexOf(cmds, logoutCmd),
          pxPage = PxPage.getCurrentPxPage();
      
      if (homeIndex >= 0) {
        cmds[homeIndex] = new commands.AsyncCommand(homeCmd.getDisplayName(), function (callbacks) {
          confirmAbandonChanges(function () {
            homeCmd.invoke();
          }, callbacks);
        });
      }
      
      if (logoutIndex >= 0) {
        cmds[logoutIndex] = new commands.AsyncCommand(logoutCmd.getDisplayName(), function (callbacks) {
          confirmAbandonChanges(function() {
            logoutCmd.invoke();
          }, callbacks);
        });
      }
      
      if (pxPage) {
        // Add refresh command
        cmds.splice(0, 0, new commands.AsyncCommand("%lexicon(mobile:refresh)%", function(callbacks) { 
          pxPage.$rootWidget.refresh(callbacks);
        }));
        
        // Only add the save command if the current view is modified
        if (PxPage.isCurrentPxPageModified()) {
          // Add save command
          cmds.splice(0, 0, new commands.Command("%lexicon(mobile:save)%", function() {
            pxPage.$rootWidget.saveWidgets(function() {
              dialogs.ok({
                title: getMobileLex().get("saved"),
                content: getMobileLex().get("pxPageSaved")
              });
            }); 
            return false;
          }));
        }
      }
      
      return cmds;
    }    
    
    // Functions exported as public
    return {
      decodeUrl: decodeUrl,
      encodeUrl: encodeUrl,
      createPage: createPage,
      pagebeforehide: pagebeforehide,
      pagelayout: pagelayout,
      pageshow: pageshow,
      pagebeforechange: pagebeforechange,
      getCommands: getCommands
    };
  }())); // pxPageHandler
  
  ////////////////////////////////////////////////////////////////
  // Scroll Position Restore
  //////////////////////////////////////////////////////////////// 
   
  (function scrollPositionRestore() {
    var scrollDataKey = "scrollData";
  
    $(document).bind("pagechange", function pagechange(e, data) {     
      // Check we're restoring to a page. If so then attempt to restore the scroll positions  
      var toPage = data.toPage,
          scrollData;
      
      if (toPage &&
          toPage instanceof $ &&
          toPage.jqmData("role") === "page") {
        // Restore the scroll information once the page has shown
        scrollData = toPage.jqmData(scrollDataKey);
        if (scrollData) {
          $(document).one("silentscroll", function () {
            // On silent scroll event, schedule a scroll reset shortly afterwards
            // (must be greater than 20 ms)
            baja.clock.schedule(function () {
              window.scrollTo(scrollData.x, scrollData.y);
            }, 30);
          });
        }
      }
    });
     
    $(document).bind("pagebeforechange", function pagebeforechange(e, data) {
      // In order to note the scroll positions so we can restore them between page
      // changes, // we need to pick up this event so we can note the scroll positions
      // down before anything on the page changes.
      
      var scrollData,
          toPage = data.toPage,
          toRole,
          fromPage = $.mobile.activePage,
          fromRole;
                    
      // If we're going to a dialog from a page then note down the scroll positions    
      if (fromPage &&
          fromPage instanceof $ &&
          toPage &&
          toPage instanceof $) {
        toRole = toPage.jqmData("role");
        fromRole = fromPage.jqmData("role");

        // If we're going to a dialog then note the scroll position
        if (fromRole === "page" &&
            toRole === "dialog") {
          scrollData = fromPage.jqmData(scrollDataKey);
          
          // Lazily create the scroll data if it doesn't exist
          if (!scrollData) { 
            scrollData = {};      
            fromPage.jqmData(scrollDataKey, scrollData);
          }
                    
          // Note down the scroll positions
          scrollData.x = window.scrollX;
          scrollData.y = window.scrollY;
        }
        else if (fromRole === "page" &&
                 toRole === "page") {
          // If we're going from page to page then remove the scroll positions
          // because we don't want to restore in this case.
         fromPage.jqmRemoveData(scrollDataKey);
        }
      }
    });  
  }()); // scrollPositionRestore
          
  ////////////////////////////////////////////////////////////////
  // Initialize
  //////////////////////////////////////////////////////////////// 
  
  // Commands button
  $('a.commandsButton').live("onShowCommands", function() {
    var pxPage = PxPage.getCurrentPxPage();
    if (pxPage) {
      views.getViewsCommand().setOrd(pxPage.getViewOrd());
    }
  });
          
  // Update the commands button if the page is modified
  $("div").live("editorchange editorsave", baja.throttle(function() {
    $("a.commandsButton").toggleClass("red", PxPage.isCurrentPxPageModified());
  }, 500));  
  
  // Handle when the user attempts to leave the page prematurely
  window.onbeforeunload = function() {
    if (PxPage.isCurrentPxPageModified()) {
      return getMobileLex().get({
        key: 'message.viewModified',
        args: [PxPage.getCurrentPxPage().getDisplayName()]
      });
    }
  };
  
  function initializeUI() {
    function resetToViewOrd() {
      $.mobile.changePage(baja.Ord.make(niagara.view.ord).toUri(), { 
        transition: "none", 
        changeHash: false 
      });
    }

    pages.register("splash", {
      pageshow: resetToViewOrd
    });
    
    resetToViewOrd();
  }
  
  baja.started(initializeUI);
  
}(baja)); // pxAppView
