//
// Copyright 2010, Tridium, Inc. All Rights Reserved.
//

/**
 * @fileOverview Gx primitives implemented in BajaScript for the Px App.
 *
 * @author Gareth Johnson
 * @version 0.0.1.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: false, nomen: false, eqeqeq: true, bitwise: true, regexp: true, newcap: true, immed: true, strict: false, indent: 2, vars: true, continue: true */

/*global $, baja, BaseBajaObj, window, niagara*/ 
  
/**
 * @namespace Niagara Px
 */ 
niagara.util.namespace("niagara.px");  
  
(function pxGx(baja) {
  // Use ECMAScript 5 Strict Mode
  "use strict";
  
  var util = niagara.util,
      gradientUtil = util.gradient,
      colorUtil = util.color,
      colorCache = {},
      colorConstants = {},
      
      strictArg = baja.strictArg, // Create local for improved minification
      strictAllArgs = baja.strictAllArgs,
      bajaDef = baja.def,
      
      parseRgba = colorUtil.parseRgba,
      COLOR_CONSTANTS = colorUtil.COLOR_CONSTANTS,
      
      notCurrentlySupported = "Not currently supported",
      
      colorName;
  
  /**
   * @class Color.
   * <p>
   * This is currently a partial implementation of Niagara's 'gx:Color' Type.
   * <p>
   * When creating a Simple, always use the 'make' method instead of creating a new Object.
   *
   * @name niagara.px.Color
   * @extends baja.Simple
   */
  niagara.px.Color = function(str, rgba) {
    niagara.px.Color.$super.apply(this, arguments);
    this.$str = strictArg(str, String);
    this.$rgba = strictArg(rgba, Object);
  }.$extend(baja.Simple);
    
  function makeRgba() {
    return {
      red: 0,
      blue: 0,
      green: 0,
      alpha: 1,
      css: ""
    };
  }  
    
  /**
   * Make a Color.
   * <p>
   * A Color can be made from a String (str) or from some rgba (red, green, blue or alpha) values.
   * If any of the RGBA values are specified, this will take precedence over a the String.
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} [obj.str] the String encoding for a Color.
   * @param {Number} [obj.red] the red color (0-255).
   * @param {Number} [obj.green] the green color (0-255).
   * @param {Number} [obj.blue] the blue color (0-255).
   * @param {Number} [obj.alpha] the alpha color (0-1).
   * @return {niagara.px.Color}
   */
  niagara.px.Color.make = function(obj) {
    obj = baja.objectify(obj, "str");
    
    var str = obj.str;
    var rgba;
    
    // Create color from rgba if specified in Object Literal arguments
    if (obj.red !== undefined || obj.green !== undefined || obj.blue !== undefined || obj.alpha !== undefined) {
      rgba = makeRgba();
      
      // TODO: Could check limits
      if (obj.red !== undefined) {
        rgba.red = Math.round(obj.red);
      }
      
      if (obj.green !== undefined) {
        rgba.green = Math.round(obj.green);
      }
      
      if (obj.blue !== undefined) {
        rgba.blue = Math.round(obj.blue);
      }
      
      if (obj.alpha !== undefined) {
        rgba.alpha = obj.alpha;
      }
      
      // Check for default
      if (rgba.red === 0 && rgba.green === 0 && rgba.blue === 0 && rgba.alpha === 1) {
        return niagara.px.Color.DEFAULT;
      }
      
      if (rgba.alpha === 1) {
        str = rgba.css = "rgb(" + rgba.red + "," + rgba.green + "," + rgba.blue + ")";
      }
      else {
        str = rgba.css = "rgba(" + rgba.red + "," + rgba.green + "," + rgba.blue + "," + rgba.alpha + ")";
      }
    }
    else {
      // Handle null
      if (str === "null" || str === "") {
        return niagara.px.Color.NULL;
      }
      
      // If this is a constant then return it
      var lowerCaseStr = str.toLowerCase();
      if (colorConstants.hasOwnProperty(lowerCaseStr)) {
        return colorConstants[lowerCaseStr];
      }
    }
    
    // Check the cache to see if it exists
    if (colorCache.hasOwnProperty(str)) {
      return colorCache[str];
    }
    
    // Create new color and add to the internal Cache
    var color = new niagara.px.Color(str, rgba || parseRgba(str));
    colorCache[str] = color;
    return color;
  };
  
  /**
   * Make a Color
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} obj.str the String encoding for a Color.
   * @return {niagara.px.Color}
   */
  niagara.px.Color.prototype.make = function(obj) {
    return niagara.px.Color.make.apply(niagara.px.Color, arguments);
  };
  
  /**
   * Decode a Color from a String
   *
   * @param {String} str
   * @return {niagara.px.Color}
   */   
  niagara.px.Color.prototype.decodeFromString = function(str) {
    return niagara.px.Color.make(str);
  };
  
  /**
   * Encode the Color to a String
   *
   * @return {String}
   */  
  niagara.px.Color.prototype.encodeToString = function() {
    return this.$str;
  };
  
  // Color constants
  
  function constant(colName, str) {
    var c = new niagara.px.Color(colName, parseRgba(str));     
    colorCache[str] = c;    
    colorConstants[colName.toLowerCase()] = c; 
    return c;
  }  
  
  // Even though these are part of CSS3, we need to declare them so we can get access to their RGBA values.
  // The RGBA values are used in the spectrum bindings.
  for (colorName in COLOR_CONSTANTS) {
    if (COLOR_CONSTANTS.hasOwnProperty(colorName)) {
      niagara.px.Color[colorName] = constant(colorName, COLOR_CONSTANTS[colorName]);
    }
  }
      
  /**
   * Default Color instance
   */
  niagara.px.Color.DEFAULT = niagara.px.Color.black;
  
  /**
   * Null Color instance
   */
  niagara.px.Color.NULL = new niagara.px.Color("", makeRgba());
  
  // Register Type
  niagara.px.Color.registerType("gx:Color"); 
    
  /**
   * Update an HTML DOM element.
   *
   * @param {Object} obj contains method arguments.
   */  
  niagara.px.Color.prototype.update = function(obj) {   
    var prefix = obj.prefix || "";
    if (!this.$rgba.css) {
      prefix = "";
    }
    obj.dom.css(obj.colorSelector, prefix + this.$rgba.css);
  };
    
  /**
   * Return the red part of the color
   *
   * @return {Number}
   */  
  niagara.px.Color.prototype.getRed = function() {  
    return this.$rgba.red;
  };
  
  /**
   * Return the green part of the color
   *
   * @return {Number}
   */  
  niagara.px.Color.prototype.getGreen = function() {  
    return this.$rgba.green;
  };
  
  /**
   * Return the blue part of the color
   *
   * @return {Number}
   */  
  niagara.px.Color.prototype.getBlue = function() {  
    return this.$rgba.blue;
  };
  
  /**
   * Return the alpha
   *
   * @return {Number}
   */  
  niagara.px.Color.prototype.getAlpha = function() {  
    return this.$rgba.alpha;
  };
  
  /**
   * Return the decoded CSS for the Color.
   */
  niagara.px.Color.prototype.getCss = function() {
    return this.$rgba.css || "";
  };
  
  // Add scope to paint functions
  (function brush() {
    // TODO: We need to support images here.
  
    // We don't support everything that BBrush does yet
    function Unsupported(str) {
      this.update = function () {
        // Unsupported brush so do nothing
      };
      
      this.encodeToString = function () {
        return str;
      };
    }
  
    // Solid color brush
    function Solid(color) {
      // Update the DOM element
      this.update = function (obj) {
        color.update.apply(color, arguments);
      };
      
      this.encodeToString = function () {
        return color.encodeToString();
      };
    }
    
    // Image Brush
    function Image(encoding) {
      var str = encoding,
          res,
          regex = /([a-zA-Z]*)\(([^()]*)\)/g,
          params = {};
   
      // Remove image(...)        
      str = str.substring(6, str.length - 1);
      
      res = regex.exec(str);
      while (res) {
        params[res[1]] = res[2];
        res = regex.exec(str);
      }
      
      // Ensure all image variables are initialized
      params.tile = params.tile || "false";
      params.halign = params.halign || "center";
      params.valign = params.valign || "center";
    
      // Update the DOM element with the image
      this.update = function (obj) {
        if (obj.imageSelector && obj.imageSelector === "background") {
          var css = "url(/ord?" + params.source + ") ";
                 
          if (params.tile === "true") {
            css += "repeat ";
          }
          else if (params.tile === "false") {
            css += "no-repeat ";
          }
          else if (params.tile === "x") {
            css += "repeat-x ";
          }
          else if (params.tile === "y") {
            css += "repeat-y ";
          }
          
          css += params.halign + " " + params.valign;         
          obj.dom.css(obj.imageSelector, css);
        }
      };
    
      this.encodeToString = function () {
        return encoding;
      };
    }
    
    var linearGradientSelections = [
      "-webkit-linear-gradient",
      "-moz-linear-gradient",
      "-o-linear-gradient",
      "-ms-linear-gradient"
      //"linear-gradient" leave out and rely on
      // vendor specific gradients
      // for now as this causes some issues with the latest versions
      // of Firefox.
    ];
    
    function parseStop(res) {
      // Parse out 'offset color'
      var stopSplitStr = res[2].split(" "),
          colorCss = niagara.px.Color.DEFAULT.decodeFromString(stopSplitStr[1]).getCss(),
          offset = stopSplitStr[0];

      return {
        color: colorCss,
        offset: offset,
        toString: function () {
          return colorCss + " " + offset;
        }
      };
    }
    
    /**
     * @private
     * @inner
     * @ignore
     * @param {String} encoding a String encoding of a BBrush.LinearGradient
     * @returns {Object} an object with <code>angle<code> and <code>stops</code>
     * properties. The stops are a two-dimensional array in the form
     * <code>[["0%", "color1"], ["100%", "color2"]]<code>
     */
    function parseAngleAndStops(encoding) {
      var str = encoding,
          regex = /([a-zA-Z]*)\(([^()]*)\)/g,
          stops = [],
          stop,
          angle = 0,
          res;
    
      // Remove linearGradient(...)        
      str = str.substring(15, str.length - 1);
    
      res = regex.exec(str);
      
      while (res) {
        // We need to decode the stops in order
        if (res[1] === "stop") {
          // Parse out offset color
          stop = parseStop(res);
          stops.push([stop.offset, stop.color]);
        }
        else if (res[1] === "angle") {
          angle = parseInt(res[2], 10);
        }
        res = regex.exec(str);
      }
      
      return { angle: angle, stops: stops };
    }
    
    // Linear Gradient Brush
    function LinearGradient(encoding) {
      var as = parseAngleAndStops(encoding),
          angle = as.angle,
          stops = as.stops;
      
      this.update = function (obj) {
        var css = gradientUtil.createLinearGradientCss(angle, stops),
            selector = obj.imageSelector,
            prefix = obj.prefix || "",
            dom = obj.dom,
            i;
        
        if (css !== null) {
          // Apply the first color for older browser support
          dom.css(selector, prefix + css.background);
          
          // Add limited IE8/9 Support for background color only
          if (selector === "background") {
            dom.css("filter", css.filter);
          }
          
          dom.css(selector, "-webkit-gradient" + css.webkitGradient);

          for (i = 0; i < linearGradientSelections.length; ++i) {
            dom.css(selector, linearGradientSelections[i] + 
                css.linearGradientNonStandard);
          }
          
          dom.css(selector, "linear-gradient" + css.linearGradient);
        }
      };
      
      this.encodeToString = function () {
        return encoding;
      };
    }
    
    var radialGradientSelections = [
      "-webkit-radial-gradient",
      "-moz-radial-gradient",
      "-o-radial-gradient",
      "-ms-radial-gradient"
      //"radial-gradient" leave out and rely on
      // vendor specific gradients
      // for now as this causes some issues with the latest versions
      // of Firefox.
    ];
    
    // Radial Gradient Brush
    function RadialGradient(encoding) {
      var str = encoding,
          regex = /([a-zA-Z]*)\(([^()]*)\)/g,
          stops = [],
          stopSplitStr,
          center = "50% 50%",
          css,
          firstColor,
          res;
    
      // Remove radialGradient(...)        
      str = str.substring(15, str.length - 1);
    
      res = regex.exec(str);
      while (res) {
        // We need to decode the stops in order
        if (res[1] === "stop") {
          // Parse out offset color
          stops.push(parseStop(res));
        }
        else if (res[1] === "c") {
          center = res[2];
        }
        res = regex.exec(str);
      }
            
      // Build base CSS
      css = "(" + center + ", circle, " + stops.join(", ") + ")";      
            
      firstColor = stops.length > 0 ? stops[0].color : "#FFFFFF";     
                      
      this.update = function (obj) {    
        // Apply the first color for older browser support (< IE10)
        obj.dom.css(obj.imageSelector, firstColor);
        
        var i;
        for (i = 0; i < radialGradientSelections.length; ++i) {
          obj.dom.css(obj.imageSelector, radialGradientSelections[i] + css);
        }
      };
      
      this.encodeToString = function () {
        return encoding;
      };
    }
    
    // Parse a String and return a paint Object
    function parseBrush(encoding) {
      if (encoding.match(/^image\(/)) {     
        return new Image(encoding);
      }
      else if (encoding.match(/^linearGradient\(/)) {     
        return new LinearGradient(encoding);
      }
      else if (encoding.match(/^radialGradient\(/)) {     
        return new RadialGradient(encoding);
      }
            
      return new Solid(niagara.px.Color.DEFAULT.decodeFromString(encoding));
    }
      
    /**
     * @class Brush.
     * <p>
     * This is currently a partial implementation of Niagara's 'gx:Brush' Type.
     * <p>
     * Please note, this isn't a complete implementation but will do for the Px App.
     * When creating a Simple, always use the 'make' method instead of creating a new Object.
     *
     * @name niagara.px.Brush
     * @extends baja.Simple
     */
    niagara.px.Brush = function(paint) {
      niagara.px.Brush.$super.apply(this, arguments);
      this.$paint = paint;
    }.$extend(baja.Simple);
    
    /**
     * Make a Brush
     *
     * @param {Object} obj Object Literal for the method's arguments.
     * @param {String} [obj.str] the String encoding for a Brush.
     * @param {niagara.px.Color} [obj.color] the solid color for the brush.
     * @return {niagara.px.Brush}
     */
    niagara.px.Brush.make = function(obj) {
      obj = baja.objectify(obj, "str");
      
      if (obj.str === "" || obj.str === "null") {
        return niagara.px.Brush.NULL;
      }
      
      var paint;
      if (obj.str) {
        paint = parseBrush(obj.str);
      }
      else if (obj.color) {
        paint = new Solid(obj.color);
      }
      
      return new niagara.px.Brush(paint);
    };
    
    /**
     * Make a Brush
     *
     * @param {Object} obj Object Literal for the method's arguments.
     * @param {String} [obj.str] the String encoding for a Brush.
     * @param {niagara.px.Color} [obj.color] the solid color for the brush.
     * @return {niagara.px.Brush}
     */
    niagara.px.Brush.prototype.make = function(obj) {
      return niagara.px.Brush.make.apply(niagara.px.Brush, arguments);
    };
    
    /**
     * Decode a Brush from a String
     * 
     * @name niagara.px.Brush#decodeFromString
     * @function
     *
     * @param {String} str
     * @return {niagara.px.Brush}
     */   
    niagara.px.Brush.prototype.decodeFromString = function(str) {
      return niagara.px.Brush.make(str);
    };
    
    /**
     * Encode the Brush to a String
     *
     * @return {String}
     */  
    niagara.px.Brush.prototype.encodeToString = function() {
      return this.$paint.encodeToString();
    };
    
    /**
     * Default Brush instance
     */
    niagara.px.Brush.DEFAULT = new niagara.px.Brush(new Solid(niagara.px.Color.DEFAULT));
    
    /**
     * Null Brush instance
     */
    niagara.px.Brush.NULL = new niagara.px.Brush(new Solid(niagara.px.Color.NULL));
    
    // Register Type
    niagara.px.Brush.registerType("gx:Brush"); 
    
    /**
     * Update an HTML DOM element.
     *
     * @param {Object} obj contains method arguments.
     */  
    niagara.px.Brush.prototype.update = function(obj) {
      this.$paint.update.apply(this.$paint, arguments);
    };
    
  }());
  
  var fontCache = {};
  
  /**
   * @class Font.
   * <p>
   * This is currently a partial implementation of Niagara's 'gx:Font' Type.
   * <p>
   * When creating a Simple, always use the 'make' method instead of creating a new Object.
   *
   * @name niagara.px.Font
   * @extends baja.Simple
   */
  niagara.px.Font = function(obj) {
    niagara.px.Font.$super.apply(this, arguments);
    this.$fontName = strictArg(obj.fontName, String);
    this.$size = strictArg(obj.size || 0, Number);
    this.$bold = strictArg(obj.bold || false, Boolean);
    this.$italic = strictArg(obj.italic || false, Boolean);
    this.$underline = strictArg(obj.underline || false, Boolean);
    this.$str = strictArg(obj.str || "", String);
  }.$extend(baja.Simple);
   
  /**
   * Make a Font.
   */
  niagara.px.Font.make = function(obj) {  
    throw new Error(notCurrentlySupported);
  };
  
  /**
   * Make a Font.
   */
  niagara.px.Font.prototype.make = function(obj) {
    return niagara.px.Font.make.apply(niagara.px.Font, arguments);
  };
  
  /**
   * Decode a Font from a String
   *
   * @param {String} str
   * @return {niagara.px.Font}
   */   
  niagara.px.Font.prototype.decodeFromString = function(str) {    
    // Attempt to get from the Cache
    if (fontCache.hasOwnProperty(str)) {
      return fontCache[str];
    }
    
    // Decode font
    var res,
        bold = false, 
        italic = false, 
        underline = false, 
        size, 
        fontName,
        i, 
        regex = /(bold|italic|underline)? ?(bold|italic|underline)? ?(bold|italic|underline)? ?([0-9\.\,]+)pt (.+)/;     
    res = regex.exec(str);
        
    if (!res || (res && res.length !== 6)) {
      throw new Error("Invalid font: " + str);
    }
 
    // Decode regular expression
    for (i = 0; i < res.length; ++i) {
      if (!res[i]) {
        continue;
      }
    
      if (i === 1 || i === 2 || i === 3) {
        if (res[i] === "bold") {
          bold = true;
        }
        else if (res[i] === "italic") {
          italic = true;
        }
        else if (res[i] === "underline") {
          underline = true;
        }
      }
      else if (i === 4) {
        size = Number(res[4]);
      }
      else if (i === 5) {
        fontName = res[5];
      }
    }
        
    // Create the font
    var font = new niagara.px.Font({
      fontName: fontName,
      size: size,
      bold: bold,
      italic: italic,
      underline: underline,
      str: str      
    });
    
    // Store the font in the Cache for later reference
    fontCache[str] = font;
    return font;
  };
  
  /**
   * Encode the Font to a String
   *
   * @return {String}
   */  
  niagara.px.Font.prototype.encodeToString = function() {
    if (this.$str) {
      return this.$str;
    }
    
    var s = "";
    if (this.bold) {
      s += "bold ";
    }
    if (this.italic) {
      s += "italic ";
    }
    if (this.underline) {
      s += "underline ";
    }
    
    s += String(this.size) + "pt ";
    s += this.fontName;
    
    this.$str = s;
    return s;
  };
  
  /**
   * Default Font instance
   */
  niagara.px.Font.DEFAULT = new niagara.px.Font({
    fontName: "sans-serif",
    size: 12
  });
  
  /**
   * Null Font instance
   */
  fontCache["null"] = niagara.px.Font.NULL = new niagara.px.Font({
    fontName: "0",
    str: "null"
  });
  
  // Register Type
  niagara.px.Font.registerType("gx:Font"); 
  
  /**
   * Return the font name.
   *
   * @return {String}
   */
  niagara.px.Font.prototype.getFontName = function() {
    return this.$fontName;
  };
  
  /**
   * Return the size.
   *
   * @return {Number}
   */
  niagara.px.Font.prototype.getSize = function() {
    return this.$size;
  };
  
  /**
   * Return true if the font is bold.
   *
   * @return {Boolean}
   */
  niagara.px.Font.prototype.getBold = function() {
    return this.$bold;
  };
  
  /**
   * Return true if the font is italic.
   *
   * @return {Boolean}
   */
  niagara.px.Font.prototype.getItalic = function() {
    return this.$italic;
  };
  
  /**
   * Return true if the font is underline.
   *
   * @return {Boolean}
   */
  niagara.px.Font.prototype.getUnderline = function() {
    return this.$underline;
  };
  
  /**
   * Update an HTML DOM element.
   *
   * @param dom the DOM Object to update
   */  
  niagara.px.Font.prototype.update = function(dom) {   
    // TODO: Support for HTML5 web fonts?
  
    // Lazily create the CSS for the font
    if (this.$fontCss === undefined) {
      var fcss = "";
      if (this !== niagara.px.Font.NULL) {      
        if (this.$italic) {
          fcss += "italic ";
        }
        if (this.$bold) {
          fcss += "bold ";
        }
        
        fcss += this.$size + "px " +
                "\"" + this.$fontName + "\", serif";
      
        // Create CSS for changing the font
        this.$fontCss = fcss;               
      }
    }
  
    // Update the DOM's CSS
    dom.css({
      "font": this.$fontCss,
      "text-decoration": (this.$underline ? "underline" : "none")
    });
  };
  
  /**
   * @class Insets.
   * <p>
   * This is currently a partial implementation of Niagara's 'gx:Insets' Type.
   * <p>
   * When creating a Simple, always use the 'make' method instead of creating a new Object.
   *
   * @name niagara.px.Insets
   * @extends baja.Simple
   */
  niagara.px.Insets = function(top, right, bottom, left, str) {
    niagara.px.Insets.$super.apply(this, arguments);
    this.$top = top;
    this.$right = right;
    this.$bottom = bottom;
    this.$left = left;
    this.$str = str;
  }.$extend(baja.Simple);
   
  /**
   * Make Insets.
   */
  niagara.px.Insets.make = function(obj) {  
    throw new Error(notCurrentlySupported);
  };
  
  /**
   * Make Insets.
   */
  niagara.px.Insets.prototype.make = function(obj) {
    return niagara.px.Insets.make.apply(niagara.px.Insets, arguments);
  };
  
  /**
   * Decode Insets from a String
   *
   * @param {String} str
   * @return {niagara.px.Insets}
   */   
  niagara.px.Insets.prototype.decodeFromString = function(str) {    
    if (str === "null") {
      return niagara.px.Insets.NULL;
    }    
    
    var res = /^([^ ]+)? ?([^ ]+)? ?([^ ]+)? ?([^ ]+)?/.exec(str);
    
    if (!res) {
      throw new Error("Error decoding Insets: " + str);
    }
        
    var top, right, bottom, left;
    
    // 1 (shortcut for 1 1 1 1)
    if (res[2] === undefined || res[2] === "") {
      top = right = bottom = left = Number(res[1]);
    }
    // 1 2 (shortcut for 1 2 1 2)
    else if (res[3] === undefined || res[3] === "") {
      top = bottom = Number(res[1]);
      right = left = Number(res[2]);
    }
    // 1 2 3   (shortcut for 1 2 3 2)
    else if (res[4] === undefined || res[4] === "") {
      top = Number(res[1]);
      right = left = Number(res[2]);
      bottom = Number(res[3]);
    }
    // 1 2 3 4 (top right bottom left)
    else if (res[5] === undefined || res[5] === "") {
      top = Number(res[1]);
      right = Number(res[2]);
      bottom = Number(res[3]);
      left = Number(res[4]);
    }
    
    return new niagara.px.Insets(top, right, bottom, left, str);
  };
  
  /**
   * Encode the Insets to a String
   *
   * @return {String}
   */  
  niagara.px.Insets.prototype.encodeToString = function() {
    return this.$str;
  };
  
  /**
   * Default and NULL Insets instance
   */
  niagara.px.Insets.NULL = niagara.px.Insets.DEFAULT = new niagara.px.Insets(0, 0, 0, 0, "null");
  
  // Register Type
  niagara.px.Insets.registerType("gx:Insets"); 
  
  /**
   * Return the top.
   *
   * @returns {Number}
   */
  niagara.px.Insets.prototype.getTop = function() {
    return this.$top;
  };
  
  /**
   * Return the right.
   *
   * @returns {Number}
   */
  niagara.px.Insets.prototype.getRight = function() {
    return this.$right;
  };
  
  /**
   * Return the bottom.
   *
   * @returns {Number}
   */
  niagara.px.Insets.prototype.getBottom = function() {
    return this.$bottom;
  };
  
  /**
   * Return the left.
   *
   * @returns {Number}
   */
  niagara.px.Insets.prototype.getLeft = function() {
    return this.$left;
  };
  
  /**
   * Update the DOM's CSS with the padding.
   *
   * @param dom the DOM element to update
   * @param {Number} [excess] the excess to add onto to any padding by default
   * @param {String} [insetsType] what to update in the CSS (defaults to padding)
   */
  niagara.px.Insets.prototype.update = function(dom, excess, insetsType) {      
    excess = excess || 0;
    insetsType = insetsType || "padding";
    
    var obj = {};
    
    obj[insetsType + "-top"] =  this.$top + excess;
    obj[insetsType + "-right"] =  this.$right + excess;
    obj[insetsType + "-bottom"] =  this.$bottom + excess;
    obj[insetsType + "-left"] =  this.$left + excess;
    
    dom.css(obj);
  };
    
  /**
   * @class Layout.
   * <p>
   * This is currently a partial implementation of Niagara's 'bajaui:Layout' Type.
   * <p>
   * When creating a Simple, always use the 'make' method instead of creating a new Object.
   *
   * @name niagara.px.Layout
   * @extends baja.Simple
   */
  niagara.px.Layout = function(x, xUnit, y, yUnit, w, wUnit, h, hUnit, str) {
    niagara.px.Layout.$super.apply(this, arguments);
    var that = this;
    that.$x = x;
    that.$xUnit = xUnit;
    that.$y = y;
    that.$yUnit = yUnit;
    that.$w = w;
    that.$wUnit = wUnit;
    that.$h = h;
    that.$hUnit = hUnit;
    that.$str = str;
  }.$extend(baja.Simple);
  
  var layoutCache = {},
      ABS = niagara.px.Layout.ABS = 0,
      PERCENT = niagara.px.Layout.PERCENT = 1,
      PREF = niagara.px.Layout.PREF = 2;
   
  /**
   * Make Layout.
   */
  niagara.px.Layout.make = function(obj) {  
    throw new Error(notCurrentlySupported);
  };
  
  /**
   * Make Layout.
   */
  niagara.px.Layout.prototype.make = function(obj) {
    return niagara.px.Layout.make.apply(niagara.px.Layout, arguments);
  };
  
  /**
   * Decode Layout from a String.
   *
   * @param {String} str
   * @return {niagara.px.Layout}
   */   
  niagara.px.Layout.prototype.decodeFromString = function(str) {
    // If we have a layout simple cached then use it.
    if (layoutCache[str]) {
      return layoutCache[str];
    }  
    
    var res,
        x,
        xUnit,
        y,
        yUnit,
        w = 0,
        wUnit,
        h = 0,
        hUnit,
        layout;

    res = /^(fill)|([^,%]+)(%)?,([^,%]+)(%)?,([^,%]+)(%)?,([^,%]+)(%)?$/.exec(str);
    
    // Handle fill
    if (res[1] === "fill") {
      layout = niagara.px.Layout.FILL;
    }
    else {
      // x
      x = parseFloat(res[2]);
      xUnit = res[3] === "%" ? PERCENT : ABS;
      
      // y
      y = parseFloat(res[4]);
      yUnit = res[5] === "%" ? PERCENT : ABS;
      
      // width
      if (res[6] === "pref") {
        wUnit = PREF;
      }
      else {
        w = parseFloat(res[6]);
        wUnit = res[7] === "%" ? PERCENT : ABS;
      }
      
      // height
      if (res[8] === "pref") {
        hUnit = PREF;
      }
      else {
        h = parseFloat(res[8]);
        hUnit = res[9] === "%" ? PERCENT : ABS;
      }
      
      layout = new niagara.px.Layout(x, xUnit, y, yUnit, w, wUnit, h, hUnit, str);
    }
    
    // Cache the decoded result
    layoutCache[str] = layout;
    return layout;
  };
  
  /**
   * Encode the Layout to a String.
   *
   * @return {String}
   */  
  niagara.px.Layout.prototype.encodeToString = function() {
    return this.$str;
  };
  
  /**
   * Default Layout instance.
   */
  niagara.px.Layout.FILL = niagara.px.Layout.DEFAULT = new niagara.px.Layout(0, ABS, 0, ABS, 100, PERCENT, 100, PERCENT, "fill");
  
  // Register Type
  niagara.px.Layout.registerType("bajaui:Layout"); 
  
  /**
   * Return the X.
   *
   * @returns {Number}
   */
  niagara.px.Layout.prototype.getX = function() {
    return this.$x;
  };
  
  /**
   * Return the X Unit.
   *
   * @returns {Number}
   */
  niagara.px.Layout.prototype.getXUnit = function() {
    return this.$xUnit;
  };
  
  /**
   * Return the Y.
   *
   * @returns {Number}
   */
  niagara.px.Layout.prototype.getY = function() {
    return this.$y;
  };
  
  /**
   * Return the Y Unit.
   *
   * @returns {Number}
   */
  niagara.px.Layout.prototype.getYUnit = function() {
    return this.$yUnit;
  };
  
  /**
   * Return the width.
   *
   * @returns {Number}
   */
  niagara.px.Layout.prototype.getWidth = function() {
    return this.$w;
  };
  
  /**
   * Return the width unit.
   *
   * @returns {Number}
   */
  niagara.px.Layout.prototype.getWidthUnit = function() {
    return this.$wUnit;
  };
  
  /**
   * Return the height.
   *
   * @returns {Number}
   */
  niagara.px.Layout.prototype.getHeight = function() {
    return this.$h;
  };
  
  /**
   * Return the height unit.
   *
   * @returns {Number}
   */
  niagara.px.Layout.prototype.getHeightUnit = function() {
    return this.$hUnit;
  };
  
  /**
   * Updates the given DOM element with the corresponding CSS for the layout.
   *
   * @param dom the DOM element to update.
   */
  niagara.px.Layout.prototype.update = function (dom) {
    var width = "0",
        height = "0";
    
    if (this.$wUnit === PERCENT) {
      width = this.$w + "%";
    }
    else if (this.$wUnit === ABS) {
      width = this.$w;
    }
    
    if (this.$hUnit === PERCENT) {
      height = this.$h + "%";
    }
    else if (this.$hUnit === ABS) {
      height = this.$h;
    }
    
    // Absolutely position the child according to the layout information
    dom.css({
      "position": "absolute",
      "top": this.$yUnit === PERCENT ? this.$y + "%" : this.$y,
      "left": this.$xUnit === PERCENT ? this.$x + "%" : this.$x,
      "width": width,
      "height": height
    });
  };
  
  /**
   * @class Size.
   * <p>
   * This is currently a partial implementation of Niagara's 'gx:Size' Type.
   * <p>
   * When creating a Simple, always use the 'make' method instead of creating a new Object.
   *
   * @name niagara.px.Size
   * @extends baja.Simple
   */
  niagara.px.Size = function(w, h, str) {
    niagara.px.Size.$super.apply(this, arguments);
    var that = this;
    that.$w = w;
    that.$h = h;
    that.$str = str;
  }.$extend(baja.Simple);
  
  var sizeCache = {};
   
  /**
   * Make Size.
   */
  niagara.px.Size.make = function(obj) {  
    throw new Error(notCurrentlySupported);
  };
  
  /**
   * Make Size.
   */
  niagara.px.Size.prototype.make = function(obj) {
    return niagara.px.Size.make.apply(niagara.px.Size, arguments);
  };
  
  /**
   * Decode Size from a String.
   *
   * @param {String} str
   * @return {niagara.px.Size}
   */   
  niagara.px.Size.prototype.decodeFromString = function(str) {
    // If we have a simple cached then use it.
    if (sizeCache[str]) {
      return sizeCache[str];
    }  
    
    var res,
        w,
        h,
        size;

    res = /^([^,]+),([^,]+)$/.exec(str);
    
    w = parseFloat(res[1]);
    h = parseFloat(res[2]);
    
    size = new niagara.px.Size(w, h, str);
    
    // Cache the result
    sizeCache[str] = size;
    return size;
  };
  
  /**
   * Encode the Size to a String.
   *
   * @return {String}
   */  
  niagara.px.Size.prototype.encodeToString = function() {
    return this.$str;
  };
  
  /**
   * Default Size instance.
   */
  niagara.px.Size.DEFAULT = niagara.px.Size.NULL = new niagara.px.Size(0, 0, "0,0");
  
  // Register Type
  niagara.px.Size.registerType("gx:Size"); 
  
  /**
   * Return the width.
   *
   * @returns {Number}
   */
  niagara.px.Size.prototype.getWidth = function() {
    return this.$w;
  };
  
  /**
   * Return the height.
   *
   * @returns {Number}
   */
  niagara.px.Size.prototype.getHeight = function() {
    return this.$h;
  };
    
}(baja));


