/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions for calculating the gradients on a property
 * sheet status swatch.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, browser: true, vars: true, plusplus: true */

/*globals $, baja, niagara */

(function statusGradients() {
  'use strict';
  
  var util = niagara.util,
      colorUtil = util.color,
      gradientUtil = util.gradient,
      
      hsvToRgb = colorUtil.hsvToRgb,
      
      cssCache = {}, // Cache CSS generation.
      
      LINEAR_GRADIENT_PREFS = ['-webkit', '-moz', '-ms', '-o' ],
      BOX_SHADOW_PREFS = [ '-webkit-', '-moz-', '' ];
      
      
      
  /**
   * Return the background color defined in the baja lexicon for this status.
   * @private
   * @inner
   * @param {baja.Status} status
   * @returns a pure hex string, e.g. "FFFFFF", or undefined if no color
   * defined for this status
   */
  function getBg(status) {
    if (!status) {
      return;
    }
    
    var nStatus = niagara.view.status;
   
    if (status.isDisabled()) {
      return nStatus.disabled;
    }
    if (status.isFault()) {
      return nStatus.fault;
    }
    if (status.isDown()) {
      return nStatus.down;
    }
    if (status.isAlarm()) {
      return nStatus.alarm;
    }
    if (status.isStale()) {
      return nStatus.stale;
    }
    if (status.isOverridden()) {
      return nStatus.overridden;
    } 
  }
  
  /**
   * @private
   * @inner
   * @ignore
   * @param {Array} stops an array of percentage - color pairs
   * @returns {Array} an array of linear gradient CSS properties to set
   * as "background-image"
   */
  function calcLinearGradients(cssData, stops) {
    var prefs = LINEAR_GRADIENT_PREFS,
        bkImg = [],
        angle = 270,
        obj = gradientUtil.createLinearGradientCss(angle, stops),
        i;

    bkImg.push("-webkit-gradient" + obj.webkitGradient);
    
    for (i = 0; i < prefs.length; i++) {
      bkImg.push(prefs[i] + "-linear-gradient" + obj.linearGradientNonStandard);
    }
    
    bkImg.push('linear-gradient', obj.linearGradient);
    
    cssData.css.background = obj.background;
    cssData.css.filter = obj.filter;
    cssData.bkImg = bkImg;
    return cssData;
  }
  
  /**
   * Applies all the appropriate CSS styling to the <code>.statusBg</code>
   * element that displays the status color.
   * 
   * @param {jQuery} element
   * @param {baja.Status} status
   */
  function applyStatusCSS(element, status) {
    var bg = getBg(status),
        cssData,
        j;

    element.toggle(!!bg);    
        
    if (!bg) {
      return;
    }
    
    cssData = cssCache[bg];
    
    // Generate the cached css for the specified background color.
    if (!cssData) {
      cssData = cssCache[bg] = {
        css: {},
        bkImg: []
      };

      var color = colorUtil.parseRgba(bg),
          h = color.h,
          s = color.s,
          i = color.v,
          stop1 = hsvToRgb(h, s * 0.5, Math.max(i, 0.85)), //highlight
          stop2 = hsvToRgb(h, s * 1.1, i), //actual status color
          stop3 = hsvToRgb(h, s * 1.2, i * 0.7), //slight darker gradient
          stop4 = hsvToRgb(h, s * 0.7, i * 0.55), //bottom shadow
          stops = [
            [ "0%", stop1 ],
            [ "10%", stop2 ],
            [ "90%", stop3 ],
            [ "100%", stop4 ]
          ],
          border = '1px solid ' + stop4.css,
          prefs = BOX_SHADOW_PREFS;
      
      cssData.css['background-color'] = bg;
      cssData.css.border = border;
      
      for (j = 0; j < prefs.length; j++) {
        cssData.css[prefs[j] + 'box-shadow'] = "0 0 2x " + stop4.css;
      }
      
      cssData = calcLinearGradients(cssData, stops);
    }
    
    // Apply CSS to the element
    element.css(cssData.css);
    for (j = 0; j < cssData.bkImg.length; ++j) {
      element.css('background-image', cssData.bkImg[j]);
    }
  }
  
  util.api('niagara.statusGradients', {
    applyStatusCSS: applyStatusCSS
  });
}());