/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to the Niagara Mobile Property Sheet
 * user interface, HTML generation, event handling, etc.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, browser: true */

/*globals $, baja, niagara, unescape */


/**
 * @namespace
 * @name niagara.propSheet.ui
 */
(function propsheetUI() {
  "use strict";

  niagara.util.require(
    'niagara.util',
    'niagara.util.mobile.commands',
    'niagara.util.mobile.views',
    'niagara.util.mobile.dialogs',
    'niagara.util.nav',
    'niagara.propSheet.PropertySheet',
    'niagara.util.mobile.PageViewManager'
  );
  
  var util = niagara.util,
      dialogs = util.mobile.dialogs,
      views = util.mobile.views,
      commands = util.mobile.commands,
      pages = util.mobile.pages,
      PageViewManager = util.mobile.PageViewManager,
      PropertySheet = niagara.propSheet.PropertySheet,
      ComponentSpaceNavModel = util.nav.ComponentSpaceNavModel,
//      NavFileNavModel = util.nav.NavFileNavModel,
      navMgr,
      actionContentHtml;
  
  actionContentHtml =
    '<div class="actionContent">\n' +
      '<div class="editorHeader">\n' +
        '<span class="actionComponentName">{componentName}</span>.<span class="actionName">{actionName}</span>\n' +
      '</div>\n' +
      '<div class="actionParams"></div>\n' +
    '</div>';
  
  function validationError(err) {
    dialogs.ok({
      content: err,
      title: baja.lex('mobile').get({
        key: 'propsheet.validationError',
        def: 'Validation Error'
      })
    });
  }
  
  /**
   * Given a property sheet object, adds the appropriate jQuery event handlers
   * to enable navigation and editing abilities.
   * 
   * @private
   * @memberOf niagara.propSheet.ui
   * @param {niagara.propSheet.PropertySheet} sheet the sheet to enable
   * jQuery events on
   */
  function armPropSheetHandlers(sheet) {
    var sub = sheet.contentView.subscriber;
    
    sub.attach('renamed', function (prop, oldName, cx) {
      navMgr.navbar.refresh(navMgr.selectedView.getPage());
    });
    
    sub.attach('flagsChanged facetsChanged', function (slot, cx) {
      sheet.refresh();
    });
    
    sub.attach('added removed changed', function (prop) {
      if (prop.getName() === "displayNames") {
        sheet.refresh();
      }
    });
  }
  
  /**
   * After creating a new property sheet, this method will enable its event
   * handlers and start its subscription (makes the property sheet interactive,
   * in other words).
   * 
   * @private
   * @memberOf niagara.propSheet.ui
   * @param {niagara.propSheet.PropertySheet} sheet the sheet to be initialized
   */
  function startSheet(sheet) {
    sheet.start();
  }
  

  


  function PropertySheetManager() {
    var that = this,
        navModel;
    
//    if (baja.nav.navfile.getRootNode()) { //we have a nav file!
//      navModel = new NavFileNavModel();
//    } else {
      navModel = new ComponentSpaceNavModel();
//    }
        
    
    PageViewManager.call(that, navModel);
    
    that.navbar = new niagara.util.mobile.nav.NavBar(navModel);
    that.navbar.attach('click', function (obj) {
      var node = obj.node,
          children = node.getNavChildren(),
          linkActive = util.ord.equivalent(node.getNavOrd(), util.ord.deriveOrd(that.selectedView.value));
      
      if (linkActive && children.length) {
        that.showNavChildren(node);
      } else {
        that.linkToOrd(node.getNavOrd());
      }
    });
  }
  
  PropertySheetManager.prototype = new niagara.util.mobile.PageViewManager();
  
  PropertySheetManager.prototype.instantiateView = function instantiateView(component, pageData, callbacks) {
    callbacks.ok(new PropertySheet());
  };
  
  util.aop.advisePrototype(PropertySheetManager, {
//    before: {
//      confirmAbandonChanges: function (args) {
//        var options = args[0] || {};
//        options.fail = validationError;
//        return [options];
//      }
//    },
    after: {
      attachSubscriberEvents: function (args, noReturn) {
        var sheet = args[0];
        armPropSheetHandlers(sheet);
      },
      setSelectedView: function setSelectedView(args, noReturn) {
        var sheet = args[0];
        this.navbar.refresh(sheet.getPage());
      }
    }
  });
  
  
  /**
   * Populates and shows the JQM page with a list of the actions that can be
   * fired on the currently selected property sheet's component.
   * 
   * @private
   * @memberOf niagara.propSheet.ui
   */
  function showAvailableActions() {
    var selectedView = navMgr.selectedView,
        value = selectedView.value;
    
    commands.getAvailableActionsCmd().invoke(value);
  }
  
  /**
   * Arms event handlers on UI components, like the save/refresh/actions buttons
   * on the footer bar and the submit/cancel buttons on the action select/fire
   * divs.
   * 
   * @private
   * @memberOf niagara.propSheet.ui
   */
  function initializeUI() {
    navMgr = new PropertySheetManager();
    
    $('a.saveLink').live('click', function () {
      var $this = $(this);

      navMgr.saveSelected({
        ok: function () {
          $this.addClass('ui-disabled');
          navMgr.refreshSelected();
        },
        fail: validationError
      });
    });
    
    $("a.commandsButton").live('onShowCommands', function() {
      views.getViewsCommand().setOrd(util.ord.deriveOrd(navMgr.selectedView.value));
    });
    
    $('a.refreshLink').live('click', function () {
      navMgr.refreshSelected();
    });
    
    $('a.actionsLink').live('click', function () {
      showAvailableActions();
    });
    
    pages.register('splash', {
      pageshow: function () {
        $.mobile.changePage(baja.Ord.make(niagara.view.ord).toUri(), { 
          transition: "none", 
          changeHash: false 
        });
      }
    });
    
    navMgr.registerPages();

    //toolbar toggle is kind of silly to have in the propsheet app since
    //basically the entire screen is links
    $.mobile.fixedToolbars.setTouchToggleEnabled(false);
  }
  
  initializeUI();
  
  util.api('niagara.propSheet.ui', {
    'public': {
      initializeUI: initializeUI
    },
    'private': {
      armPropSheetHandlers: armPropSheetHandlers,
      showAvailableActions: showAvailableActions,
      startSheet: startSheet
    }
  });
}());