
//Copyright 2011, Tridium, Inc. All Rights Reserved.


/**
 * Niagara Mobile History
 *
 * @author Gareth Johnson
 * @author Jason Spangler
 * @version 0.1.0.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: true, nomen: true, eqeqeq: true, bitwise: true, regexp: true, newcap: true, immed: true, strict: false, vars: true, browser: true */

/*global $, baja, niagara, window, localStorage, update, dojo*/ 

(function history(pages) {

  // Use ECMAScript 5 Strict Mode
  "use strict";  
  
  niagara.util.require(
      '$.mobile',
      'niagara.util.View',
      'niagara.fieldEditors',
      'niagara.util.mobile.commands',
      'niagara.util.mobile.dialogs',
      'niagara.util.mobile.history'
    );

      //imports
  var util = niagara.util,
      mobileUtil = util.mobile,
      ListView = mobileUtil.ListView,
      PageView = mobileUtil.PageView,
      PageViewManager = mobileUtil.PageViewManager,
      historyUtil = mobileUtil.history,
      commands = mobileUtil.commands,
      dialogs = mobileUtil.dialogs,
      Command = commands.Command,
      fieldEditors = niagara.fieldEditors,
      
      
      //local vars
      mobileLex = util.lazyLex('mobile'),
      historyManager,
      HistoryNavListView,
      HistoryNavPageView,
      lex,
      timeRangeOnQuery = {},
      navPageTitles = {},
      
      //constants
      TABLE_BTN_DISPLAY_KEY = 'history.table.select',
      CHART_BTN_DISPLAY_KEY = 'history.chart.select',
      TIME_RANGE_BTN_DISPLAY_KEY = 'history.timeRange.select',
      START_TIME_FACET_NAME = 'START_TIME',
      END_TIME_FACET_NAME = 'END_TIME',
      PARAM_STRING_FACET_NAME = 'PARAM_STRING',
      TIME_RANGE_TYPE = 'baja:AbsTimeRange',
      NUMERIC_TREND_RECORD_TYPE = 'history:NumericTrendRecord',
      ENUM_TREND_RECORD_TYPE = 'history:EnumTrendRecord',
      BOOLEAN_TREND_RECORD_TYPE = 'history:BooleanTrendRecord',
      MOBILE_HISTORY_APP_TYPE = 'mobile:MobileHistoryApp',
      CHART_VIEW_ID = 'mobile:MobileHistoryAppChartView',
      TABLE_VIEW_ID = 'mobile:MobileHistoryAppTableView',
      FOLDER_TYPE = 'baja:Folder',
      SSH_TYPE = 'mobile:HistoryServerSideHandler',
      MTD_HISTORY_NAV_LIST = 'getHistoryNavList',
      MTD_NAV_CHILDREN = 'getNavChildren',
      MTD_ORD_DISPLAY_NAME = 'getOrdDisplayName',
      MTD_LIST_DEVICE_HISTORIES = 'getDeviceHistories',
      HISTORY_BTN_HTML = "<a data-role='button' class='ui-btn-up-c " +
                           "ui-btn-corner-all' data-inline='true' " +
                           "data-icon='{icon}' data-iconpos='notext' " +
                           "data-theme='c'>{id}</a>",
      HOME_BTN_HTML = '<a class="appHome ui-btn ui-btn-inline ' +
                        'ui-btn-icon-notext ui-btn-corner-all ui-shadow ' +
                        'ui-btn-up-a" href="#" title="$appHomeBtn" ' +
                        'data-icon="home" data-iconpos="notext" ' +
                        'data-inline="true" data-role="button" data-theme="a">' +
                        '<span class="ui-btn-inner ui-btn-corner-all">' +
                          '<span class="ui-btn-text">&nbsp;</span>' +
                          '<span class="ui-icon ui-icon-home ui-icon-shadow"></span>' +
                        '</span>' +
                      '</a>';
  
  /**
   * Default ok callback handler.
   *
   * @private
   */
  function defOk() {    
    $.mobile.hidePageLoadingMsg();
  }

  /**
   * Default fail callback handler.
   *
   * @private
   */
  function defFail(err) {
    baja.error(err);

    // Hide any loading title      
    $.mobile.hidePageLoadingMsg();

    // Load error page     
    $("<a href='#error' data-rel='dialog' data-transition='flip' />").click();  
  }

  function getMobileLex() {
    if (!lex) {
      lex = baja.lex("mobile");
    }
    return lex;
  }
  
  /**
   * This method redirects the current page to the given view ord.
   * 
   * @param el - HTML element representing our history slot.
   * @param viewQueryId - String view parameter - history or table
   */
  function redirect(el, viewQueryId) {
    var editor,
        li = el.parents('li'),
        expandedDiv = li.next('.expanded'),
        queryOn = li.find('.timeButton.ui-checkbox-on').length > 0,
        historyId = el.jqmData('historyId'),
        historyGroupRegex = /\/\/.*\/\//,
        query = historyId.$ord.replace(historyGroupRegex,"/"); 

    function doLink() {
      window.location.assign("/ord?" + query + "%7Cview:" + viewQueryId);
    }
    
    if (queryOn) {
      editor = expandedDiv.find('.editor.baja-String').data('view');
      editor.validate({
        ok: function () {
          editor.getSaveData(function (paramString) {
            query += paramString.replace(/\+/g, "%2B");
            doLink();
          });
        },
        fail: dialogs.error
      });
    } else {
      doLink();
    }
  }
  
  
////////////////////////////////////////////////////////////////
//  History Device List View
////////////////////////////////////////////////////////////////
  
  /**
   * Create a unordered list of the devices and history group folders
   * within the base history space of our station. Each item in the list
   * represents either a history device or a history grouping that the
   * user can navigate through in order to reach specific history objects.
   * 
   * @param component
   * @param targetElement
   * @returns {HistoryNavListView}
   */
  HistoryNavListView = ListView.subclass();

  HistoryNavListView.prototype.shouldIncludeSlot = function shouldIncludeSlot(slot) {
    return slot.getType().is(FOLDER_TYPE);
  };

  /**
   * create the first level of list items for display. These are the 
   * history groups and devices objects.
   * 
   * @param deviceId - String name of the device to which the histories belong.
   */
  HistoryNavListView.prototype.makeListItemParams = function (row,slotName) {
    if (row.get('id') !== null) {
      timeRangeOnQuery[row.get('id')] = false;
    }
    
    return {
      ord: row.get('ord') !== null ? row.get('ord') : "history:" + row.get('id'),
      title: row.get('displayName') !== null ? row.get('displayName') : row.get('id'),
      id: slotName,
      recordType: row.get('recordType'),
      linkable: row.get('id') === null
    };
  };
  
  /**
   * This callback is used to populate our expanded div created by our
   * doExpand method call when the time query button is clicked. This will pull
   * the start/end time facets off the slot, and instantiate a field editor for
   * a history query param string (either with start/end or period params).
   */
  HistoryNavListView.prototype.populateExpandedDiv = function (prop, expandedDiv) {
    var that = this,
        paramDiv = $('<div class="property"/>'),
        facets = prop.getFacets(),
        start = facets.get(START_TIME_FACET_NAME) || baja.AbsTime.DEFAULT,
        end = facets.get(END_TIME_FACET_NAME) || baja.AbsTime.now(),
        range = baja.$(TIME_RANGE_TYPE, { startTime: start, endTime: end }),
        recordType = expandedDiv.prev().data('params').recordType;
    
    recordType = baja.lt(recordType.toString());

    fieldEditors.makeFor({ 
      value: historyUtil.toParamString(range), 
      showDelta: recordType.is(NUMERIC_TREND_RECORD_TYPE),
      key: 'historyQueryParams' 
    }, function (editor) {
      expandedDiv.append(paramDiv);
      editor.buildAndLoad(paramDiv);
    });
  };
  
  /**
   * Create history list items that display graph, table, and time range
   * buttons. 
   */
  HistoryNavListView.prototype.makeListItem = function (params, slot) {
    var li = $('<li class="property"/>').attr('id', params.id || ''),
        recordType = params.recordType,
        that = this,
        btnContainer = $('<div data-role="controlgroup" data-type="horizontal" />').addClass('historyViewSelector'),
        historyId = params.ord,
        isHistoryItem = recordType !== null,
        a;
    
    li.data('params', params);
    
    $('<label class="title" />') //label with property name
      .text(params.title || '')
      .addClass( isHistoryItem ? 'historyViewSelectorLabel' : '')
      .appendTo(li);
  
    $('<div class="display" />') 
      .text(params.display || '')
      .appendTo(li);
  
    if (params.linkable || params.expandable) {
      a = $('<a />');
      a.jqmData('ord', params.ord);
      a.jqmData('title', params.title);
      navPageTitles[params.ord] = params.title;
      
      if (params.expandable) {
        a.addClass('expandable');
        li.attr('data-icon', 'arrow-d');
      } else {
        a.addClass('link');
      }
      li.wrapInner(a);
    }
     
    // check if our list item represents a history
    if (isHistoryItem) {
      recordType = baja.lt(recordType.toString()); //ensure a Type object
      btnContainer.appendTo(li);
      li.css('overflow', 'auto');
      
      // add chart button if our history record type is a trend record
      if (recordType.is(NUMERIC_TREND_RECORD_TYPE) ||
          recordType.is(ENUM_TREND_RECORD_TYPE) ||
          recordType.is(BOOLEAN_TREND_RECORD_TYPE)) {
        $(HISTORY_BTN_HTML.patternReplace({ 
          id: mobileLex.get(CHART_BTN_DISPLAY_KEY), 
          icon: 'chart' 
        }))
          .addClass('chartButton')
          .jqmData('historyId', historyId)
          .appendTo(btnContainer)
          .click(function () {
            redirect($(this), CHART_VIEW_ID);
          });
      }
      
      //create table button
      $(HISTORY_BTN_HTML.patternReplace({ 
        id: mobileLex.get(TABLE_BTN_DISPLAY_KEY), 
        icon: 'grid' 
      }))
        .addClass('tableButton')
        .jqmData('historyId', historyId)
        .appendTo(btnContainer)
        .click(function () {
          redirect($(this), TABLE_VIEW_ID);
        });
      
      //create time query button
      $(HISTORY_BTN_HTML.patternReplace({
        id: mobileLex.get(TIME_RANGE_BTN_DISPLAY_KEY), 
        icon: 'gear' 
      }))
        .addClass('timeButton ui-checkbox-off')
        .appendTo(btnContainer)
        .click(function () {
          that.doExpand(slot.getName());
          $(this).toggleClass('ui-checkbox-off ui-checkbox-on ui-btn-active');
          timeRangeOnQuery[slot.getName()] = $(this).hasClass('ui-checkbox-on');
        });
    }
    
    return li;
  };

  
  /**
   * Create the history nav list view
   */
  HistoryNavListView.prototype.makeListView = function (value, callbacks) {
    var ul = $('<ul data-role="listview" data-theme="c"/>'),
        type = value.getType(),
        that = this,
        listBuilder = function (row) {
          var params = that.makeListItemParams(row, row.getPropertyInParent().getName());
          ul.append(that.makeListItem(params, row.getPropertyInParent()));
        };
    
    if (type.is(MOBILE_HISTORY_APP_TYPE)) {
      //use server side call to resolve our groups and history devices
      value.serverSideCall({
        typeSpec: SSH_TYPE,
        methodName: MTD_HISTORY_NAV_LIST,
        ok: function (val) {
          val.getSlots().is('baja:Folder').eachValue(function (folder) {
            listBuilder(folder);
          });
          callbacks.ok(ul);
        },
        fail: callbacks.fail
      });
    } else {
      //iterate through our box collection rows and create list items
      value.getSlots().is('baja:Folder').eachValue(listBuilder);
      callbacks.ok(ul);
    }
  };
  
  util.aop.advisePrototype(HistoryNavListView, {
    after: {
      armHandlers: function () {
        this.$dom.bind('editorchange', function (event) {
          return false;
        });
      }
    }
  });
  
////////////////////////////////////////////////////////////////
//  History Nav Page View
////////////////////////////////////////////////////////////////
  
  /**
   * The history nav page view is the page view implementation of the 
   * history navigation view. 
   * 
   * @param {baja.Ord} ord the ORD of the nav view.
   */
  HistoryNavPageView = PageView.subclass(function HistoryNavPageView(ord) {
    var that = this;
    baja.strictArg(ord,'baja:String');
    this.$title = navPageTitles[ord];
    this.ord = ord;
    
   
    if (this.$title === undefined) {
      util.ord.get(niagara.view.app.appOrd, function (component) {
        component.serverSideCall({
          typeSpec: SSH_TYPE,
          methodName: MTD_ORD_DISPLAY_NAME,
          value: baja.Ord.make(ord),
          ok: function (val) {
            that.$title = val;
            if (that.$dom) {
              that.getHeaderDiv().find('h1').text(val);
            }
          }
        });
      });
    }
  });
  
  HistoryNavPageView.prototype.instantiateContentView = function () {
    return new HistoryNavListView();
  };
  
  /**
   * This method initializes our commands button in the header bar and is called
   * after the DOM is initialized.
   */
  util.aop.after(HistoryNavPageView.prototype, 'createPage', function (args, page) {
    var homeBtn = commands.getCommandsButton(), 
        cmds = commands.getDefaultCommands(),
        homeIndex = util.indexOf(cmds,commands.getHomeCmd()),
        pageHeader = page.children(':jqmData(role=header)');
        
    pageHeader.append(homeBtn);    
    homeBtn.click(commands.showCommandsHandler);
    
    if (homeIndex >= 0) {
      cmds[homeIndex] = new commands.Command(getMobileLex().get('commands.home'), function () {
        util.mobile.linkToOrd(baja.getUserHome());
      });
    }
  });

  
////////////////////////////////////////////////////////////////
// History View Manager
////////////////////////////////////////////////////////////////
  
  /**
   * The component view manager for the History app.
   * @returns {HistoryPageViewManager}
   */
  function HistoryPageViewManager() {
    PageViewManager.call(this);
  }
  HistoryPageViewManager.prototype = new PageViewManager();

  util.aop.after(HistoryPageViewManager.prototype, 'createPagesHandlers', function (args, handlers) {
	  handlers.getCommands = function (obj) {
      return commands.getDefaultCommands();
	  };
  });
  
  /**
   * This method is used to instantiate the view objects for this app. 
   * 
   * The table and chart views are checked for to instantiate DataTablePageView
   * and DataChartView objects with the ORD values necessary to implement
   * them.
   */
  HistoryPageViewManager.prototype.instantiateView = function (component, pageData, callbacks) {
    var type = component.getType();
    
    if (type.is(MOBILE_HISTORY_APP_TYPE) || 
        component.get('resultType') === 'deviceList' ||
        component.get('resultType') === 'navNodeList') {
      return callbacks.ok(new HistoryNavPageView(pageData.ord));
    } 
    
    throw new Error("Could not instantiate view for type " + type);
  };
  
  /**
   * The doResolveOrd method is used to intercept a given ORD at time of 
   * resolution and perform any necessary server side calls that are 
   * particular to our history app.
   */
  HistoryPageViewManager.prototype.doResolveOrd = function doResolveOrd(ord, callbacks) {
    var historyListRegex = /history:\/[^\W\/]*/,
        tableViewRegex = /display=table/,
        navNodeRegex = /history:\/\//,
        historyRootRegex = /history:\/\/$/,
        historyRootMatch = ord.match(historyRootRegex),
        tableMatch = ord.match(tableViewRegex),
        navNodeMatch = ord.match(navNodeRegex),
        match = ord.match(historyListRegex),
        deviceId;
    
    //append final slash for special history root case
    if (historyRootMatch) {
      ord += "/";
    }
    
    if (match && match[0].length === ord.length) {
      //ord is a list of device histories for a particular device ID
      deviceId = ord.substring(ord.indexOf('/') + 1);
      util.ord.get(niagara.view.app.appOrd, function (component) {
        component.serverSideCall({
          typeSpec: SSH_TYPE,
          methodName: MTD_LIST_DEVICE_HISTORIES,
          value: deviceId,
          ok: function (result) {
            result.add({
              slot: 'resultType',
              value: 'deviceList',
              ok: function () {
                callbacks.ok(result);
              },
              fail: callbacks.fail
            });
          },
          fail: callbacks.fail
        });
      });
    } else if (tableMatch) {
      callbacks.ok(baja.Ord.make(ord));
    } else if (navNodeMatch){
      util.ord.get(niagara.view.app.appOrd, function (component) {
        component.serverSideCall({
          typeSpec: SSH_TYPE,
          methodName: MTD_NAV_CHILDREN,
          value: baja.Ord.make(ord),
          ok: function (result) {
            result.add({
              slot: 'resultType',
              value: 'navNodeList',
              ok: function () {
                callbacks.ok(result);
              },
              fail: callbacks.fail
            });
          },
          fail: callbacks.fail
        });
      });
    } else {
      baja.Ord.make(ord).get(callbacks);
    }
  };

////////////////////////////////////////////////////////////////
//  Main
////////////////////////////////////////////////////////////////
    
  function initializeUI() {
    pages.register("loading", (function loading() {   
      function pageshow() {
        // Ensure this function is only called once BajaScript has started
        $.mobile.changePage(baja.Ord.make(niagara.view.ord).toUri(), { 
          transition: "slide",
          changeHash: false
        });  
      }
      
      // Functions exported as public
      return {
        pageshow: pageshow
      };
    }()));
    
    historyManager = new HistoryPageViewManager();
    historyManager.registerPages(); 
  }
  
  initializeUI();
   
}(niagara.util.mobile.pages));