/*
 * Copyright 2004 Tridium, Inc. All Rights Reserved.
 */
package appliance.ui;

import java.io.*;
import java.util.*;
import javax.baja.sys.*;
import javax.baja.file.*;
import javax.baja.naming.*;
import javax.baja.nav.*;
import javax.baja.status.*;
import javax.baja.util.*;
import javax.baja.gx.*;
import javax.baja.ui.*;
import javax.baja.ui.event.*;
import javax.baja.ui.pane.*;
import javax.baja.ui.tree.*;
import javax.baja.workbench.*;
import javax.baja.workbench.sidebar.*;
import javax.baja.workbench.nav.menu.*;
import javax.baja.schedule.*;
import appliance.*;
import appliance.driver.*;

/**
 * BApplianceSideBar.
 *
 * @author    Brian Frank       
 * @creation  18 Nov 04
 * @version   $Revision$ $Date$
 * @since     Niagara 3.0
 */
public class BApplianceSideBar  
  extends BWbSideBar  
  implements NavListener
{   

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BApplianceSideBar.class);

////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////

  public BApplianceSideBar()
  {
    model = new Model();
    tree = new BTree(model);
    tree.setController(new Controller());
    setContent(new BTreePane(tree));   
    updateRoots();
  }

////////////////////////////////////////////////////////////////
// Model
////////////////////////////////////////////////////////////////

  public void started()
  {
    BNavRoot.INSTANCE.addNavListener(this);
  }

  public void stopped()
  {                  
    BNavRoot.INSTANCE.removeNavListener(this);
  }
  
  public boolean hasCloseCommand()
  {
    return false;
  }

  public synchronized void navEvent(NavEvent event)
  {                 
    // if event on hosts or sessions, update roots                  
    BINavNode src = event.getParent();     
    if (src instanceof BHost || src instanceof BISession)
      updateRoots();                       
    
    // if a replace on NavFile.nav, then update session root
    if (event.getOldChildName() != null && event.getOldChildName().equals("NavFile.nav"))
    {
      SessionNode node = findRoot(event.getParentOrd().toString());
      if (node != null)
        node.reloadNavFile();
    }
  }           
  
  public void updateRoots()
  {      
    // find all the fox sessions to stations
    Array acc = new Array();
    BHost[] hosts = BHost.getAllHosts();
    for(int i=0; i<hosts.length; ++i)
    {
      BISession[] sessions = hosts[i].getChildSessions();
      for(int j=0; j<sessions.length; ++j)
      {                   
        BISession session = sessions[j];
        if (!session.getType().getTypeName().equals("FoxSession")) continue;
        SessionNode node = findRoot(session); // see if there is an existing node
        if (node == null) node = new SessionNode(model, session);        
        acc.add(node);
      }
    }                
    roots = acc;             
    model.updateTree(); 
  }                 
  
  public SessionNode findRoot(BISession session)
  {
    for(int i=0; i<roots.size(); ++i)
    {
      SessionNode node = (SessionNode)roots.get(i);
      if (node.session == session) 
      {
        node.kids = null; // force reload
        return node;
      }
    }
    return null;
  }

  public SessionNode findRoot(String ord)
  {                           
    for(int i=0; i<roots.size(); ++i)
    {
      SessionNode node = (SessionNode)roots.get(i);
      if (ord.startsWith(node.session.getNavOrd().toString()))
        return node;
    }
    return null;
  }

  public class Model
    extends TreeModel
  {                 
    public int getRootCount() { return roots.size(); }
    public TreeNode getRoot(int index) { return (TreeNode)roots.get(index); }
  }
  
////////////////////////////////////////////////////////////////
// Node
////////////////////////////////////////////////////////////////

  public class Node
    extends TreeNode
  {
    public Node(TreeModel model, Node parent, BINavNode navNode)
    {
      super(model, parent);
      this.navNode = navNode;      
      this.ord    = navNode.getNavOrd();
      this.icon   = BImage.make(navNode.getNavIcon());
    }                 
    
    public String getText()
    {
      return navNode.getNavDisplayName(null);
    }

    public BImage getIcon() 
    {
      return icon;
    }

    public boolean hasChildren()
    {
      return navNode.hasNavChildren();
    }
    
    public int getChildCount()
    {              
      return load().length;
    }
      
    public TreeNode getChild(int index)
    {
      return load()[index];
    }           
    
    public BMenu getMenu()
    {                                             
      BObject target = (BObject)navNode;
      return BNavMenuAgent.makeFor(target).makeMenu(getTree(), target);      
    }
    
    final Node[] load()
    {
      if (this.kids == null)
      {       
        try
        {                          
          Node[] kids = doLoad(navNode);
          if (oldKids != null)
          {
            preserve(oldKids, kids);
            oldKids = null;
          }    
          this.kids = kids;
        }
        catch(Exception e)
        {
          e.printStackTrace();
          kids = new Node[0];
        }
      }
      return kids;
    }          
    
    Node[] doLoad(BINavNode navNode)
      throws Exception
    {    
      BINavNode[] navKids = navNode.getNavChildren();
      Node[] kids = new Node[navKids.length];
      for(int i=0; i<kids.length; ++i)
        kids[i] = new Node(getModel(), this, navKids[i]);
      return kids;
    }
    
    void preserve(Node[] oldKids, Node[] curKids)
    {            
      // this code doesn't quite work yet (sometimes 
      // the tree will collapse)    
      for(int i=0; i<curKids.length; ++i) 
      {
        Node cur = curKids[i];
        for(int j=0; j<oldKids.length; ++j)
        {
          Node old = oldKids[j];
          if (cur.ord.equals(old.ord))
          {             
            cur.setExpanded(old.isExpanded());
            cur.oldKids = old.kids;
            break;
          }
        }
      }  
    }        
    
    BOrd getHyperlink()
    {
      return navNode.getNavOrd();
    }
    
    BINavNode navNode;
    BOrd ord; 
    BImage icon;
    Node[] kids;
    Node[] oldKids;
  }

////////////////////////////////////////////////////////////////
// SessionNode
////////////////////////////////////////////////////////////////

  public class SessionNode
    extends Node
  {
    public SessionNode(TreeModel model, BISession session)
    {
      super(model, null, session);                       
      this.session = session;            
    }                    
    
    public String getText()
    {                                       
      // really should be changed in lexicon for real system
      return TextUtil.replace(session.getNavDisplayName(null), "Station", "Site");
    }
    
    public BImage getIcon() 
    {        
      if (session.isConnected())
        return siteIcon;
      else
        return siteIcon.getDisabledImage();
    }          
    
    void reloadNavFile()
    {                   
      oldKids = kids;
      kids = null;
      model.updateTree();
    }
        
    Node[] doLoad(BINavNode navNode)
      throws Exception
    {
      if (session.isConnected())
      { 
        // load nav file to build tree
        BIFile navFile = (BIFile)BOrd.make("file:^NavFile.nav").get((BObject)session);
System.out.println("loadNavFile " + navFile);  
        BNavFileSpace space = new NavFileDecoder(navFile).decodeDocument();
        return super.doLoad(space.getRootNode());
      }
      else
      {
        return new Node[0];
      }
    }

    public BMenu getMenu()
    {       
      BMenu menu = super.getMenu();
      BMenuItem[] items = menu.getMenuItems();
      for(int i=0; i<items.length; ++i)
        if (items[i].getText().indexOf("Spy") >= 0) menu.remove(items[i]);
      return menu;      
    }

    BOrd getHyperlink()
    {
      return BOrd.make(session.getNavOrd(), "station:|slot:/App");
    }
    
    BISession session;
  }     
  
////////////////////////////////////////////////////////////////
// Controller
////////////////////////////////////////////////////////////////

  class Controller
    extends TreeController
  {
    protected void popup(BMouseEvent event, TreeNode node)
    {
      if (node != null)
      {                      
        BMenu menu = ((Node)node).getMenu();
        if (menu != null) 
        {
          menu.removeConsecutiveSeparators();
          menu.open(event);
        }
      }
    }
    
    public void nodeDoubleClicked(BMouseEvent event, TreeNode node) 
    {                            
      BWbShell shell = getWbShell();
      Node n = (Node)node;
      BOrd ord = n.getHyperlink();
      if (shell != null && ord != null)
        shell.hyperlink(ord);
    }
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////
  
  static BImage siteIcon = BImage.make("module://icons/x16/site.png");
  
  BTree tree;
  Model model;
  Array roots = new Array(); 
}

