/*
scramsh1.js
(c) 2013 by Tridium, Inc. All rights reserved.
*/

/*jslint bitwise: true, continue: true, nomen: true, plusplus: true, unparam: true, vars: true, white: true */
/*global CryptoJS: false, sendHttp:false */
////////////////////////////////////////////////////////////////
// ScramSha256Client
////////////////////////////////////////////////////////////////

var ScramSha256Client = (function(){
	'use strict';
	
	/* Scram Sha1 Client class */
	function ScramSha256Client(username, password){
		this._username = ScramSha256Client._usernamePrep(username);
		this._password = ScramSha256Client._passwordPrep(password);
	}

	ScramSha256Client.generatePBKDF2 = function (password, salt, count, length, callback) {
	  sjcl.misc.pbkdf2Async({
	    password: password,
	    salt: salt,
	    count: count,
	    length: length * 8
	  }, function (err, result) {
	    if (err) {
	      callback.fail(err);
	    } else {
	      callback.ok(result);
	    }
	  });
	};
	
	// static placeholder for an eventual saslprep implementation
	ScramSha256Client._usernamePrep = function(value){
		value = UNorm.normalize('NFKC', value);
		//console.log("val = " + value);
		value = value.replace(/=/g, '=3D');
		//console.log("val = " + newValue);
		value = value.replace(/,/g,'=2C');
		//console.log("val = " + newValue);
		//return newValue;
		return value;
	};
	
	// static placeholder for an eventual saslprep implementation
	ScramSha256Client._passwordPrep = function(value){
		value = UNorm.normalize('NFKC', value);
		return value;
	};
	ScramSha256Client._random = function(nBytes){		
		
		var bits = [];
        for (var i = 0; i < nBytes; i++) {
        	bits.push((Math.random() * 0x100000000) | 0);
        }

        return bits;
	}
	
	/* method for parsing a key value pair, comma delimited list. */
	ScramSha256Client._parseMessage = function(message){
		var i, tuples = message.split(','), values = {};
		for (i = 0; i < tuples.length; i++){
			var equals = tuples[i].indexOf('=');
			if (equals > 0){
				var key = tuples[i].substring(0, equals);
				var value = tuples[i].substring(equals+1);
				values[key] = value;
			}			
		}

    return values;
	};
	
	/* static method for creating a salted password */
	ScramSha256Client._createSaltedPassword = function (password, salt, iterationCount, callback) {
		var bitSalt = sjcl.codec.base64.toBits(salt);
		ScramSha256Client.generatePBKDF2(password, bitSalt, iterationCount, 32, callback);
	};
	
	/* static method for creating the client first message bare */
	ScramSha256Client._createClientFirstMessageBare = function(userName, clientNonce){
		var clientFirstMessageBare = "n=" + userName + ",r=" + clientNonce;
		return clientFirstMessageBare;
	};
	
	/* static method for create the client final message without proof */
	ScramSha256Client._createClientFinalMessageWithoutProof = function(message){
		var values = ScramSha256Client._parseMessage(message);
		var clientFinalMessageWithoutProof = 'c=biws,r=' + values.r;
		return clientFinalMessageWithoutProof;
	};
	
	/* static method for creating the auth message */
	ScramSha256Client._createAuthMessage = function(clientFirstMessageBare, serverFirstMessage, clientFinalMessageWithoutProof){
		var authMessage = clientFirstMessageBare + ',' + serverFirstMessage + ',' + clientFinalMessageWithoutProof;
		return authMessage;
	};
	
    ScramSha256Client._hmacSha256 = function(key, message){
      if (typeof key === "string") {
        key = sjcl.codec.utf8String.toBits(key);
      }
      var hasher = new sjcl.misc.hmac(key, sjcl.hash.sha256);
      return hasher.encrypt(message);
    };    
	
	/* static method for creating the client proof */
	ScramSha256Client._createClientProof = function(saltedPassword, authMessage){
		var i;
		var clientKey = ScramSha256Client._hmacSha256(saltedPassword, "Client Key");
		var storedKey = sjcl.hash.sha256.hash(clientKey);
		var clientSignature = ScramSha256Client._hmacSha256(storedKey, authMessage);
		var clientProof = [];
		for (i = 0; i < clientKey.length; i++)
		{
			clientProof[i] = clientKey[i] ^ clientSignature[i];
		}
		
		return clientProof;
	};
	
	/* static method for creating the server signature */
	ScramSha256Client._createServerSignature = function(saltedPassword, authMessage){
		var serverKey = ScramSha256Client._hmacSha256(saltedPassword, "Server Key");
		var serverSignature = ScramSha256Client._hmacSha256(serverKey, authMessage);
		return serverSignature;
	};
	
	/* instance method for creating the client first message that will be sent to the server */
	ScramSha256Client.prototype.createClientFirstMessage = function(){
		//var clientNonce = sjcl.random.randomWords(16);
		var clientNonce = ScramSha256Client._random(4);
		this._clientNonce = sjcl.codec.base64.fromBits(clientNonce);
		this._clientFirstMessageBare = ScramSha256Client._createClientFirstMessageBare(this._username, this._clientNonce);
		var clientFirstMessage = "n,," + this._clientFirstMessageBare;
		return clientFirstMessage;
	};
	
	/* instance method for creating the client final message for the server */
	ScramSha256Client.prototype.createClientFinalMessage = function (serverFirstMessage, callback)
	{
		var values = ScramSha256Client._parseMessage(serverFirstMessage),
		    that = this;
		
		/* lets make sure that the client nonce matches what we sent */
		if (this._clientNonce !== values.r.substring(0, this._clientNonce.length)){
			return callback.fail("invalid client nonce");
		}
		
		ScramSha256Client._createSaltedPassword(this._password, values.s, values.i, {
		  ok: function (saltedPassword) {
  		  that._saltedPassword = saltedPassword;
  	    var clientFinalMessageWithoutProof = ScramSha256Client._createClientFinalMessageWithoutProof(serverFirstMessage);
  	    that._authMessage = ScramSha256Client._createAuthMessage(that._clientFirstMessageBare, serverFirstMessage, clientFinalMessageWithoutProof);
  	    var clientProof = ScramSha256Client._createClientProof(saltedPassword, that._authMessage);
  	    var clientFinalMessage = clientFinalMessageWithoutProof + ',p=' +  sjcl.codec.base64.fromBits(clientProof);
  		  callback.ok(clientFinalMessage);
  		},
  		fail: callback.fail
		});
	};
	
	/* instance method for validating the final server response */
	ScramSha256Client.prototype.processServerFinalMessage = function(serverFinalMessage){
		var values = ScramSha256Client._parseMessage(serverFinalMessage);
		var serverSignature = ScramSha256Client._createServerSignature(this._saltedPassword, this._authMessage);
		var remoteServerSignature = sjcl.codec.base64.toBits(values.v);
		
		if (remoteServerSignature.toString() !== serverSignature.toString()){
			throw new Error("invalid server signature");
		}
	};
	
	/* static method to simply perform an authentication using scram-sha1 */
	ScramSha256Client.authenticate = function(absPathBase, username, password, callback){
		try{
			var client = new ScramSha256Client(username, password);
			var clientFirstMessage = client.createClientFirstMessage();
			var body="action=sendClientFirstMessage&clientFirstMessage=" + clientFirstMessage;
			sendHttp("POST", absPathBase + "login/", body, {
				ok:function(message){
				  ScramSha256Client.receiveServerFirstMessage(absPathBase, client, message, callback);
				},
				fail:function(cause){
					callback.fail(cause);			
				}	
			});
		}
		catch(e){
			callback.fail(e);
		}
	};
	
	/* static method to handle the server response to the first message */
	ScramSha256Client.receiveServerFirstMessage = function(absPathBase, client, message, callback){
		client.createClientFinalMessage(message, {
		  ok: function (clientFinalMessage) {
        var body="action=sendClientFinalMessage&clientFinalMessage=" + clientFinalMessage;
			sendHttp("POST", absPathBase + "login/", body, {
          ok: function (message) {
            ScramSha256Client.receiveServerFinalMessage(client, message, callback);
          },
          fail: function (cause) {
            callback.fail(cause);
          } 
        });
  		},
  		fail: callback.fail
		});
	};
	
	/* static method to handle the server final response */
	ScramSha256Client.receiveServerFinalMessage = function(client, message, callback){
		try{
			client.processServerFinalMessage(message);		
			callback.ok("authenticated");
		}
		catch(e){
			callback.fail(e);
		}
	};
	
	return ScramSha256Client;
}());

