/**
 * @fileOverview Google Maps and BajaScript Demo.
 *
 * @author Gareth Johnson
 */
 
//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: false, nomen: false, eqeqeq: true, 
bitwise: true, regexp: true, newcap: true, immed: true, strict: false, indent: 2, vars: true, continue: true */

/*global $, baja, location, window, document, google, dialogs*/ 

(function googleMaps() {
  // Use ECMAScript 5 Strict Mode
  "use strict";
    
  var sub = new baja.Subscriber(),
      maps = google.maps,
      infoPopupHtml = "<div class='infoPopup'>" + 
                        "<h3><a href='#' class='infoPopupTitle'>Loading...</a></h3>" + 
                        "<table class='info'>" +
                          "<tbody>" +
                            "<tr>" + 
                              "<th>Name</th>" + 
                              "<th>Value</th>" + 
                            "</tr>" + 
                          "</tbody>" +
                        "</table>" +
                      "</div>",
      infoRowHtml = "<tr><td class='infoRowName'></td><td class='infoRowValue'></td></tr>",
      introDialogHtml = "<p>" +
                          "Please note, this is a free and publically available code example" +
                          "of how BajaScript can integrate with the Google Maps API." +
                        "</p>" +
                        "<p>" +
                          "This is not currently a product and is not supported by Tridium Technical Support." +
                        "</p>" +
                        "<p>" +
                          "This Web App will scan a Station for Components that have a 'long' (longitude) and 'lat' (latitude) BDouble Properties." +
                        "</p>" +
                        "<p>" +
                          "Each Component is displayed as a marker on the Map. If the Component is a point and in alarm, the marker will bounce." +
                        "</p>" +
                        "<p>" +
                          "Each Property that's marked with the Summary Slot Flag will be displayed in a table when the marker for" +
                          "the Component is clicked." +
                        "</p>" +
                        "<p>" +
                          "The default map shows the area where Tridium headquarters is based (Richmond, Virginia, USA)." +
                        "</p>";
  
  /**
   * Add a Marker to the Map.
   */
  function addMarker(comp, map) {
  
    // Create the marker
    var marker = new maps.Marker({
      position: new maps.LatLng(comp.get("lat").valueOf(), comp.get("long").valueOf()),
      title: comp.getName(),
      optimized: false // Ensures animated GIFs work ok
    });
       
    // Add the marker to the Map
    marker.setMap(map);
        
    // Create the pop up window and dynamically update it
    var infowindow = new maps.InfoWindow({content: "<div></div>"});
        
    var createInfoWindow = function () {  
        
      // Always look up the content to update
      var content = $(infoPopupHtml),
          tbody = $("tbody", content),
          title =  $(".infoPopupTitle", content);
          
      // Update the title             
      title.text(comp.getDisplayName());
      title.attr("href", comp.getNavOrd().toUri());
          
      // For each Property create an entry in the table
      comp.getSlots().properties().flags(baja.Flags.SUMMARY).is("baja:StatusValue", "baja:Simple").each(function (slot) {
        // Add a Row
        var tableRow = $(infoRowHtml);
        tbody.append(tableRow);
        
        // Wait for updates and update the row
        var updateInfoWindow = function (prop) {
          if (prop.getName() === slot.getName()) {
            tableRow.find(".infoRowName").text(comp.getDisplayName(prop));
            tableRow.find(".infoRowValue").text(comp.getDisplay(prop));
          }
        };
        
        updateInfoWindow(slot);
        sub.attach("changed", updateInfoWindow);
      });  
                   
      // Set the content
      infowindow.setContent(content.get(0));
      infowindow.open(map, marker);
    };
    
    var isAlarm = false;
    function updateMarkerStatus(marker, comp) {
      var status = baja.Status.getStatusFromIStatus(comp);
      if (status.isAlarm() !== isAlarm) {
        isAlarm = status.isAlarm();
        marker.setAnimation(isAlarm ? maps.Animation.BOUNCE : null);
      }
    }
    
    // If the Component implements BIStatus then monitor its alarm status
    if (comp.getType().is("baja:IStatus")) {
      // If the Component changes then check its Status
      comp.attach("changed", function (prop) {
        updateMarkerStatus(marker, comp);
      });
      
      // Update its Status for the first time
      updateMarkerStatus(marker, comp);
    }
        
    // When a marker is clicked, make a pop up happen
    maps.event.addListener(marker, "mousedown", createInfoWindow);       
  }
  
  /**
   * Initialize the Application.
   */
  function init() {
    // Always show the dialog when the page loads
    dialogs.ok({
      title: "Google Maps BajaScript Demo",
      content: introDialogHtml
    });
  
    // Initialize Google Maps
    var latlng = new maps.LatLng(37.64258, -77.57822),
        options = {
          zoom: 15,
          center: latlng,
          mapTypeId: maps.MapTypeId.ROADMAP
        },
        map = new maps.Map($("#map_canvas").get(0), options),
        ords = [];
    
    // Query and subscribe all of the points on the map
    baja.Ord.make("station:|slot:/|bql:select navOrd from baja:Component where lat != null and long != null").get({
      cursor: {
        each: function (row) {
          // Note down each resolve ORD
          ords.push(baja.Ord.make(row.get("navOrd")));
        },
        after: function () {
          // Resolve all of the ORDs in question and subscribe
          new baja.BatchResolve(ords).resolve({
            each: function () {      
              addMarker(this, map);
            },
            subscriber: sub
          });
        }
      }
    });     
  }
  
  // Initialize the App once BajaScript has started and the DOM is ready  
  baja.started(init); 
  
  // Stop BajaScript when the page is hidden or unloaded
  $(window).bind("unload pagehide", baja.stop);
  
  // If the page shows again but BajaScript is already stopped then reload the page
  $(window).bind("pageshow", function () {
    if (baja.isStopped()) {
      location.reload(false);
    }
  });
  
  // Start BajaScript
  baja.start();
}());