//
// Copyright 2010, Tridium, Inc. All Rights Reserved.
//

//JsLint options (see http://www.jslint.com )
/*jslint white: true, vars: true */

// Globals for JsLint to ignore 
/*global baja, testFolder,
         callbackify, runAndWait, errTest, verify, verifyEq, failTest, 
         describe, it, xit, expect, beforeEach, afterEach, runs, spyOn, xdescribe,
         console,
         typeMismatchError, undefinedError */

var ordTest = { version: 1.0 };

describe("ord.js", function ordTest() {
  "use strict";
  
  var ordTestFolder;
  
  //fake an OrdQueryListCursor that does nothing but track calls to 
  //resolveNext
  function SpyCursor() {
    var $called = false;
    this.resolveNext = function () {
      $called = true;
    };
    this.isCalled = function () {
      return $called;
    };
    
    spyOn(this, 'resolveNext').andCallThrough();
  }
  
  //fake an OrdQuery
  function SpyQuery(obj) { 
    obj = obj || {};  
    this.$scheme = obj.scheme;
    this.$schemeName = obj.schemeName;
    this.$body = obj.body;
    this.$isHost = obj.isHost || false;
    this.$isSession = obj.isSession || false;
    
    // Override any functions
    var p;
    for (p in obj) {
      if (obj.hasOwnProperty(p) && typeof obj[p] === "function") {
        this[p] = obj[p];
      }
    }
    
    this.normalize = function () {};
    this.getBody = function () { return this.$body; };
    this.getScheme = function () { return this.$scheme; };
    this.getSchemeName = function () { return this.$schemeName; };
    
    spyOn(this, 'normalize');
  }
  
  function toQueryList(ord) {
    return baja.Ord.make(ord).parse();
  }

  beforeEach(function () {
    runs(function () {
      if (!testFolder.has('ordTestFolder')) {
        var addCallbacks = callbackify({
          ok: function () {
            ordTestFolder = testFolder.get('ordTestFolder');
          },
          slot: 'ordTestFolder',
          value: baja.$('baja:Folder')
        });
        
        runAndWait(function () {
          testFolder.add(addCallbacks);
        }, addCallbacks);
      } else {
        ordTestFolder = testFolder.get('ordTestFolder');
      }
    });
  });
  
  describe("baja.nav.localhost.station", function bajaNavLocalhostStation() {
    describe("getOrdInSession()", function getOrdInSession() {
      it("returns station session ORD", function returnsStationSessionOrd() {
        verifyEq(baja.nav.localhost.station.getOrdInSession().toString(), "station:");
      });
    });
  });
  
  describe("baja.station", function bajaStation() {
    describe("getOrdInSession()", function getOrdInSession() {
      it("returns station session ORD", function returnsStationSessionOrd() {
        verifyEq(baja.station.getOrdInSession().toString(), "station:");
      });
    });
  }); 
  
  describe("baja.SlotPath", function SlotPath() {
    
    function verifyMerge(path1, path2, expected) {
      path1 = new baja.SlotPath(path1);
      path2 = new baja.SlotPath(path2);
      var merge = path1.merge(path2);
      verifyEq(merge, expected);
    }
    
    function verifyValid(name) {
      verify(baja.SlotPath.isValidName(name));
    }
    
    function verifyInvalid(name) {
      verify(!baja.SlotPath.isValidName(name));
    }
    
    function verifyDepth(path, depth) {
      verifyEq(new baja.SlotPath(path).depth(), depth);
    }
    
    function verifyBackup(path, depth) {
      verifyEq(new baja.SlotPath(path).getBackupDepth(), depth);
    }
    
    //unescaped at [0], escaped at [1]
    var names = [
                 ["", ""],
                 ["a", "a"],
                 ["z", "z"],
                 ["A", "A"],
                 ["Z", "Z"],
                 ["4", "$34"],
                 ["_", "$5f"],
                 ["$", "$24"],
                 ["Name", "Name"],
                 ["With Space", "With$20Space"],
                 ["With/Slash", "With$2fSlash"],
                 ["9foo", "$39foo"],
                 ["a_b_c", "a_b_c"],
                 ["x" + String.fromCharCode(0x7f) + "_" + String.fromCharCode(0xf0) + "_" + String.fromCharCode(0x100) + "_" + String.fromCharCode(0xa00) + "_" + String.fromCharCode(0xabcd), "x$7f_$f0_$u0100_$u0a00_$uabcd"],
                 [" ",    "$20"],
                 [" x",   "$20x"],
                 ["x ",   "x$20"],
                 ["x x",  "x$20x"],
                 ["  ",   "$20$20"],
                 [String.fromCharCode(0x1234),        "$u1234"],
                 ["x" + String.fromCharCode(0x1234),  "x$u1234"],
                 [String.fromCharCode(0x1234) + "x",  "$u1234x"],
                 ["x" + String.fromCharCode(0x1234) + "x", "x$u1234x"]
               ];
  
    describe("constructor", function SlotPath_constructor() {
      
      it("allows an absolute slot path", function allowsAbsolute() {
        var sp = new baja.SlotPath("/this/that/foo/bar");
        verify(true);
      });
        
      it("resolves a non-absolute slot path", function resolvesNonAbsolute() {
        var sp = new baja.SlotPath("this/that/foo/bar");
        verify(true);
      });
      
      it("resolves root", function resolvesRoot() {
        // Root
        var sp = new baja.SlotPath("/");   
        verify(true);
      });
      
      it("allows backup", function allowsBackup() {
        var sp = new baja.SlotPath("../a");
        verify(true);
      });
      
      it("dies on trailing slash", function diesOnTrailingSlash() {
        errTest(function () {
          var sp = new baja.SlotPath("this/that/foo/bar/");
        });
      });
  
      it("dies on double slash", function diesOnDoubleSlash() {
        errTest(function () {
          var sp = new baja.SlotPath("this/that//foo/bar");
        });
      });
  
      it("dies on space", function diesOnSpace() {
        errTest(function () {
          var sp = new baja.SlotPath("/this/th at/foo/bar");
        });
      });
        
      it("dies on numeric path", function diesOnNumeric() {
        errTest(function () {
          var sp = new baja.SlotPath("/123");
        });
      });
  
      it ("dies on invalid character", function diesOnInvalid() {
        errTest(function () {
          var sp = new baja.SlotPath("/@/34");
        });
      });
      
      it("dies on single dot", function diesOnSingleDot() {
        errTest(function () {
          var sp = new baja.SlotPath("./a");
        });
      });
      
    });
    

    describe("isValidName() (static)", function SlotPath_isValidName() {
      it("verifies good names", function verifiesGood() {
        verifyValid("a");
        verifyValid("A");
        verifyValid("z");
        verifyValid("zeta");
        verifyValid("Z");
        verifyValid("abc");
        verifyValid("abc_ABC");
        verifyValid("a0");
        verifyValid("a_0_");
        verifyValid("z9");
        verifyValid("a$00");
        verifyValid("a$aa");
        verifyValid("a$ff");
        verifyValid("a$AA");
        verifyValid("a$FF");
        verifyValid("a$AA");
        verifyValid("a$FF");
        verifyValid("a$u01AB");
        verifyValid("a$u0000");
        verifyValid("$20");
        verifyValid("$20xx");
        verifyValid("$u01AB");
      });
      
      it("verifies bad names", function verifiesBad() {
        verifyInvalid("");
        verifyInvalid(".");
        verifyInvalid("./a");
        verifyInvalid("../a");
        verifyInvalid("9");
        verifyInvalid("2bad");
        verifyInvalid("_");
        verifyInvalid("a$");
        verifyInvalid("a$0");
        verifyInvalid("a$0x");
        verifyInvalid("a$0g");
        verifyInvalid("a$u");
        verifyInvalid("a$u0");
        verifyInvalid("a$u00");
        verifyInvalid("a$u000");
        verifyInvalid("a$u000g");
        verifyInvalid("a$u000$");
      });
      
      it("disallows invalid characters", function disallowsInvalidChars() {
        var invalidChars = ["`", "~", "!", "@", "#", "%", "^", "&", "*", "(", 
                            ")", "=", "+", "[", "]", "{", "}", "\\", "|", ";", 
                            ":", "'", "\"", ",", "<", ">", "/", "?", 
                            String.fromCharCode(0x5B), 
                            String.fromCharCode(0x7F),
                            String.fromCharCode(0xFF), 
                            String.fromCharCode(0x100), 
                            String.fromCharCode(0xABCD)];
                            
        var i;
        for (i = 0; i < invalidChars.length; ++i) {
          verifyInvalid(invalidChars[i]);
          verifyInvalid("a" + invalidChars[i]);
        }
      });
    });
    
    describe("escape() (static)", function SlotPath_escape() {
      it("escapes properly", function escapesProperly() {
        var i;
        for (i = 0; i < names.length; ++i) {
          var toEscape = names[i][0];
          var expected = names[i][1];
          var escaped = baja.SlotPath.escape(toEscape);

          // verify we don't alloc a new instance if unneeded
          if (escaped.indexOf("$") < 0) {
            verifyEq(toEscape, escaped);
          }
          else {
            verify(!escaped.equals(toEscape));
          }
          
          verify(expected.equals(escaped));
        }
      });
    });
    
    describe("unescape() (static)", function SlotPath_unescape() {
      it("unescapes properly", function unescapesProperly() {
        var i;
        for (i = 0; i < names.length; ++i) {
          var expected = names[i][0];
          var toUnescape = names[i][1];
          var unescaped = baja.SlotPath.unescape(toUnescape);    
          
          verify(expected.equals(unescaped));
        }
      });
    });
    
    describe("verifyValidName() (static)", function SlotPath_verifyValidName() {
      it("passes for valid name", function passesForValid() {
        baja.SlotPath.verifyValidName("a");
        verify(true);
      });
      
      it("dies for invalid name", function diesForInvalid() {
        errTest(function () {
          baja.SlotPath.verifyValidName("$no sir$");
        }, "Illegal name for Slot: $no sir$");
      });
    });
    
    describe("depth()", function SlotPath_depth() {
      it("returns depth of slot path", function returnsDepth() {
        verifyDepth("/a/b/c", 3);
        verifyDepth("a/b/c", 3);
        verifyDepth("a/b/c/d/e/f/g", 7);
        verifyDepth("", 0);
        verifyDepth("/", 0);
        verifyDepth("../a", 1);
        verifyDepth("../a/b", 2);
        verifyDepth("../../a", 1);
      });
    });
    
    describe("getBackupDepth()", function SlotPath_getBackupDepth() {
      it("returns backup depth", function returnsBackupDepth() {
        verifyBackup("a", 0);
        verifyBackup("a/b/c", 0);
        verifyBackup("../a", 1);
        verifyBackup("../../a", 2);
        verifyBackup("..", 1);
        verifyBackup("../..", 2);
      });
    });
    
    describe("isAbsolute()", function SlotPath_isAbsolute() {
      it("returns true if path is absolute", function returnsTrueIfAbs() {
        verify(new baja.SlotPath("/a/b/c").isAbsolute());
        verify(new baja.SlotPath("/").isAbsolute());
        verify(!new baja.SlotPath("a/b/c").isAbsolute());
        verify(!new baja.SlotPath("").isAbsolute());
        verify(!new baja.SlotPath("../a").isAbsolute());
      });
    });
    
    describe("nameAt()", function SlotPath_nameAt() {
      it("returns the name at the specified index", function nameAtIndex() {
        var sp = new baja.SlotPath("this/that/foo/bar");
        verifyEq(sp.nameAt(0), "this");
        verifyEq(sp.nameAt(1), "that");
        verifyEq(sp.nameAt(2), "foo");
        verifyEq(sp.nameAt(3), "bar");
      });
      
      it("does not include leading / as name", function noLeadingSlash() {
        var sp = new baja.SlotPath("/this/that/foo/bar");
        verifyEq(sp.nameAt(0), "this");
        verifyEq(sp.nameAt(1), "that");
        verifyEq(sp.nameAt(2), "foo");
        verifyEq(sp.nameAt(3), "bar");
      });
      
      it("returns undefined if index not found", function indexNotFound() {
        var sp = new baja.SlotPath("/this/that/foo/bar");
        expect(sp.nameAt(-1)).toBeUndefined();
        expect(sp.nameAt(4)).toBeUndefined();
      });
    });
    
    describe("isValidPathName()", function SlotPath_isValidPathName() {
      it("proxies through to static isValidName()", function proxies() {
        verify(new baja.SlotPath("a").isValidPathName("a"));
        verify(!new baja.SlotPath("a").isValidPathName(""));
      });
    });
    
    describe("merge()", function SlotPath_merge() {
      it("merges absolute path with relative", function absRelative() {
        verifyMerge("/", "a", "/a");
        verifyMerge("/a", "b", "/a/b");
        verifyMerge("/a/b/c", "d/e/f", "/a/b/c/d/e/f");
      });
      
      it("merges absolute path with backup", function absBackup() {
        verifyMerge("/a", "..", "/");
        verifyMerge("/a/b", "../..", "/");
        verifyMerge("/a/b", "../../", "/");
        verifyMerge("/a/b/c", "../../d/e", "/a/d/e");
      });
      
      it("merges relative path with relative", function relativeRelative() {
        verifyMerge("", "a", "a");
        verifyMerge("a", "b", "a/b");
        verifyMerge("a/b", "c", "a/b/c");
        verifyMerge("a", "b/c", "a/b/c");
        verifyMerge("a/b", "c/d", "a/b/c/d");
      });
      
      it("merges relative path with backup", function relativeBackup() {
        verifyMerge("a", "..", "");
        verifyMerge("a/b", "..", "a");
        verifyMerge("a/b/c", "../..", "a");
        verifyMerge("a/b/c", "..", "a/b");
        verifyMerge("a/b/c", "../", "a/b");
        verifyMerge("a/b/c/d", "../../e/f", "a/b/e/f");
      });
      
      it("merges backup with backup", function backupBackup() {
        //TODO: commented tests fail - why?
        verifyMerge("", "..", "../");
        verifyMerge("..", "..", "../../");
        //verifyMerge("..", "", "../");
        verifyMerge("../..", "../..", "../../../../");
        //verifyMerge("..", "a", "../a");
        verifyMerge("..", "../a", "../../a");
        //verifyMerge("../a", "..", "../");
      });
      
      it("merges absolute with absolute", function absAbs() {
        verifyMerge("/", "/", "/");
        verifyMerge("/a", "/a", "/a");
        verifyMerge("/a", "/b", "/b");
        verifyMerge("/a/b/c", "/a", "/a");
        verifyMerge("/a", "/b/c/d", "/b/c/d");
      });
    });
  });
  
  describe("baja.Ord", function Ord() {
    function verifyGet(base, slotPath, expected) {
      var callbacks = callbackify({
        base: base,
        ok: function (obj) {
          expect(obj).toBe(expected);
        },
        fail: failTest
      });
      
      runAndWait(function () {
        baja.Ord.make(slotPath).get(callbacks);
      }, callbacks);
    }
    
    function verifyResolve(base, slotPath, func) {
      var callbacks = callbackify({
        base: base,
        ok: func,
        fail: failTest
      });
      
      runAndWait(function () {
        baja.Ord.make(slotPath).resolve(callbacks);
      }, callbacks);
    }
  
    describe("constructor", function Ord_constructor() {
      it("has correct type", function hasCorrectType() {
        var ord = baja.$("baja:Ord");
        verify(ord.getType().is(baja.lt("baja:Ord")));
        verify(ord.getType().is(baja.lt("baja:IComparable")));
      });
      
      it("initializes to DEFAULT", function initializesToDEFAULT() {
        expect(baja.$('baja:Ord')).toBe(baja.Ord.DEFAULT);
      });
      
      it("takes an ord string", function takesOrdString() {
        var ord = new baja.Ord("station:|slot:/");
        verify(ord !== baja.Ord.DEFAULT);
      });
    });
    
    describe("make() (static)", function Ord_make() {
      it("takes undefined", function takesUndefined() {
        var ord = baja.Ord.make();
        expect(ord).toBe(baja.Ord.DEFAULT);
      });
      
      it("takes empty string", function takesEmptyString() {
        var ord = baja.Ord.make("");
        expect(ord).toBe(baja.Ord.DEFAULT); 
      });
      
      it("takes an ord string", function takesOrdString() {
        var ord = baja.Ord.make("station:|slot:/");
        verify(ord !== baja.Ord.DEFAULT);
      });
      
      it("takes object literal with base/child", function takesObjectLiteral() {
        var ord = baja.Ord.make({
          base: "station:",
          child: "slot:/This/That"
        });
        verifyEq(ord.toString(), "station:|slot:/This/That");
      });
    });
    
    describe("encodeToString()", function Ord_encodeToString() {
      it("gives string representation of the ord", function givesString() {
        var ord = baja.Ord.make("station:|slot:/");
        verifyEq(ord.encodeToString(), "station:|slot:/");
      });
    });
    
    describe("parse()", function Ord_parse() {
      describe("from cursor", function parseCursor() {
        var list = baja.Ord.make("local:|station:|slot:/|bql:select * from " +
                                 "baja:Component").parse();
        var c = list.getCursor();

        it("parses local scheme", function parsesLocalScheme() {
          verify(c.next());
          var query = c.get();
          verifyEq(query.getSchemeName(), "local"); 
          verifyEq(query.getBody(), ""); 
          verifyEq(query.getScheme(), baja.LocalScheme.DEFAULT); 
          verifyEq(c.getOrd().toString(), "local:"); 
        });
        
        it("parses station scheme", function parsesStationScheme() {
          verify(c.next());
          var query = c.get();
          verifyEq(query.getSchemeName(), "station"); 
          verifyEq(query.getBody(), ""); 
          verifyEq(query.getScheme(), baja.StationScheme.DEFAULT); 
          verifyEq(c.getOrd().toString(), "local:|station:");
        });
        
        it("parses slot scheme", function parsesSlotScheme() {
          verify(c.next());
          var query = c.get();
          verifyEq(query.getSchemeName(), "slot"); 
          verifyEq(query.getBody(), "/"); 
          verifyEq(query.getScheme(), baja.SlotScheme.DEFAULT); 
          verifyEq(c.getOrd().toString(), "local:|station:|slot:/");
        });
        
        it("parses bql scheme", function parsesBqlScheme() {
          verify(c.next());
          var query = c.get();
          verifyEq(query.getSchemeName(), "bql"); 
          verifyEq(query.getBody(), "select * from baja:Component"); 
          verifyEq(query.getScheme(), baja.UnknownScheme.DEFAULT); 
          verifyEq(c.getOrd().toString(), "local:|station:|slot:/|bql:select * from baja:Component");
        });
        
        it("has correct number of schemes", function hasCorrectNumber() {
          verify(!c.next());
        });
        
        it("parses an empty string", function parsesEmptyString() {
          verify(baja.Ord.make("").parse().isEmpty());
        });
      });
      
      describe("from list", function parseList() {
        var list = baja.Ord.make("station:|slot:/this/that" +
                                 "|view:myModule:MyType").parse();

        it("gets by index", function getsByIndex() {
          verifyEq(list.get(0).getSchemeName(), "station");
          verifyEq(list.get(1).getSchemeName(), "slot");
          verifyEq(list.get(2).getSchemeName(), "view");
        });
        
        it("returns null if index not found", function indexNotFound() {
          expect(list.get(3)).toBeNull();
        });
        
        it("gets by scheme name", function getsBySchemeName() {
          verifyEq(list.get("station").getSchemeName(), "station");
          verifyEq(list.get("station").getBody(), "");
          verifyEq(list.get("slot").getSchemeName(), "slot");
          verifyEq(list.get("slot").getBody(), "/this/that");
          verifyEq(list.get("view").getSchemeName(), "view");
          verifyEq(list.get("view").getBody(), "myModule:MyType");
        });
        
        it("returns null if scheme name not found", function schemeNotFound() {
          expect(list.get("foo")).toBeNull();
        });
      });
    });
    
    describe("resolve()", function Ord_resolve() {
      //setup our test components - our tests are read-only so we can use them
      //repeatedly
      var comp = new baja.Component(),
          test = new baja.Component(),
          that = new baja.Component(),
          foo = new baja.Component(),
          bar = new baja.Component(),
          statusNum = baja.$('baja:StatusNumeric'),
          slotPath = "slot:test/that/foo/bar/statusNum/value";
      
      statusNum.setValue(34.4);
      
      comp.add({ slot: "test", value: test });
      test.add({ slot: "that", value: that });
      that.add({ slot: "foo", value: foo });
      foo.add({ slot: "bar", value: bar });
      bar.add({ slot: "statusNum", value: statusNum });

      
      it("resolves local scheme", function resolvesLocal() {
        baja.Ord.make("local:").resolve({
          ok: function (tg) {
            expect(tg.object).toBe(baja.nav.localhost);
          },
          fail: failTest
        });
      });
      
      it("resolves station scheme", function resolvesStation() {
        baja.Ord.make("local:|station:").resolve({ 
          ok: function (tg) {
            expect(tg.object).toBe(baja.nav.localhost.station);
            expect(tg.object).toBe(baja.station);
          },
          fail: failTest
        });
      });      
      
      it("resolves using handles", function resolvesWithHandles() {
        var c1 = new baja.Component(),
            c2 = new baja.Component(),
            c3 = new baja.Component(),
            c4 = new baja.Component(),
            c5 = new baja.Component(),
            cs = new baja.ComponentSpace();
        
        c1.$handle = "1";
        c2.$handle = "2";
        c3.$handle = "3";
        c4.$handle = "4";
        c5.$handle = "5";
        
        c1.add({ slot: "kid", value: c2 });
        c2.add({ slot: "kid", value: c3 });
        c3.add({ slot: "kid", value: c4 });
        c4.add({ slot: "kid", value: c5 });
        
        // Mount Components inside a Component Space
        cs.$fw("mount", c1);
        
        verifyGet(cs, "h:1", c1);
        verifyGet(cs, "h:3", c3);
        verifyGet(cs, "h:5", c5);
        
        baja.Ord.make("h:6").get({
          base: cs, 
          ok: function (comp) {
            failTest("should not have resolved non-existent handle");
          }, 
          fail: function (err) {
            verify(true);
          }
        });
      });
      
      it("resolves the mounted station root", function resolvesStationRoot() {
        var stationOrd = "station:|slot:/",
            resolveCallbacks = callbackify(function (target) {
              var comp = target.getComponent();
              verify(target);
              verifyEq(String(comp.getNavOrd()), "local:|" + stationOrd);
              verifyEq(String(comp.getType()), "baja:Station");
              verify(comp.isMounted());
            });
        
        runAndWait(function () {
          baja.Ord.make(stationOrd).resolve(resolveCallbacks);
        }, resolveCallbacks);
      });
      
      it("resolves the mounted test folder", function resolvesTestFolder() {
        var folderOrd = "station:|slot:/BajaScriptTestFolder",
            resolveCallbacks = callbackify(function (target) {
              var comp = target.getComponent();
              verify(target);
              verifyEq(String(comp.getNavOrd()), "local:|" + folderOrd);
              verifyEq(String(comp.getType()), "testBajaScript:TestFolder");
              verify(comp.isMounted());
            });
        
        runAndWait(function () {
          baja.Ord.make(folderOrd).resolve(resolveCallbacks);
        }, resolveCallbacks);
      });
      
      it("calls fail if not found", function callsFailIfNotFound() {
        var unknownOrd = "station:|slot:/IDoNoExist",
            resolveCallbacks = callbackify({
              ok: function () {
                failTest("should not call ok if not found");
              },
              fail: function () {
                verify(true);
              }
            });
        
        runAndWait(function () {
          baja.Ord.make(unknownOrd).resolve(resolveCallbacks);
        }, resolveCallbacks);
      });
      
      it("resolves the Station's Save Action", function resolveSaveAction() {
        var getCallbacks = callbackify(function (target) {
          expect(target.getObject()).toBeNull();
          expect(target.container).toBeTruthy();
          expect(target.slot.isAction()).toBeTruthy();
        });
        
        runAndWait(function () {
          baja.Ord.make("station:|slot:/save").resolve(getCallbacks);
        }, getCallbacks);
      });
      
      it("resolves the default alarm class's alarm Topic", function resolveAlarmTopic() {
        var getCallbacks = callbackify(function (target) {
          expect(target.getObject()).toBeNull();
          expect(target.container).toBeTruthy();
          expect(target.slot.isTopic()).toBeTruthy();
        });
        
        runAndWait(function () {
          baja.Ord.make("station:|slot:/Services/AlarmService/defaultAlarmClass/alarm").resolve(getCallbacks);
        }, getCallbacks);
      });
    });
    
    describe("get()", function Ord_get() {
      it("resolves the mounted station root", function resolvesFromStation() {
        var getCallbacks = callbackify(function (retrieved) {
          verify(retrieved);
          verifyEq(retrieved.getNavOrd().toString(), "local:|station:|slot:/");
        });
        
        runAndWait(function () {
          baja.Ord.make("station:|slot:/").get(getCallbacks);
        }, getCallbacks);
      });
      
      it("resolves the mounted test folder", function resolvesTestFolder() {
        var getCallbacks = callbackify(function (retrieved) {
          verify(retrieved);
          verify(retrieved.getNavOrd().toString(), 
                 "local:|station:|slot:/BajaScriptTestFolder");
        });
        
        runAndWait(function () {
          baja.Ord.make("station:|slot:/BajaScriptTestFolder").get(getCallbacks);
        }, getCallbacks);
      });
      
      it("calls fail if not found", function callsFailIfNotFound() {
        var getCallbacks = callbackify({
          ok: function () {
            failTest("should not call ok if not found");
          },
          fail: function () {
            verify(true);
          }
        });
        
        runAndWait(function () {
          baja.Ord.make("station:|slot:/BajaScriptTestFolder/IDoNotExist")
            .get(getCallbacks);
        }, getCallbacks);
      });
    });
    
    describe("equals()", function Ord_equals() {
      it("equals same instance", function equalSameInstance() {
        verify(baja.Ord.DEFAULT.equals(baja.Ord.DEFAULT));
      });
      
      it("equals on empty string", function equalEmptyString() {
        verify(baja.Ord.DEFAULT.equals(baja.Ord.make("")));
        verify(baja.Ord.make("").equals(baja.Ord.DEFAULT));
      });
      
      it("equals on same string", function equalSameString() {
        verify(baja.Ord.make("slot:/").equals(baja.Ord.make("slot:/"))); 
      });
      
      it("doesn't equal different string", function unequalDifferentString() {
        verify(!baja.Ord.make("slot:/this").equals(baja.Ord.make("slot:/"))); 
      });
    });
    
    describe("equivalent()", function Ord_equivalent() {
      it("is equiv. same instance", function equivSameInstance() {
        verify(baja.Ord.DEFAULT.equivalent(baja.Ord.DEFAULT));
      });
      
      it("is equiv. on empty string", function equivEmptyString() {
        verify(baja.Ord.DEFAULT.equivalent(baja.Ord.make("")));
        verify(baja.Ord.make("").equivalent(baja.Ord.DEFAULT));
      });
      
      it("is equiv. on same string", function equivSameString() {
        verify(baja.Ord.make("slot:/").equivalent(baja.Ord.make("slot:/"))); 
      });
      
      it("is not equiv. different string", function unequivDifferentString() {
        verify(!baja.Ord.make("slot:/this").equivalent(
            baja.Ord.make("slot:/"))); 
      });
    });
    
    describe("normalize()", function Ord_normalize() {
      function verifyNorm(ord1, ord2) {
        verifyEq(baja.Ord.make(ord1).normalize().toString(), ord2);
      }
  
      //TODO: move virtual scheme normalization tests into virtTest.js
      it("normalizes 2 slot schemes plus virtual", function twoSlotsVirtual() {
        verifyNorm("slot:/this/that|slot:foo/boo|virtual:goo", 
                   "slot:/this/that/foo/boo|virtual:goo");
      });
      
      it("normalizes 1 virtual scheme to same", function singleVirtual() {
        verifyNorm("virtual:/this/that/foo/boo", 
                   "virtual:/this/that/foo/boo");
      });
      
      it("normalizes 2 virtual schemes", function twoVirtual() {
        verifyNorm("virtual:/this/that|virtual:foo/boo", 
                   "virtual:/this/that/foo/boo");
      });
      
      it("normalizes 3 virtual schemes", function threeVirtual() {
        verifyNorm("virtual:/this/that|virtual:foo/boo|virtual:goo", 
                   "virtual:/this/that/foo/boo/goo");
      });
      
      it("normalizes 2 virtuals and a slot", function twoVirtualsPlusSlot() {
        verifyNorm("virtual:/this/that|virtual:foo/boo|slot:goo", 
                   "virtual:/this/that/foo/boo|slot:goo");
      });
    });
    
    describe("toUri()", function Ord_toUri() {
      function verifyUri(ord, expectedUri) {
        verifyEq(baja.Ord.make(ord).toUri(), expectedUri);
      }
      
      it("adds /ord? and escapes", function addsOrdAndEscapes() {
        verifyUri("station:|slot:/Folder/Foo/Boo", 
                 "/ord?station:%7Cslot:/Folder/Foo/Boo");
        verifyUri("station:|slot:/Folder/Foo/Boo|virtual:/", 
                 "/ord?station:%7Cslot:/Folder/Foo/Boo%7Cvirtual:/");
      });
      
      it("drops local scheme", function dropsLocal() {
        verifyUri("local:|station:|slot:/Folder/Foo/Boo|virtual:/",
                  "/ord?station:%7Cslot:/Folder/Foo/Boo%7Cvirtual:/");
      });
      
      it("drops fox scheme", function dropsFox() {
        verifyUri("local:|fox:|station:|slot:/Folder/Foo/Boo|virtual:/", 
                  "/ord?station:%7Cslot:/Folder/Foo/Boo%7Cvirtual:/");
      });
      
      it("normalizes duplicate schemes", function normalizesDuplicate() {
        verifyUri("local:|station:|slot:/Folder/Foo/Boo|slot:Doo", 
                  "/ord?station:%7Cslot:/Folder/Foo/Boo/Doo");
      });
      
      it("leaves URLs as-is", function leavesUrls() {
        verifyUri("http://www.niagara-central.com", 
                  "http://www.niagara-central.com");
      });
      
      it("drops extra schemes from URLs", function dropsSchemesFromUrls() {
        verifyUri("local:|http://www.niagara-central.com", 
                  "http://www.niagara-central.com");
        verifyUri("local:|station:|http://www.niagara-central.com", 
                  "http://www.niagara-central.com");
      });
    });
    
    describe("relativizeToSession()", function Ord_relativizeToSession() {
      function verifyRelativize(ord, expected) {
        verifyEq(baja.Ord.make(ord).relativizeToSession().toString(), expected);
      }
      
      it("relativizes non-session to same", function noSession() {
        verifyRelativize("slot:/this/that", 
                         "slot:/this/that");
        verifyRelativize("station:|slot:/this/that", 
                         "station:|slot:/this/that");  
      });
      
      it("drops session schemes", function dropsSessionSchemes() {
        verifyRelativize("local:|station:|slot:/this/that",
                         "station:|slot:/this/that");
        verifyRelativize("local:|fox:|station:|slot:/this/that", 
                         "station:|slot:/this/that");
      });
    });
    
  });
  
  describe("baja.BatchResolve", function BatchResolve() {
    var stationOrd = "station:|slot:/",
        folderOrd = "station:|slot:/BajaScriptTestFolder",
        resolveOrd = "station:|slot:/BajaScriptTestFolder/batchResolveFolder",
        unknownOrd = "station:|slot:/BajaScriptTestFolder/IDoNotExist";
    
    function addAndSubscribeTestFolder(callbacks) {
      callbacks = callbackify(callbacks);
      
      runAndWait(function () {
        testFolder.add({
          slot: 'testFolder?',
          value: baja.$('testBajaScript:TestFolder'),
          ok: function (slot) {
            var perfFolder = testFolder.get(slot);
            
            new baja.Subscriber().subscribe({
              comps: perfFolder,
              ok: function () {
                callbacks.ok(perfFolder);
              },
              fail: callbacks.fail
            });
          }
        });
      }, callbacks);
    }
    
    function verifyResolve(ordArray, cb) {
      var callbacks = callbackify(cb);
      
      runAndWait(function () {
        new baja.BatchResolve(ordArray).resolve(callbacks);
      }, callbacks);
    }
    
    describe("constructor", function BatchResolve_constructor() {
      it("takes array of strings", function takesArrayOfStrings() {
        var br = new baja.BatchResolve([ stationOrd, folderOrd ]);
        verify(br);
      });
      
      it("takes array of Ords", function takesArrayOfOrds() {
        var br = new baja.BatchResolve([ baja.Ord.make(stationOrd), 
                                         baja.Ord.make(folderOrd) ]);
        verify(br);
      });
      
      it("takes empty array", function takesEmptyArray() {
        var br = new baja.BatchResolve([]);
        verify(br);
      });
      
      it("dies on undefined", function diesOnUndefined() {
        errTest(function () {
          var br = new baja.BatchResolve();
        }, undefinedError(Array));
      });
      
      it("dies on undefined in array", function diesOnUndefinedInArray() {
        errTest(function () {
          var br = new baja.BatchResolve([undefined]);
        });
      });
    });
    
    describe("getOrd()", function BatchResolve_getOrd() {
      it("returns ord at index", function returnsOrdAtIndex() {
        var br = new baja.BatchResolve([ stationOrd, folderOrd ]);
        verifyEq(br.getOrd(0).toString(), stationOrd);
        verifyEq(br.getOrd(1).toString(), folderOrd);
      });
      
      it("returns null for unknown index", function returnsNullUnknownIndex() {
        var br = new baja.BatchResolve([ stationOrd, folderOrd ]);
        expect(br.getOrd(2)).toBeNull();
      });
    });
    
    describe("resolve()", function BatchResolve_resolve() {
      var folderCreated = false,
          createResolveFolderCallbacks = callbackify({
            slot: 'batchResolveFolder',
            value: baja.$('baja:Folder')
          });
      
      beforeEach(function () {
        if (!testFolder.has('batchResolveFolder')) {
          runAndWait(function () {
            baja.Ord.make(folderOrd).get(function (folder) {
              folder.add(createResolveFolderCallbacks);
            });
          }, createResolveFolderCallbacks);
        }
      });

      it("resolves multiple ORDs", function resolvesMultiple() {
        verifyResolve([ stationOrd, folderOrd ], function () { verify(true); });
      });
      
      it("calls fail if ORD not found", function ordNotFound() {
        verifyResolve([ stationOrd, unknownOrd ], {
          ok: function () {
            failTest("should have called fail() for ord not found");
          }, 
          fail: function () {
            verify(true);
          }
        });
      });
      
      it("returns unsubscribed objects", function returnsUnsubscribed() {
        verifyResolve([ resolveOrd ], {
          ok: function () {
            verify(this.get(0).isMounted());
            //TODO: isSubscribed is true in browser, false in rhino. why?
//            verify(!this.get(0).isSubscribed());
          },
          subscriber: null
        });
      });
      
      it("immediately calls ok if no ORDs are given", function callsOkIfNoOrds() {
        verifyResolve([], function () {
          expect(true).toBe(true);
        });
      });
      
      it("accepts a subscriber", function acceptsSubscriber() {
        verifyResolve([ resolveOrd ], {
          ok: function () {
            verify(this.get(0).isMounted());
            verify(this.get(0).isSubscribed());
          },
          subscriber: new baja.Subscriber()
        });
      });
      
      it("dies if already resolved", function diesIfAlreadyResolved() {
        var br = new baja.BatchResolve([ stationOrd ]),
            doubleCallbacks = callbackify({
              ok: function () {
                failTest("should not allow a double resolve");
              },
              fail: function () {
                verify(true);
              }
            });
        
        runAndWait(function () {
          br.resolve({
            ok: function () {
              br.resolve(doubleCallbacks);
            }
          });
        }, doubleCallbacks);
      });
      
      describe("(detailed)", function detailed() {
        var batchResolveFolderOrd = 'local:|station:|slot:/batchResolveFolder',
            resolveBySlot1 = batchResolveFolderOrd + '/resolveBySlot/slot1',
            resolveBySlot2 = batchResolveFolderOrd + '/resolveBySlot/slot2',
            resolveByHandle = batchResolveFolderOrd + '/resolveByHandle',
            unknownScheme = batchResolveFolderOrd + '/unknownScheme',
            unknownSchemeWithBQL = unknownScheme + '|bql: select * from testBajaScript:TestFolder',
            preload = batchResolveFolderOrd + '/preload',
            preloadResolveBySlot = preload + '/resolveBySlot',
            preloadResolveByHandle = preload + '/resolveByHandle',
            virtualGateway = 'local:|station:|slot:/Drivers/NiagaraNetwork/virtuals',
            virtualServices = virtualGateway + '|virtual:/Services',
            preloadedHandle = '123',
            
            stationSpace,
            virtualSpace,
            station,
            oldStation = baja.station,
            stationTemplate,
            
            TEST_FOLDER = 'testBajaScript:TestFolder';

        function fake(type, subs) { 
          var tf = baja.$(type, subs);
          tf.getNavOrd = function () {
            var ord = this.getParent().getNavOrd() + '/' + this.getName();
            return baja.Ord.make(ord.replace('//', '/'));
          };
          tf.$space = stationSpace;
          tf.isMounted = function () {
            return true;
          };
          return tf;
        }
        
        function fakeStation() {
          var station = fake('baja:Station');
          station.getNavOrd = function () {
            return baja.Ord.make('local:|station:|slot:/');
          };
          return station;
        }
        
        // Preload Ords into our fake station, so our fake resolve/get has
        // something to return.
        function preloadOrds() {
          function doPreload(slotPath) {
            var comp = station,
                i,
                nameAtDepth;
            
            for (i = 0; i < slotPath.depth(); i++) {
              nameAtDepth = slotPath.nameAt(i);
              if (!comp.has(nameAtDepth)) {
                comp.add({
                  slot: nameAtDepth,
                  value: fake(TEST_FOLDER),
                  cx: { commit: true }
                });
              }
              comp = comp.get(nameAtDepth);
            }
          }
          
          var ords = Array.prototype.slice.call(arguments),
              i,
              j,
              queryList,
              slotPath,
              nameAtDepth;
          
          for (i = 0; i < ords.length; i++) {
            queryList = baja.Ord.make(ords[i]).parse();
            slotPath = queryList.get('slot');
            if (slotPath) {
              doPreload(slotPath);
            }
          }
        }
        
        function doBatchResolve(ords, callbacks) {
          callbacks = callbackify(callbacks);
          
          runAndWait(function () {
            new baja.BatchResolve(ords).resolve(callbacks);
          }, callbacks);
        }

        //try and setup offline, unmounted stubs to test batch resolution
        //without having to make network calls or try to get bajascript
        //to finagle loaded vs. unloaded components
        function setupStubs() {
          stationSpace = new baja.ComponentSpace();
         
          // one would expect this would resolve to a baja.VirtualGateway,
          // but in its current state this just resolves to a Component.
          virtualSpace = fake('baja:Component');

          var spaces = {
                'local:|station:': stationSpace,
                'local:|station:|slot:/Drivers/NiagaraNetwork/virtuals': virtualSpace
              };
          
          station = fakeStation();
          
          stationSpace.getAbsoluteOrd = function () {
            return baja.Ord.make('local:|station:');
          };
          
          stationSpace.$root = station;
          
          stationSpace.$callbacks = {
            // fake loadSlotPath - preload the ORDs into our fake station,
            // so resolve/get has something to return
            loadSlotPath: function (slotPathInfo, container, cb, importAsync) {
              var i,
                  info,
                  ords = [];
              
              for (i = 0; i < slotPathInfo.length; i++) {
                info = slotPathInfo[i];
                ords.push((info.o + '/' + info.sn).replace('//', '/'));
              }
              
              preloadOrds.apply(this, ords);
              cb.ok();
            },
            add: function () {
              //don't try to fire add events up to the station
            },
            subscribe: function () {
              // don't actually subscribe - just spy on this
            }
          };
           
          
          // fake Ord resolve
          spyOn(baja.Ord.prototype, 'resolve').andCallFake(function (obj) {
            obj = baja.objectify(obj, 'ok');
            
            var cb = obj.cb || new baja.comm.Callback(obj.ok, obj.fail),
                ordStr = this.toString(),
                space = spaces[ordStr],
                queries = this.parse(),
                slotPath = queries.get('slot'),
                i,
                nameAtDepth,
                value = station;
            
            if (space) {
              // resolved component space directly, so call ok with a fake
              // OrdTarget
              return obj.ok.call(space, {
                getComponent: function () { return space; },
                getObject: function () { return space; },
                object: space
              });
            } 
            
            // resolve a slotpath directly off of our fake station
            if (slotPath) {
              for (i = 0; value && (i < slotPath.depth()); i++) {
                nameAtDepth = slotPath.nameAt(i);
                value = value.get(nameAtDepth);
              }
              
              if (value) {
                cb.ok.call(value, {
                  getComponent: function () { return value; },
                  getObject: function () { return value; },
                  object: value
                });
              } else {
                cb.fail('could not resolve fake ORD');
              }
            }
          });
          
          // fake Ord get
          spyOn(baja.Ord.prototype, 'get').andCallFake(function (obj) {
            obj = baja.objectify(obj, 'ok');
            
            this.resolve({
              ok: function (target) {
                var comp = target.object;
                obj.ok.call(comp, comp);
              },
              fail: obj.fail
            });
          });
          
          spyOn(stationSpace.$callbacks, 'loadSlotPath').andCallThrough();
          spyOn(stationSpace.$callbacks, 'subscribe').andCallThrough();
        }
        
        afterEach(function () {
          baja.station = oldStation;
        });
        
        beforeEach(function () {
          oldStation = baja.station;
          setupStubs();
          baja.station = stationSpace;
          
          this.addMatchers({
            toHaveBeenCalledAs: function (expected) {
              var i,
                  actual = this.actual,
                  calls = actual.calls;
              
              for (i = 0; i < calls.length; i++) {
                if (String(calls[i].object) === String(expected)) {
                  return true;
                }
              }
              
              this.message = function () {
                return "Expected " + actual.identity + " to have been called on ORD " + 
                  expected;
              };
              
              return false;
            },
            toHaveBeenCalledNTimes: function (expected) {
              var actual = this.actual,
                  calls = actual.calls;
              
              this.message = function () {
                var i,
                    msg = "Expected function " + actual.identity + 
                      " to have been called " + expected + " times, not " + 
                      calls.length + ". (";
                
                for (i = 0; i < calls.length; i++) {
                  msg += calls[i].object;
                  if (i < calls.length - 1) {
                    msg += ', ';
                  }
                }
                
                msg += ')';
                
                return msg;
              };
              
              return calls.length === expected;
            }
          });
        });
        
        function lastGotOrd() {
          var call = baja.Ord.prototype.get.mostRecentCall;
          
          return call && call.object && String(call.object);
        }
        
        function lastResolvedOrd() {
          var call = baja.Ord.prototype.resolve.mostRecentCall;
          
          return call && call.object && String(call.object);
        }
        
        function lastSlotPathLoaded() {
          var args = stationSpace.getCallbacks().loadSlotPath.mostRecentCall.args;
          
          return args && args[0];
        }

        
        describe("when identifying and getting component space ORD", 
            function gettingComponentSpaceORD() {
          
          it("short circuits 'station:' space to baja.station, bypassing Ord#get()", 
              function shortCircuitsStationSpace() {
            
            doBatchResolve([resolveBySlot1], function () {
              expect(baja.Ord.prototype.get).not.toHaveBeenCalled();
            });
          });
          
          it("demarcates a component space just before slot scheme", 
              function demarcatesWithSlot() {
            preloadOrds(resolveBySlot1);
            
            doBatchResolve([resolveBySlot1], function () {
              expect(baja.Ord.prototype.get).not.toHaveBeenCalled();
            });
          });
          
          it("demarcates a component space just before virtual scheme", 
              function demarcatesWithVirtual() {
            
            preloadOrds(virtualGateway);
            doBatchResolve([virtualServices], function () {
              expect(lastGotOrd()).toBe(virtualGateway);
            });
          });
          
          it("will get() a component space ORD that is not local:|station:", 
              function getsComponentSpaceOrd() {
            
            preloadOrds(virtualGateway);
            doBatchResolve([virtualServices], function () {
              expect(lastGotOrd()).toBe(virtualGateway);
            });
          });
          
          it("demarcates a component space just before handle scheme", 
              function demarcatesWithHandle() {
            var ord = preload + '|h:' + preloadedHandle;
            
            preloadOrds(ord);
            doBatchResolve([ord], function () {
              expect(baja.Ord.prototype.get).not.toHaveBeenCalled();
            });
          });

          it("uses the base nav ord to get component space, if the ord to " +
              "resolve does not contain a component space", function usesBase() {
            
            preloadOrds(preload, resolveBySlot1);
            
            var base = station.get('batchResolveFolder').get('preload');
            
            doBatchResolve([resolveBySlot1.replace(/\S*station:\|/, '')], {
              ok: function () {
                expect(lastGotOrd()).toEqual(base.getNavOrd().toString());
              },
              base: base
            });
          });
          
          it("resolves unknown scheme directly, without a check for " +
              "component space", function resolvesUnknown() {
            
            preloadOrds(unknownScheme);
            
            doBatchResolve([unknownSchemeWithBQL], function () {
              expect(lastResolvedOrd()).toEqual(unknownSchemeWithBQL);
//              expect(baja.Ord.prototype.resolve).toHaveBeenCalledNTimes(1);
              expect(baja.Ord.prototype.resolve).toHaveBeenCalledAs(unknownSchemeWithBQL);
            });
          });
          
          it("will not get() the component space if the ORD contains an " +
              "unknown scheme", function wontCallGetIfUnknown() {
            
            preloadOrds(unknownScheme);
            
            doBatchResolve([ unknownSchemeWithBQL ], function () {
//              expect(baja.Ord.prototype.get).toHaveBeenCalledNTimes(0);
              expect(baja.Ord.prototype.get).not.toHaveBeenCalledAs(unknownScheme);
            });
          });
        });
        
        describe("when parsing slot paths from component spaces", 
            function parsingSlotPaths() {
          
          it("does not make a network call if the slot path is already fully " +
              "loaded", function slotPathFullyLoaded() {
            preloadOrds(preloadResolveBySlot);
            
            doBatchResolve([ preloadResolveBySlot ], function () {
              expect(lastSlotPathLoaded()).not.toBeDefined();
            });
          });
          
          it("escapes the slot path, if the path is a VirtualPath", 
              function escapesVirtualPath() {
            preloadOrds(virtualGateway);
            
            spyOn(baja.SlotPath, 'escape').andCallThrough();
            
            doBatchResolve([ virtualServices ], function () {
              expect(baja.SlotPath.escape).toHaveBeenCalledWith('Services');
            });
          });
          
          it("makes a network call if the slot path is not loaded", 
              function slotPathNotLoaded() {
            
            doBatchResolve([ resolveBySlot1 ], function () {
              var pathInfo = lastSlotPathLoaded(),
                  info = pathInfo[pathInfo.length - 1];
              
              expect(info).toEqual({
                o: 'slot:/batchResolveFolder/resolveBySlot',
                sn: 'slot1'
              });
            });
          });
        });
        
        describe("when resolving ORDs", function resolvingOrds() {
          it("will resolve each ord passed in", function resolvesEachOrd() {
            preloadOrds(resolveBySlot1, resolveBySlot2, resolveByHandle, 
                unknownScheme);
            
            doBatchResolve([ resolveBySlot1,
                             resolveBySlot2,
                             resolveByHandle,
                             unknownSchemeWithBQL ], function () {
              var resolve = baja.Ord.prototype.resolve;
              
              expect(resolve).toHaveBeenCalledAs(resolveBySlot1);
              expect(resolve).toHaveBeenCalledAs(resolveBySlot2);
              expect(resolve).toHaveBeenCalledAs(resolveByHandle);
              expect(resolve).toHaveBeenCalledAs(unknownSchemeWithBQL);
              
              // once per ORD.
//              expect(resolve).toHaveBeenCalledNTimes(4); 
            });
          });
          
          it("will make a network call to resolve an ORD with an unknown " +
              "scheme station-side", function resolvesUnknownStationside() {
            var resolve = baja.Ord.prototype.resolve;
            
            preloadOrds(unknownScheme);
            
            doBatchResolve([ unknownSchemeWithBQL ], function () {
//              expect(resolve).toHaveBeenCalledNTimes(1);
              expect(resolve).toHaveBeenCalledAs(unknownSchemeWithBQL);
            });
          });
        });
        
        describe("when performing subscription/leasing", 
            function whenSubscribingLeasing() {
          
          var subscriber;
          beforeEach(function () {
            subscriber = new baja.Subscriber();
            
            spyOn(subscriber, 'subscribe').andCallThrough();
          });
          
          it("groups subscription calls by component space", function groupsBySpace() {
            preloadOrds(resolveBySlot1, resolveBySlot2, resolveByHandle,
                unknownScheme);
            
            doBatchResolve([ resolveBySlot1,
                             resolveBySlot2,
                             resolveByHandle,
                             unknownSchemeWithBQL ], {
              
              ok: function () {
                //TODO: virtuals
                
                var subCalls = subscriber.subscribe.calls,
                    sub = subCalls[0].object;
                
                expect(subCalls.length).toBe(1);
                expect(sub.$comps.length).toBe(3);
              },
              subscriber: subscriber
            });
          });
          
          it("uses the subscriber passed into resolve() when components are " +
              "already loaded", function usesPassedInSubscriberLoaded() {
            
            preloadOrds(resolveBySlot1);
            
            doBatchResolve([ resolveBySlot1 ], {
              ok: function () {
                expect(subscriber.subscribe).toHaveBeenCalled();
                expect(subscriber.$comps[0].getNavOrd().toString())
                  .toBe(resolveBySlot1);
              },
              subscriber: subscriber
            });
          });
          
          it("uses the subscriber passed into resolve() when components are " +
              "not yet loaded", function usesPassedInSubscriberNotLoaded() {
            doBatchResolve([ resolveBySlot2 ], {
              ok: function () {
                expect(subscriber.subscribe).toHaveBeenCalled();
                expect(subscriber.$comps[0].getNavOrd().toString())
                  .toBe(resolveBySlot2);
              },
              subscriber: subscriber
            });
          });
          
          it("calls baja.Component.lease, if a lease period is provided",
              function callsComponentLease() {
            
            spyOn(baja.Component, 'lease');
            
            var leaseTime = 8675309;
            
            doBatchResolve([ resolveBySlot1, resolveBySlot2 ], {
              ok: function () {
                var obj = baja.Component.lease.mostRecentCall.args[0],
                    comps = obj.comps,
                    time = obj.time;
                
                expect(time).toBe(leaseTime);
                expect(comps.length).toBe(2);
                expect(comps[0].getNavOrd().toString()).toBe(resolveBySlot1);
                expect(comps[1].getNavOrd().toString()).toBe(resolveBySlot2);
              },
              lease: true,
              leaseTime: leaseTime
            });
          });
        }); 
      });
    });
    
    describe("get()", function BatchResolve_get() {
      it("returns resolved target at index", function returnsAtIndex() {
        verifyResolve([ stationOrd, folderOrd ], function () {
          var target1 = this.get(0),
              target2 = this.get(1);
          verifyEq(target1.getNavOrd().toString(), "local:|" + stationOrd);
          verifyEq(String(target1.getType()), "baja:Station");
          verifyEq(target2.getNavOrd().toString(), "local:|" + folderOrd);
          verifyEq(String(target2.getType()), "testBajaScript:TestFolder");
        });
      });
      
      it("returns null for index not found", function indexNotFound() {
        verifyResolve([ stationOrd, folderOrd ], function () {
          var that = this;
          expect(function () { that.getTarget(-1); }).toThrow('Unresolved ORD');
          expect(function () { that.getTarget(2); }).toThrow('Unresolved ORD');
        });
      });
    });
    
    describe("each()", function BatchResolve_each() {
      it("executes for each resolved object", function executesForEach() {
        verifyResolve([ stationOrd, folderOrd ], function () {
          var str = "";
          this.each(function (obj) {
            str += String(obj.getType());
          });
          verifyEq(str, "baja:StationtestBajaScript:TestFolder");
        });
      });
      
      it("stops and returns on truthy value", function stopsOnTruthy() {
        verifyResolve([ stationOrd, folderOrd, resolveOrd ], function () {
          var result = this.each(function (obj) {
            if (obj.getType().is('baja:Folder')) {
              return "I'm a folder!";
            }
          });
          verifyEq(result, "I'm a folder!");
        });
      });
    });
    
    describe("getTargets()", function BatchResolve_getTargets() {
      it("returns resolved Ord targets", function returnsTargets() {
        verifyResolve([ stationOrd, folderOrd, resolveOrd ], function () {
          var targets = this.getTargets();
          
          verifyEq(targets.length, 3);
          expect(targets[0].getComponent()).toBe(this.get(0));
          verifyEq(String(targets[0].getComponent().getType()), 'baja:Station');
          expect(targets[1].getComponent()).toBe(this.get(1));
          verifyEq(String(targets[1].getComponent().getType()), 'testBajaScript:TestFolder');
          expect(targets[2].getComponent()).toBe(this.get(2));
          verifyEq(String(targets[2].getComponent().getType()), 'baja:Folder');
        });
      });
      
      it("dies before calling resolve()", function diesPreResolve() {
        errTest(function () {
          new baja.BatchResolve([ stationOrd, folderOrd ]).getTargets();
        }, "Unresolved ORD");
      });
    });
    
    describe("getTarget()", function BatchResolve_getTarget() {
      it("returns target at index", function returnsTargetAtIndex() {
        verifyResolve([ stationOrd, folderOrd, resolveOrd ], function () {
          var target = this.getTarget(2);
          expect(target.getComponent()).toBe(this.get(2));
          verifyEq(String(target.getComponent().getNavOrd()), 
                   'local:|' + resolveOrd);
        });
      });
      
      it("throws error if index not found", function returnsNullIfNotFound() {
        verifyResolve([ stationOrd, folderOrd, resolveOrd ], function () {
          var that = this;
          expect(function () { that.getTarget(-1); }).toThrow('Unresolved ORD');
          expect(function () { that.getTarget(3); }).toThrow('Unresolved ORD');
        });
      });
    });
    
    describe("getTargetObjects()", function BatchResolve_getTargetObjects() {
      it("returns target objects", function returnsTargetObjects() {
        verifyResolve([ stationOrd, folderOrd, resolveOrd ], function () {
          var objects = this.getTargetObjects();
          
          verifyEq(String(objects[0].getNavOrd()), 'local:|' + stationOrd);
          verifyEq(String(objects[0].getType()), 'baja:Station');
          verifyEq(String(objects[1].getNavOrd()), 'local:|' + folderOrd);
          verifyEq(String(objects[1].getType()), 'testBajaScript:TestFolder');
          verifyEq(String(objects[2].getNavOrd()), 'local:|' + resolveOrd);
          verifyEq(String(objects[2].getType()), 'baja:Folder');
        });
      });
    });
    
    describe("isResolved()", function BatchResolve_isResolved() {
      it("returns true if ord at index is resolved", function returnsIfResolved() {
        verifyResolve([ stationOrd, folderOrd, unknownOrd ], {
          ok: function () {
            failTest("should not call ok if ord not found");
          },
          fail: function () {
            verify(this.isResolved(0));
            verify(this.isResolved(1));
            verify(!this.isResolved(2));
          }
        });
      });
    });
    
    describe("size()", function BatchResolve_size() {
      it("returns number of ords", function returnsNumberOfOrds() {
        verifyEq(new baja.BatchResolve([ stationOrd, folderOrd, unknownOrd ])
          .size(), 3);
        verifyEq(new baja.BatchResolve([]).size(), 0);
      });
    });
    
    // this dies in Rhino. uncomment for browsers
    describe("performance characteristics", function BatchResolve_performance() {
      function resolveOrds(number) {
        var resolveCallbacks = callbackify({
              ok: function () {
                verify(true);
              },
              fail: function (err) {
                failTest("failed to resolve:" + err);
              }
            });
        
        runAndWait(function () {
          addAndSubscribeTestFolder(function (perfFolder) {
            perfFolder.serverSideCall({
              typeSpec: "testBajaScript:TestBajaScriptServerSideHandler",
              methodName: 'createTestFolders',
              value: number,
              ok: function (val) {
                var perfFolderOrd = perfFolder.getNavOrd().toString(),
                    ords = [],
                    i,
                    br;
                for (i = 0; i < number; i++) {
                  ords.push(perfFolderOrd + '/testFolder' + (i || ''));
                }
                
                baja.station.sync({
                  ok: function () {
                    br = new baja.BatchResolve(ords);
                    br.resolve(resolveCallbacks);
                  }
                });
              },
              fail: failTest
            }); 
          });
        }, resolveCallbacks);
      }
      
      it("can resolve 10 ORDs at once", function resolves10Ords() {
        resolveOrds(10);
      });
      
      it("can resolve 25 ORDs at once", function resolves25Ords() {
        resolveOrds(25);
      });
        
      it("can resolve 50 ORDs at once", function resolves50Ords() {
        resolveOrds(50);
      });
    });
  });
  
  describe("baja.ViewQuery", function ViewQuery() {
    describe("constructor", function ViewQuery_constructor() {
      function verifyParamsLength(params, length) {
        var i = 0;
        baja.iterate(params, function () { i++; });
        verifyEq(i, length);
      }
      
      it("takes a single view id", function takesSingleViewId() {
        verifyEq(new baja.ViewQuery("module:MyType").getViewId(), "module:MyType");
        verifyEq(new baja.ViewQuery("fooView").getViewId(), "fooView");
      });
      
      it("takes view id plus params", function takesViewIdPlusParams() {
        var body = "viewId?name1=param1;name2=param2;name3=param3",
            query = new baja.ViewQuery(body),
            outParams = query.getParameters();

        verifyEq(query.getViewId(), "viewId");
        verifyEq(query.getBody(), body);
        verifyEq(outParams.name1, "param1");
        verifyEq(outParams.name2, "param2");
        verifyEq(outParams.name3, "param3");
        verifyParamsLength(outParams, 3);
      });

      it("takes an object literal", function takesLiteral() {
        var body = "viewId?name1=param1;name2=param2;name3=param3",
            id = "viewId",
            inParams = { name1: "param1", name2: "param2", name3: "param3" },
            query = new baja.ViewQuery({ id: id, params: inParams }),
            outParams = query.getParameters();
        
        verifyEq(query.getViewId(), id);
        verifyEq(query.getBody(), body);
        verifyEq(outParams.name1, "param1");
        verifyEq(outParams.name2, "param2");
        verifyEq(outParams.name3, "param3");
        verifyParamsLength(outParams, 3);
      });
      
      it("takes params only", function takesParamsOnly() {
        var body = "?name1=param1;name2=param2;name3=param3",
            query = new baja.ViewQuery(body),
            outParams = query.getParameters();
        
        verifyEq(query.getViewId(), "");
        verifyEq(query.getBody(), body);
        verifyEq(outParams.name1, "param1");
        verifyEq(outParams.name2, "param2");
        verifyEq(outParams.name3, "param3");
        verifyParamsLength(outParams, 3);
      });
      
      it("takes object literal with params only", function takesLiteralParams() {
        var body = "?name1=param1;name2=param2;name3=param3",
            inParams = { name1: "param1", name2: "param2", name3: "param3" },
            query = new baja.ViewQuery({ params: inParams }),
            outParams = query.getParameters();
        
        verifyEq(query.getViewId(), "");
        verifyEq(query.getBody(), body);
        verifyEq(outParams.name1, "param1");
        verifyEq(outParams.name2, "param2");
        verifyEq(outParams.name3, "param3");
        verifyParamsLength(outParams, 3);
      });
      
      it("dies on empty string", function diesOnEmptyString() {
        errTest(function() {
          var query = new baja.ViewQuery("");
        }, "Invalid view query: ");
      });
    });
  });
  
  describe("baja.OrdQueryList", function OrdQueryList() {
    describe("add()", function OrdQueryList_add() {
      it("appends a query", function appendsAQuery() {
        var list = toQueryList("station:|slot:"),
            query = new SpyQuery("hi");
        verifyEq(list.size(), 2);
        list.add(query);
        expect(list.get(2)).toBe(query);
      });
      
      it("resets $hasUnknown", function resets$hasUnknown() {
        var list = toQueryList("station:|slot:"),
            query = new SpyQuery("hi");
        expect(list.$hasUnknown).toBeUndefined();
        list.$hasUnknown = true;
        list.add(query);
        expect(list.$hasUnknown).toBeUndefined();
      });
    });
    
    describe("isEmpty()", function OrdQueryList_isEmpty() {
      it("returns true if list is empty", function () {
        var list = toQueryList("");
        verify(list.isEmpty());
        list.add(new SpyQuery());
        verify(!list.isEmpty());
      });
    });
    
    describe("hasUnknown()", function OrdQueryList_hasUnknown() {
      it("returns true if it has a query with UnknownScheme", function hasUnknownScheme() {
        var list = toQueryList("station:");
        verify(!list.hasUnknown());
        list.add(new SpyQuery({ scheme: new baja.StationScheme() }));
        verify(!list.hasUnknown());
        list.add(new SpyQuery({ scheme: new baja.UnknownScheme() }));
        verify(list.hasUnknown());
        list.remove(2);
        verify(!list.hasUnknown());
      });
    });
    
    describe("getCursor()", function OrdQueryList_getCursor() {
      it("returns a cursor to iterate over queries", function returnsCursor() {
        var list = toQueryList("local:|station:|slot:/"),
            cursor = list.getCursor();
        verify(cursor.next());
        expect(cursor.get()).toBe(list.get(0));
        verify(cursor.next());
        expect(cursor.get()).toBe(list.get(1));
        verify(cursor.next());
        expect(cursor.get()).toBe(list.get(2));
        verify(!cursor.next());
      });
      
      describe(".each()", function OrdQueryList_getCursor_each() {
        it("executes a function for each query", function executesForEach() {
          var cursor = toQueryList("local:|station:|slot:").getCursor(),
              s = "";
          cursor.each(function (query) {
            s += query.getSchemeName() + " :) ";
          });
          
          verifyEq(s, "local :) station :) slot :) ");
        });
      });
    });
    
    describe("normalize()", function OrdQueryList_normalize() {
      it("normalizes the query list", function normalizes() {
        var list = toQueryList("local:|local:|station:|station:|slot:/|slot:/");
        list.normalize();
        verifyEq(list.toString(), "local:|station:|slot:/");
      });
    });
    
    describe("get()", function OrdQueryList_get() {
      it("retrieves by number", function retrievesByNumber() {
        var list = toQueryList("local:|station:|slot:");
        verifyEq(list.get(0).getSchemeName(), "local");
        verifyEq(list.get(1).getSchemeName(), "station");
        verifyEq(list.get(2).getSchemeName(), "slot");
      });
      
      it("retrieves by scheme name", function retrievesBySchemeName() {
        var list = toQueryList("local:|station:|slot:");
        verifyEq(list.get('local').getSchemeName(), 'local');
        verifyEq(list.get('station').getSchemeName(), 'station');
        verifyEq(list.get('slot').getSchemeName(), 'slot');
      });
      
      it("returns null for not found", function returnsNullIfNotFound() {
        var list = toQueryList("local:|station:|slot:");
        expect(list.get(-1)).toBeNull();
        expect(list.get(3)).toBeNull();
        expect(list.get(0)).not.toBeNull();
        expect(list.get('WHARRGARBL')).toBeNull();
        expect(list.get('station')).not.toBeNull();
      });
    });
    
    describe("set()", function OrdQueryList_set() {
      it("sets query at given index", function setsAtIndex() {
        var list = toQueryList("local:|station:|slot:");
        verifyEq(list.get(1).getSchemeName(), 'station');
        list.set(1, new SpyQuery({ schemeName: 'virtual' }));
        verifyEq(list.get(1).getSchemeName(), 'virtual');
      });
      
      it("resets $hasUnknown", function resets$hasUnknown() {
        var list = toQueryList("local:|WHARRGARBL:|slot:");
        list.hasUnknown();
        verify(list.$hasUnknown);
        list.set(1, new SpyQuery({ schemeName: 'station' }));
        expect(list.$hasUnknown).toBeUndefined();
      });
      
      it("dies for invalid index", function diesForInvalidIndex() {
        var list = toQueryList("local:|station:|slot:");
        errTest(function () {
          list.set(4, new SpyQuery());
        }, "Invalid index (4)");
        errTest(function () {
          list.set(-1, new SpyQuery());
        }, "Invalid index (-1)");
      });
    });
    
    describe("remove()", function OrdQueryList_remove() {
      it("removes and returns query at index", function removesAndReturns() {
        var list = toQueryList("local:|station:|slot:"),
            query = list.remove(1);
        verifyEq(query.getSchemeName(), 'station');
        verifyEq(list.toString(), 'local:|slot:');
      });
      
      it("returns null if not found", function returnsNullIfNotFound() {
        var list = toQueryList("local:|station:|slot:");
        expect(list.remove(-1)).toBeNull();
        expect(list.remove(3)).toBeNull();
      });
      
      it("resets $hasUnknown if query removed", function resets$hasUnknown() {
        var list = toQueryList("local:|WHARRGARBL:|slot:");
        list.hasUnknown();
        verify(list.$hasUnknown);
        list.remove(-1);
        verify(list.$hasUnknown);
        list.remove(0);
        expect(list.$hasUnknown).toBeUndefined();
      });
    });
    
    describe("size()", function OrdQueryList_size() {
      it("returns number of queries", function returnsNumberOfQueries() {
        var list = toQueryList("");
        verifyEq(list.size(), 0);
        list.add(new SpyQuery());
        verifyEq(list.size(), 1);
        list.add(new SpyQuery());
        verifyEq(list.size(), 2);
        list.remove(0);
        verifyEq(list.size(), 1);
      });
    });
  });
  
  describe("baja.OrdScheme", function OrdScheme() {
    
    function normalized(list, index) {
      return list.get(index).normalize(list, index);
    }
    
    function verifyDirectResolveNext(scheme) {
      var target = {},
          cursor = new SpyCursor(),
          arg;
      
      scheme.resolve(target, undefined, cursor);
      arg = cursor.resolveNext.mostRecentCall.args[0];
      expect(arg.base).toBe(target);
    }
    
    describe("lookup (static)", function OrdScheme_lookup() {
      it("returns a registered ord scheme", function returnsRegistered() {
        verify(baja.OrdScheme.lookup('station'));
      });
      
      it("returns UnknownScheme.DEFAULT if not found", function notFound() {
        expect(baja.OrdScheme.lookup('WHARRGARBL'))
          .toBe(baja.UnknownScheme.DEFAULT);
      });
    });
    
    describe("subclass baja.FoxScheme", function FoxScheme() {
      var foxScheme = baja.OrdScheme.lookup("fox");
      it("is registered", function isRegistered() {
        expect(foxScheme).toBeDefined();
        expect(foxScheme).not.toBe(baja.UnknownScheme.DEFAULT);
      });
      
      describe("resolve()", function FoxScheme_resolve() {
        it("resolves next with no action", function resolvesWithNoAction() {
          verifyDirectResolveNext(foxScheme);
        });
      });
      
      describe("normalize()", function FoxScheme_normalize() {
        it("returns same for single fox string", function singleFox() {
          var list = toQueryList('fox:');
          verify(!normalized(list, 0));
          verifyEq(list.toString(), 'fox:');
        });
        
        it("makes no change if previous is host", function previousIsHost() {
          var list = toQueryList('local:|fox:');
          verify(!normalized(list, 0));
          verify(!normalized(list, 1));
          verifyEq(list.toString(), 'local:|fox:');
        });
        
        it("trims left to nearest host", function trimsToNearestHost() {
          var list = toQueryList('local:|station:|slot:|fox:');
          verify(!normalized(list, 0));
          verify(!normalized(list, 1));
          verify(!normalized(list, 2));
          verify(normalized(list, 3));
          verifyEq(list.toString(), 'local:|fox:');
        });
      });
    });

    describe("subclass baja.HandleScheme", function HandleScheme() {
      var handleScheme = baja.OrdScheme.lookup("h");
      
      function HandleQuery(body) {
        return new SpyQuery({
          scheme: handleScheme,
          schemeName: 'h',
          body: body
        });
      }
      
      it("is registered", function isRegistered() {
        expect(handleScheme).toBeDefined();
        expect(handleScheme).not.toBe(baja.UnknownScheme.DEFAULT);
      });
      
      describe("resolve()", function HandleScheme_resolve() {
        
        function verifyHandleResolve(target, handle, expectedType, expectedOrd) {
          var cursor = new SpyCursor(),
              query = new HandleQuery(handle);
    
          runAndWait(function () {
            handleScheme.resolve(target, query, cursor);
          }, cursor);
          
          runs(function () {
            var resolved = cursor.resolveNext.mostRecentCall.args[0].object;
            verify(resolved.getType().is(expectedType));
            verifyEq(resolved.getHandle(), handle);
            verify(resolved.isMounted());
            verifyEq(resolved.getNavOrd().toString(), expectedOrd);
          });
        }
        
        it("dies for invalid OrdTarget object", function diesForInvalid() {
          errTest(function () {
            handleScheme.resolve({ object: [] }, 
                new HandleQuery('1'));
          }, 'Not based via ComponentSpace. Invalid Object');
        });
        
        it("dies for missing component space", function diesForMissingSpace() {
          errTest(function () {
            handleScheme.resolve({ object: baja.$('control:NumericWritable') }, 
                new HandleQuery('1'));
          }, 'Not based via ComponentSpace');
        });
        
        it("resolves on mounted component", function resolvesMounted() {
          var target = { object: testFolder },
              handle = ordTestFolder.getHandle(),
              expectedType = 'baja:Folder',
              expectedOrd = String(ordTestFolder.getNavOrd());
          verifyHandleResolve(target, handle, expectedType, expectedOrd);
        });
        
        it("resolves user service", function resolvesUserService() {
          var getCallbacks = callbackify(function (userService) {
            var target = { object: testFolder },
                handle = userService.getHandle(),
                expectedType = 'baja:UserService',
                expectedOrd = String(userService.getNavOrd());
            verifyHandleResolve(target, handle, expectedType, expectedOrd);
          });
          
          runAndWait(function () {
            baja.Ord.make('station:|slot:/Services/UserService').get(getCallbacks);
          }, getCallbacks);
        });
        
        it("resolves using component space", function resolvesComponentSpace() {
          var getCallbacks = callbackify(function (webService) {
            var target = { object: baja.nav.localhost.station },
                handle = webService.getHandle(),
                expectedType = 'web:WebService',
                expectedOrd = String(webService.getNavOrd());
            verifyHandleResolve(target, handle, expectedType, expectedOrd);
          });
          
          runAndWait(function () {
            baja.Ord.make('station:|slot:/Services/WebService').get(getCallbacks);
          }, getCallbacks);
        });
        
        it("resolves on a manually mounted component", function resolvesManuallyMounted() {
          function verifyUnmountedHandleResolve(target, handle, expected) {
            var query = new HandleQuery(handle),
                cursor = new SpyCursor();
            
            runAndWait(function () {
              handleScheme.resolve(target, query, cursor);
            }, cursor);
            
            runs(function () {
              var target = cursor.resolveNext.mostRecentCall.args[0];
              expect(target.object).toBe(expected);
            });
          }
          
          var c1, c2, c3, c4, c5, cs = new baja.ComponentSpace();
          c1 = baja.$('baja:Component', {
            kid: c2 = baja.$('baja:Component', {
              kid: c3 = baja.$('baja:Component', {
                kid: c4 = baja.$('baja:Component', {
                  kid: c5 = baja.$('baja:Component')
                })
              }) 
            })
          });
          
          c1.$handle = "1";
          c2.$handle = "2";
          c3.$handle = "3";
          c4.$handle = "4";
          c5.$handle = "5";
          
          // Mount Components inside a Component Space
          cs.$fw("mount", c1);
          
          verifyUnmountedHandleResolve({ object: cs }, '1', c1);
        });
        
        it("calls fail if handle not found", function callsFailIfNotFound() {
          var target = { object: baja.nav.localhost.station },
              query = new HandleQuery("WHARRGARBL"),
              cursor = new SpyCursor(),
              resolveCallbacks = callbackify({
                ok: function () {
                  failTest("should not call ok if not found");
                },
                fail: function () {
                  verify(true);
                }
              }),
              options = { callback: resolveCallbacks };
          
          runAndWait(function () {
            handleScheme.resolve(target, query, cursor, options);
          }, resolveCallbacks);
        });
      });
    });
    
    describe("subclass baja.HttpScheme", function HttpScheme() {
      var httpScheme = baja.OrdScheme.lookup("http");
      it("is registered", function isRegistered() {
        expect(httpScheme).toBeDefined();
        expect(httpScheme).not.toBe(baja.UnknownScheme.DEFAULT);
      });
      
      describe("normalize()", function HttpScheme_normalize() {
        it("returns same for single http string", function singleHttp() {
          var list = toQueryList("http://www.google.com");
          verify(!normalized(list, 0));
          verifyEq(list.toString(), "http://www.google.com");
        });
        
        it("trims all previous", function trimsToStart() {
          var list = toQueryList("station:|slot:/|http://www.google.com");
          verify(!normalized(list, 0));
          verify(!normalized(list, 1));
          verify(normalized(list, 2));
          verifyEq(list.toString(), "http://www.google.com");
        });
      });
      
      describe("resolve()", function HttpScheme_resolve() {
        it("resolves next with no action", function resolvesWithNoAction() {
          verifyDirectResolveNext(httpScheme);
        });
      });
    });
    
    describe("subclass baja.LocalScheme", function LocalScheme() {
      var localScheme = baja.OrdScheme.lookup("local");
      it("is registered", function isRegistered() {
        expect(localScheme).toBeDefined();
        expect(localScheme).not.toBe(baja.UnknownScheme.DEFAULT);
      });
      
      describe("resolve()", function LocalScheme_resolve() {
        it("resolves directly to localhost", function resolvesToLocalhost() {
          var target = {},
              cursor = new SpyCursor(),
              arg;
          localScheme.resolve(target, undefined, cursor);
          arg = cursor.resolveNext.mostRecentCall.args[0];
          expect(arg.base).toBe(target);
          expect(arg.object).toBe(baja.nav.localhost);
        });
      });
      
      describe("normalize()", function LocalScheme_normalize() {
        it("returns same string for single local scheme", function singleLocal() {
          var list = toQueryList('local:');
          verify(!normalized(list, 0));
          verifyEq(list.toString(), 'local:');
        });
        
        it("trims all previous", function trimsToStart() {
          var list = toQueryList('local:|station:|slot:/|local:|service:baja:UserService');
          verify(!normalized(list, 0));
          verify(!normalized(list, 1));
          verify(!normalized(list, 2));
          verify(normalized(list, 3));
          verifyEq(list.toString(), 'local:|service:baja:UserService');
        });
      });
    });
    
    describe("subclass baja.ServiceScheme", function ServiceScheme() {
      var serviceScheme = baja.OrdScheme.lookup("service");
      
      function ServiceQuery(body) {
        return new SpyQuery({
          scheme: serviceScheme,
          schemeName: 'service',
          body: body
        });
      }
      
      it("is registered", function isRegistered() {
        expect(serviceScheme).toBeDefined();
        expect(serviceScheme).not.toBe(baja.UnknownScheme.DEFAULT);
      });
      
      describe("resolve()", function ServiceScheme_resolve() {
        function verifyServiceResolve(target, expectedType, expectedOrd) {
          var cursor = new SpyCursor(),
              query = new ServiceQuery(expectedType);
    
          runAndWait(function () {
            serviceScheme.resolve(target, query, cursor);
          }, cursor);
          
          runs(function () {
            var resolved = cursor.resolveNext.mostRecentCall.args[0].object;
            verify(resolved.getType().is(expectedType));
            verify(resolved.isMounted());
            verifyEq(resolved.getNavOrd().toString(), expectedOrd);
          });
        }
        
        function resolveAllServices(target) {
          var services = 'local:|station:|slot:/Services/';
          
          verifyServiceResolve(target, 'baja:UserService', 
              services + 'UserService');
          verifyServiceResolve(target, 'web:WebService',
              services + 'WebService');
          verifyServiceResolve(target, 'box:BoxService',
              services + 'BoxService');
        }
        
        it("dies for invalid OrdTarget object", function diesForInvalid() {
          errTest(function () {
            serviceScheme.resolve({ object: [] }, 
                new ServiceQuery('baja:UserService'));
          }, 'Not based via ComponentSpace. Invalid Object');
        });
        
        it("dies for missing component space", function diesForMissingSpace() {
          errTest(function () {
            serviceScheme.resolve({ object: baja.$('baja:UserService') }, 
                new ServiceQuery('baja:UserService'));
          }, 'Not based via ComponentSpace');
        });
        
        it("dies for unmounted component space", function diesForUnmountedSpace() {
          errTest(function () {
            serviceScheme.resolve({ object: baja.$('baja:ComponentSpace') },
                new ServiceQuery('baja:UserService'));
          }, 'Unable to resolve Service: baja:UserService');
        });
        
        it("resolves using a Component", function resolvesComponent() {
          resolveAllServices({ object: testFolder });
        });
        
        it("resolves using the station", function resolvesStation() {
          resolveAllServices({ object: baja.nav.localhost.station });
        });
        
        it("resolves using baja.nav.localhost", function resolvesLocalhost() {
          resolveAllServices({ object: baja.nav.localhost });
        });
        
        it("resolves any ISession with station property", function resolvesISession() {
          var object = baja.$('baja:LocalHost');
          object.station = baja.nav.localhost.station;
          resolveAllServices({ object: object });
        });
      });
      
      describe("normalize()", function ServiceScheme_normalize() {
        it("does nothing and returns false", function doesNothing() {
          var ord = 'service:baja:UserService|service:web:WebService',
              list = toQueryList(ord);
          verify(!normalized(list, 0));
          verify(!normalized(list, 1));
          verifyEq(list.toString(), ord);
        });
      });
    });
    
    describe("subclass baja.SlotScheme", function SlotScheme() {
      var slotScheme = baja.OrdScheme.lookup("slot");
      it("is registered", function isRegistered() {
        expect(slotScheme).toBeDefined();
        expect(slotScheme).not.toBe(baja.UnknownScheme.DEFAULT);
      });
      
      describe("parse()", function SlotScheme_parse() {
        it("returns a baja.SlotPath", function returnsSlotPath() {
          var body = 'a/b/c',
              path = new baja.SlotScheme().parse('slot', body);
          verifyEq(path.getBody(), body);
          verifyEq(path.getSchemeName(), 'slot');
        });
      });
      
      describe("normalize()", function SlotScheme_normalize() {
        it("normalizes 1 slot scheme to same", function singleSlot() {
          var list = toQueryList('slot:/this/that/foo/boo');
          verify(!normalized(list, 0));
          verifyEq(list.toString(), 'slot:/this/that/foo/boo');
        });
        
        it("merges 2 slot schemes", function twoSlots() {
          var list = toQueryList('slot:/this/that|slot:foo/boo');
          verify(normalized(list, 0));
          verify(!normalized(list, 0));
          verifyEq(list.toString(), "slot:/this/that/foo/boo");
        });
        
        it("merges 3 slot schemes", function threeSlots() {
          var list = toQueryList('slot:/this/that|slot:foo/boo|slot:goo');
          verify(normalized(list, 0));
          verify(normalized(list, 0));
          verify(!normalized(list, 0));
          verifyEq(list.toString(), 'slot:/this/that/foo/boo/goo');
        });
      });
      
      describe("resolve()", function SlotScheme_resolve() {
        var comp = new baja.Component(),
            test = new baja.Component(),
            that = new baja.Component(),
            foo = new baja.Component(),
            bar = new baja.Component(),
            statusNum = baja.$('baja:StatusNumeric'),
            slotPath = "slot:test/that/foo/bar/statusNum/value";
        
        statusNum.setValue(34.4);
        
        comp.add({ slot: "test", value: test });
        test.add({ slot: "that", value: that });
        that.add({ slot: "foo", value: foo });
        foo.add({ slot: "bar", value: bar });
        bar.add({ slot: "statusNum", value: statusNum });
        
        /**
         * verify slot path resolution - expected is either the actual
         * BValue we expect the slot path to resolve to, or a verification
         * function on the resolved OrdTarget
         */
        function verifyResolve(target, path, expected, netcall) {
          var query = new baja.SlotPath(path),
              cursor = new SpyCursor(),
              options = {
                callback: callbackify(),
                full: false
              };
          
          runAndWait(function () {
            slotScheme.resolve(target, query, cursor, options, netcall);
          }, cursor);
          
          runs(function () {
            var target = cursor.resolveNext.mostRecentCall.args[0],
                object = target.object;
            if (typeof expected === 'function') {
              expected(target);
            } else {
              expect(object).toBe(expected);
            }
          });
        }
        
        it("resolves empty path", function resolvesEmptyPath() {
          verifyResolve({ object: comp }, "", comp);
        });
        
        it("resolves relative path to complex", function resolvesToComplex() {
          verifyResolve({ object: comp }, "test/that/foo", foo);
          verifyResolve({ object: comp }, "test/that/foo/bar", bar);
          verifyResolve({ object: test }, "../test/that/foo/bar", bar);
          verifyResolve({ object: that }, "../that/foo/bar", bar);
          verifyResolve({ object: foo }, "../../that/foo/bar", bar);
        });
        
        it("resolves relative path to simple", function resolvesToSimple() {
          function verifyTarget(target) {
            var object = target.object,
                propertyPath = target.propertyPath,
                slot = target.slot;
            verifyEq(object.getType().getTypeSpec(), "baja:Double");
            verifyEq(object.valueOf(), 34.4);
            verifyEq(propertyPath.length, 2);
            verifyEq(propertyPath[0].getName(), "statusNum");
            verifyEq(propertyPath[1].getName(), "value");
            verifyEq(slot.getName(), "statusNum");  
          }
        
          verifyResolve({ object: comp }, "test/that/foo/bar/statusNum/value",
              verifyTarget);
        });
        
        //TODO: doesn't work because it is dependent on no other tests having
        //resolved the services container first. find out if there is a way to
        //"reset" bajascript's memory so it doesn't have a cached instance with
        //preloaded slots anymore
        xit("does not load slots when not resolving full", function nonFull() {
          var target = { object: baja.nav.localhost.station },
              query = new baja.SlotPath('/Services/WebService'),
              cursor = new SpyCursor(),
              options = {
                callback: callbackify(),
                full: false
              },
              netcall = true;
          
          runAndWait(function () {
            slotScheme.resolve(target, query, cursor, options, netcall);
          }, cursor);
          
          runs(function () {
            var arg = cursor.resolveNext.mostRecentCall.args[0],
                object = arg.object,
                parentSlots = object.getParent().getSlots().toArray();
            verify(object.getType().is('web:WebService'));
            //only loaded the one slot because full = false
            verifyEq(parentSlots.length, 1);
            verifyEq(parentSlots[0].getName(), 'WebService');
          });
        });
        
        it("loads slots when resolving full", function resolveFull() {
          var target = { object: baja.nav.localhost.station },
              query = new baja.SlotPath('/Services/WebService'),
              cursor = new SpyCursor(),
              options = { 
                callback: callbackify(),
                full: true 
              },
              netcall = true;
          
          runAndWait(function () {
            slotScheme.resolve(target, query, cursor, options, netcall);
          }, cursor);
          
          runs(function () {
            var arg = cursor.resolveNext.mostRecentCall.args[0],
                object = arg.object,
                parentSlots = object.getParent().getSlots().toArray();
            verify(object.getType().is('web:WebService'));
            //should load all slots because full = true
            verify(parentSlots.length > 1);
            verify(parentSlots.join().indexOf('WebService') >= 0);
            verify(parentSlots.join().indexOf('UserService') >= 0);
            verify(parentSlots.join().indexOf('BoxService') >= 0);
          });
        });
      });
    });
    
    describe("subclass baja.StationScheme", function StationScheme() {
      var stationScheme = baja.OrdScheme.lookup("station");
      
      it("is registered", function isRegistered() {
        expect(stationScheme).toBeDefined();
        expect(stationScheme).not.toBe(baja.UnknownScheme.DEFAULT);
      });
      
      describe("normalize()", function StationScheme_normalize() {
        it("normalizes single query to same", function singleQuery() {
          var list = toQueryList('station:');
          verify(!normalized(list, 0));
          verifyEq(list.toString(), 'station:');
        });
        
        it("normalizes double query to one", function doubleQuery() {
          var list = toQueryList("station:|station:");
          verify(!normalized(list, 0));
          verify(normalized(list, 1));
          verifyEq(list.size(), 1);
          verifyEq(list.toString(), 'station:');
        });
        
        it("trims left to session", function trimsToSession() {
          var list = toQueryList("local:|fox:|station:|station:|station:");
          spyOn(list, 'remove').andCallThrough();
          verifyEq(list.size(), 5);
          verify(!normalized(list, 2));
          verify(normalized(list, 4));
          verifyEq(list.size(), 3);
          verifyEq(list.remove.argsForCall[0][0], 3); //first chops 2nd station
          verifyEq(list.remove.argsForCall[1][0], 2); //and then chops 1st
          verifyEq(list.toString(), 'local:|fox:|station:');
        });
      });
      
      describe("resolve()", function StationScheme_resolve() {
        it("accepts a ComponentSpace", function acceptsComponentSpace() {
          var space = new baja.ComponentSpace("testSpace"),
              target = { object: space },
              query = null,
              cursor = new SpyCursor(),
              arg;
          
          new baja.StationScheme().resolve(target, query, cursor);
          
          arg = cursor.resolveNext.mostRecentCall.args[0];
          expect(arg.object).toBe(space);
        });
        
        it("accepts mounted station root", function acceptsMountedStation() {
          var callbacks = callbackify(function (station) {
            var target = { object: station },
                query = null,
                cursor = new SpyCursor(),
                arg;
            
            new baja.StationScheme().resolve(target, query, cursor);
            
            arg = cursor.resolveNext.mostRecentCall.args[0];
            expect(arg.object).toBe(station);
          });
          
          runAndWait(function () {
            baja.Ord.make('station:').get(callbacks);
          }, callbacks);
        });
        
        it("falls back to baja.nav.localhost.station if object is not a space", function fallsBack() {
          var writable = baja.$('control:NumericWritable'),
              target = { object: writable },
              query = null,
              cursor = new SpyCursor(),
              arg;
          
          new baja.StationScheme().resolve(target, query, cursor);
          arg = cursor.resolveNext.mostRecentCall.args[0];
          expect(arg.object).toBe(baja.nav.localhost.station);
        });
      });
    });
    
    describe("subclass baja.UnknownScheme", function UnknownScheme() {
      var unknownScheme = baja.OrdScheme.lookup("WHARRGARBL");
      it("is registered for not found", function isRegisteredForNotFound() {
        expect(unknownScheme).toBeDefined();
        expect(unknownScheme).toBe(baja.UnknownScheme.DEFAULT);
      });
      
      describe("resolve()", function UnknownScheme_resolve() {
        it("always calls fail()", function callsFail() {
          var query = new SpyQuery({ schemeName: 'unknownScheme' }),
              options = { callback: { fail: function () {} } };
          
          spyOn(options.callback, 'fail');
          
          unknownScheme.resolve(undefined, query, undefined, options);
          expect(options.callback.fail).toHaveBeenCalledWith(
              "Unknown BajaScript ORD Scheme: unknownScheme");
        });
      });
    });
    
    describe("subclass baja.ViewScheme", function ViewScheme() {
      var viewScheme = baja.OrdScheme.lookup("view");
      it("is registered", function isRegistered() {
        expect(viewScheme).toBeDefined();
        expect(viewScheme).not.toBe(baja.UnknownScheme.DEFAULT);
      });
      
      describe("parse()", function ViewScheme_parse() {
        it("returns instance of ViewQuery", function returnsViewQuery() {
          var vq = viewScheme.parse("view", "viewId?param=value");
          verifyEq(vq.getViewId(), "viewId");
          verifyEq(vq.getParameters().param, "value");
        });
      });
      
      describe("resolve()", function ViewScheme_resolve() {
        it("appends view info to the OrdTarget", function appendsViewInfo() {
          var query = new baja.ViewQuery("viewId?param=value"),
              target = {},
              cursor = new SpyCursor();
          
          viewScheme.resolve(target, query, cursor);
          verifyEq(target.view.id, "viewId");
          verifyEq(target.view.params.param, "value");
        });
      });
      
      describe("normalize()", function ViewScheme_normalize() {
        it("returns same for single view scheme", function singleView() {
          var list = toQueryList('view:viewId');
          verify(!normalized(list, 0));
          verifyEq(list.toString(), 'view:viewId');
        });
        
        it("chops a subsequent view query", function chopsSubsequent() {
          var list = toQueryList('station:|view:view1|view:view2');
          verify(!normalized(list, 0));
          verify(normalized(list, 1));
          verify(!normalized(list, 1));
          verifyEq(list.toString(), 'station:|view:view1');
        });
        
        it("chops itself if not last in list", function chopsSelf() {
          var list = toQueryList('station:|view:view1|slot:/|view:view2');
          verify(!normalized(list, 0));
          verify(normalized(list, 1));
          verifyEq(list.toString(), 'station:|slot:/|view:view2');
          verify(!normalized(list, 1));
          verify(!normalized(list, 2));
          verifyEq(list.toString(), 'station:|slot:/|view:view2');
        });
      });
    });
  });
});
