/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * @module nmodule/driver/rc/wb/mgr/DeviceMgrModel
 */

define(['baja!', 'underscore', 'nmodule/webEditors/rc/wb/mgr/model/folderMgrModelMixin', 'nmodule/webEditors/rc/wb/mgr/model/MgrModel', 'nmodule/webEditors/rc/wb/table/model/Row', 'nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource', 'baja!baja:Folder,driver:Device'], function (baja, _, addFolderMgrModelMixin, MgrModel, Row, ContainerComponentSource) {
  'use strict';

  var DEVICE_TYPE = baja.lt('driver:Device'),
    CHANGE_EVENT_THROTTLE_MS = 1000;

  /**
   * Default filter for the component source if one is not specified by the
   * constructor parameters coming from a derived class. This one will simply
   * allow any BDevice derived classes through the filter and, if a type was
   * specified, folders too.
   *
   * @param {Object} params - parameters passed to the model constructor.
   * @returns {Function} a function used by the component source when filtering slots.
   */
  function makeDefaultSourceFilter(params) {
    return function (prop) {
      var type = prop.getType(),
        folderType = params.folderType;
      if (prop.getFlags() & baja.Flags.HIDDEN) {
        return false;
      }
      return !!(type.is(DEVICE_TYPE) || folderType && type.is(folderType));
    };
  }

  /**
   * API Status: **Development**
   * 
   * A `MgrModel` type for a `DeviceMgr` derived type as an agent on a driver's
   * BDeviceNetwork or BDeviceFolder concrete type.
   *
   * @class
   * @alias module:nmodule/driver/rc/wb/mgr/DeviceMgrModel
   * @extends module:nmodule/webEditors/rc/wb/mgr/model/MgrModel
   *
   * @param {Object} params object containing the constructor parameters
   * @param {baja.Component} params.component the component containing the devices to
   * be shown in the manager, typically a device network or device folder.
   * @param {string|Type} [params.folderType] optional parameter indicating the folder
   * type for the manager.
   */
  var DeviceMgrModel = function DeviceMgrModel(params) {
    var that = this,
      folderType = params.folderType;
    MgrModel.call(that, _.extend({
      componentSource: new ContainerComponentSource({
        container: params.component,
        filter: params.filter || makeDefaultSourceFilter(params)
      }),
      rowsChangedEventDelay: CHANGE_EVENT_THROTTLE_MS
    }, params));
    if (folderType) {
      addFolderMgrModelMixin(that, {
        folderType: folderType
      });
    }
  };
  DeviceMgrModel.prototype = Object.create(MgrModel.prototype);
  DeviceMgrModel.prototype.constructor = DeviceMgrModel;

  /**
   * Get the display name from the display name of the root component container.
   * This is used for the title of the tab in the HTML5 hx profile.
   */
  DeviceMgrModel.prototype.getNavDisplayName = function () {
    return this.getComponentSource().getContainer().getNavDisplayName();
  };

  /**
   * Make a row for the given subject with the appropriate icon for the row. Overrides
   * TableModel.makeRow().
   *
   * @override
   * @param subject The subject of the row. Should be a device or folder instance.
   * @returns {module:nmodule/webEditors/rc/wb/table/model/Row}
   */
  DeviceMgrModel.prototype.makeRow = function (subject) {
    return new Row(subject, subject.getNavIcon());
  };
  return DeviceMgrModel;
});
//# sourceMappingURL=DeviceMgrModel.js.map
