/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

define(['baja!', 'baja!schedule:DateSchedule,schedule:DayOfMonthSchedule,' + 'schedule:MonthSchedule,schedule:YearSchedule', 'lex!baja,schedule', 'Promise', 'underscore', 'nmodule/schedule/rc/model/TimeOfDay', 'nmodule/webEditors/rc/fe/baja/util/compUtils'], function (baja, types, lexs, Promise, _, TimeOfDay, compUtils) {
  'use strict';

  var MILLIS_IN_DAY = TimeOfDay.MILLIS_IN_DAY,
    bajaLex = lexs[0],
    scheduleLex = lexs[1];
  function doFormat(pattern) {
    return baja.Format.format({
      pattern: pattern
    });
  }
  function formatAndEscape(pattern) {
    return doFormat(pattern).then(baja.SlotPath.escape);
  }
  var monthFormats = ['%lexicon(schedule:month.anyMonth)%', '%lexicon(baja:january.short)%', '%lexicon(baja:february.short)%', '%lexicon(baja:march.short)%', '%lexicon(baja:april.short)%', '%lexicon(baja:may.short)%', '%lexicon(baja:june.short)%', '%lexicon(baja:july.short)%', '%lexicon(baja:august.short)%', '%lexicon(baja:september.short)%', '%lexicon(baja:october.short)%', '%lexicon(baja:november.short)%', '%lexicon(baja:december.short)%', '%lexicon(schedule:month.jan_mar_may_jul_sep_nov)%', '%lexicon(schedule:month.feb_apr_jun_aug_oct_dec)%'],
    toMonthTags = Promise.all(monthFormats.map(formatAndEscape));

  /**
   * API Status: **Private**
   *
   * Miscellaneous utility methods for schedule apps.
   *
   * @exports nmodule/schedule/rc/util/scheduleUtils
   */
  var exports = {};
  exports.setTopAndBottom = function (elem, block) {
    return elem.css({
      top: getTopPercent(block) + '%',
      bottom: getBottomPercent(block) + '%'
    });
  };

  /**
   * When creating a new `TimeSchedule` block, the `effectiveValue` is
   * calculated from the containing `WeeklySchedule`'s default output.
   *
   * @param {baja.Struct} defaultOutput the `baja:StatusValue` from the weekly
   * schedule's `defaultOutput` slot
   * @param {baja.Facets} facets the weekly schedule's `facets` slot
   * @returns {baja.Struct} a default `effectiveValue` calculated from the
   * default output and facets
   */
  exports.getNewBlockValue = function (defaultOutput, facets) {
    var type = String(defaultOutput.getType()),
      defaultValue = defaultOutput.getValue(),
      outValue;
    switch (type) {
      case 'baja:StatusBoolean':
        outValue = !defaultValue;
        break;
      case 'baja:StatusEnum':
        outValue = baja.DynamicEnum.make({
          ordinal: defaultValue.getOrdinal()
        });
        break;
      case 'baja:StatusNumeric':
        outValue = defaultValue;
        var max = facets.get('max'),
          min = facets.get('min');
        if (max !== null && outValue > max) {
          outValue = max.valueOf();
        }
        if (min !== null && outValue < min) {
          outValue = min.valueOf();
        }
        break;
      case 'baja:StatusString':
        outValue = defaultValue;
        break;
    }
    return baja.$(type, {
      status: baja.Status.ok,
      value: outValue
    });
  };
  exports.toMonthTags = function (includeGroups) {
    return toMonthTags.then(function (monthTags) {
      if (includeGroups) {
        return monthTags;
      }
      // If not includeGroups, drop the last two items.
      return monthTags.slice(0, monthTags.length - 2);
    });
  };
  exports.toWeekTags = function () {
    var anyWeek = formatAndEscape('%lexicon(schedule:week.anyWeek)%'),
      tags = [anyWeek],
      week = '%lexicon(schedule:week.week)%',
      calendarWeek = '%lexicon(schedule:week.calendarWeek)%',
      x;
    for (x = 1; x < 6; x++) {
      tags.push(formatAndEscape(week + ' ' + x));
    }
    tags.push(formatAndEscape('%lexicon(schedule:week.last7days)%'));
    for (x = 1; x < 7; x++) {
      tags.push(formatAndEscape(calendarWeek + ' ' + x));
    }
    return Promise.all(tags);
  };

  /**
   * Get the first day of the week.
   */
  exports.getFirstDayOfWeek = function () {
    return bajaLex.get('weekday.firstDayOfWeek');
  };
  var weekdays = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];

  /**
   * Get a list of weekdays, starting with the specified day.
   *
   * @returns {Array.<String>} array of seven lower case days ordered by the standard
   *  week, starting with the specified day.  If the specified day is not a valid weekday,
   *  the week is returned starting with sunday.
   */
  exports.getWeekdaysForStartingDay = function () {
    var start = weekdays.indexOf(exports.getFirstDayOfWeek());
    if (start < 1) {
      return weekdays;
    }
    return weekdays.slice(start).concat(weekdays.slice(0, start));
  };
  var dayTagsByStartDay = {};

  /**
   * Get the day tags, arrange appropriately according to weekday.firstDayOfWeek that contain tag and ordinal information.
   *
   * @returns {Promise.<Array>} resolves to an array of Objects, each with a {String} "tag" and a {Number} "ordinal".
   */
  exports.toDayTags = function () {
    var startDay = exports.getFirstDayOfWeek();
    if (startDay in dayTagsByStartDay) {
      return dayTagsByStartDay[startDay];
    }
    var dayTags = [{
      tag: baja.SlotPath.escape(scheduleLex.get('weekday.anyWeekday')),
      ordinal: -1
    }].concat(exports.getWeekdaysForStartingDay(startDay).map(function (day) {
      return {
        tag: baja.SlotPath.escape(bajaLex.get(day)),
        ordinal: weekdays.indexOf(day)
      };
    }));
    dayTags = Promise.all(dayTags);
    dayTagsByStartDay[startDay] = dayTags;
    return dayTags;
  };

  /**
   * Get the day of month tags
   *
   * @param includeGroups true if should include last_day, last7days, oddDays, and evenDays groups
   * @returns {Promise.<Array>} resolves to an array of escaped strings
   */
  exports.toDayOfMonthTags = function (includeGroups) {
    // any day, 1 - 31, last day, last 7 days.
    var tags = [scheduleLex.get('dayofmonth.anyDay')].concat(_.range(1, 32).map(String));
    if (includeGroups) {
      tags = tags.concat([scheduleLex.get('dayofmonth.last_day'), scheduleLex.get('dayofmonth.last7days'), scheduleLex.get('dayofmonth.oddDays'), scheduleLex.get('dayofmonth.evenDays')]);
    }
    return Promise.all(tags.map(baja.SlotPath.escape));
  };
  function timeToPercent(time) {
    return time.getTimeOfDayMillis() / MILLIS_IN_DAY * 100;
  }
  function getTopPercent(block) {
    return timeToPercent(block.getStart());
  }
  function getBottomPercent(block) {
    return 100 - (timeToPercent(block.getFinish()) || 100);
  }
  function getYear(value, slot) {
    return value[slot];
  }
  function getEnum(value, slot) {
    return value[slot].getOrdinals()[0];
  }

  /**
   * Verify a start / end date range where start must be <= end, but taking
   * into consideration -1 means "any".
   *
   * Start / End are arrays of
   * [ {EnumSet} weekday, {EnumSet} day, {EnumSet} month, {Integer} year]
   *
   * These values are the values resolved by a {DateScheduleEditor#read}.
   *
   * @param {Array} start
   * @param {Array} end
   * @returns {boolean}
   */
  exports.checkValidDateRange = function (start, end) {
    // Iterate through year, month, day values
    var getter = getYear;

    //weekday entry is optional
    start = start.filter(function (o) {
      return baja.hasType(o);
    });
    start = start.slice(start.length - 3);
    end = end.filter(function (o) {
      return baja.hasType(o);
    });
    end = end.slice(end.length - 3);
    for (var slot = 2; slot >= 0; slot--) {
      var startValue = getter(start, slot),
        endValue = getter(end, slot);
      // Once something is "any", or end > start, everything else is ok
      if (endValue < 0 || startValue < 0 || endValue > startValue) {
        return true;
      }
      // Error if start is after end
      if (startValue > endValue) {
        return false;
      }
      getter = getEnum;
    }
    return true;
  };

  /**
   * Convert a date to a `schedule:DateSchedule`.
   * @param {Date} date
   * @returns {baja.Component}
   */
  exports.dateToDateSchedule = function (date) {
    return baja.$('schedule:DateSchedule', {
      yearSchedule: baja.$('schedule:YearSchedule', {
        year: date.getFullYear()
      }),
      monthSchedule: baja.$('schedule:MonthSchedule', {
        set: baja.EnumSet.make({
          ordinals: [date.getMonth()]
        })
      }),
      daySchedule: baja.$('schedule:DayOfMonthSchedule', {
        set: baja.EnumSet.make({
          ordinals: [date.getDate()]
        })
      })
    });
  };

  /**
   * @param {string} weekday lowercase weekday name, e.g. 'monday'
   * @returns {number} the index of that weekday as expected by Niagara, where
   * sunday=0 and saturday=6
   */
  exports.getWeekdayIndex = function (weekday) {
    return weekdays.indexOf(weekday);
  };

  /**
   * Configure a Edit Command to make it readonly or switch it back.
   * @param {module:bajaux/commands/Command} cmd
   * @param {boolean} readonly
   */
  exports.configureEditCommandReadonly = function (cmd, readonly) {
    if (readonly) {
      cmd.setDisplayNameFormat(scheduleLex.get('commands.viewEvent.displayName'));
      cmd.setDescriptionFormat(scheduleLex.get('commands.viewEvent.description'));
      cmd.setIcon(scheduleLex.get('commands.viewEvent.icon'));
    } else {
      cmd.setDisplayNameFormat(scheduleLex.get('commands.editEvent.displayName'));
      cmd.setDescriptionFormat(scheduleLex.get('commands.editEvent.description'));
      cmd.setIcon(scheduleLex.get('commands.editEvent.icon'));
    }
  };

  /**
   * Specify an ORD base for looking up schedule references. This is necessary
   * because the schedule will not be mounted when it is sent to the station.
   * @param {baja.Component} sched
   * @param {baja.Ord} [refBase]
   * @see BScheduleReference#getSchedule()
   * @returns {baja.Component}
   */
  exports.appendRefBase = function (sched, refBase) {
    var copy = sched.newCopy(true);
    if (refBase) {
      copy.add({
        slot: 'refBase',
        value: refBase,
        flags: baja.Flags.TRANSIENT | baja.Flags.HIDDEN
      });
    }
    return copy;
  };
  return exports;
});
