/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/model/ScheduleBlock
 */
define(['underscore', 'nmodule/schedule/rc/model/TimeOfDay'], function (_, TimeOfDay) {
  'use strict';

  var MILLIS_IN_DAY = TimeOfDay.MILLIS_IN_DAY;

  /**
   * Data model representing a single block within a day's schedule.
   *
   * @class
   * @alias module:nmodule/schedule/rc/model/ScheduleBlock
   */
  var ScheduleBlock = function ScheduleBlock(params) {
    if (params instanceof ScheduleBlock) {
      return params;
    }
    if (!params || !params.start) {
      throw new Error('start required');
    }
    if (!params.finish) {
      throw new Error('finish required');
    }
    this.$start = new TimeOfDay(params.start);
    this.$finish = new TimeOfDay(params.finish);
    this.$value = params.value;
    this.$timeRangeDisplay = params.timeRangeDisplay;
    this.$valueDisplay = params.valueDisplay;
  };

  /**
   * @returns {module:nmodule/schedule/rc/model/TimeOfDay}
   */
  ScheduleBlock.prototype.getStart = function () {
    return this.$start;
  };

  /**
   * @returns {module:nmodule/schedule/rc/model/TimeOfDay}
   */
  ScheduleBlock.prototype.getFinish = function () {
    return this.$finish;
  };

  /**
   * @returns {*}
   */
  ScheduleBlock.prototype.getValue = function () {
    return this.$value;
  };

  /**
   * @returns {string}
   */
  ScheduleBlock.prototype.getTimeRangeDisplay = function () {
    return this.$timeRangeDisplay || String(this.$start) + ' - ' + String(this.$finish);
  };

  /**
   * @returns {string}
   */
  ScheduleBlock.prototype.getValueDisplay = function () {
    var valueDisplay = this.$valueDisplay;
    if (typeof valueDisplay === 'undefined') {
      valueDisplay = this.$value;
    }
    return String(valueDisplay);
  };

  /**
   * Return true if this block overlaps the time range specified by the given
   * start and finish.
   * @param {module:nmodule/schedule/rc/model/TimeOfDay} start
   * @param {module:nmodule/schedule/rc/model/TimeOfDay} finish
   * @returns {boolean}
   */
  ScheduleBlock.prototype.overlaps = function (start, finish) {
    var myStart = ms(this.$start),
      myFinish = ms(this.$finish) || MILLIS_IN_DAY,
      hisStart = ms(start),
      hisFinish = ms(finish) || MILLIS_IN_DAY;
    if (myFinish < myStart) {
      myFinish += MILLIS_IN_DAY;
    }
    if (hisFinish < hisStart) {
      hisFinish += MILLIS_IN_DAY;
    }
    if (myFinish > hisStart && hisFinish > myStart) {
      return true;
    }
    hisStart += MILLIS_IN_DAY;
    hisFinish += MILLIS_IN_DAY;
    return myFinish > hisStart && hisFinish > myStart;
  };
  ScheduleBlock.prototype.toString = function () {
    return String(this.$start) + ' - ' + String(this.$finish);
  };
  ScheduleBlock.prototype.toJson = function () {
    return {
      start: this.$start.toJson(),
      finish: this.$finish.toJson(),
      value: this.$value
    };
  };
  function ms(time) {
    return time.getTimeOfDayMillis();
  }
  return ScheduleBlock;
});
