/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/baja/fe/DateScheduleEditor
 */
define(['bajaux/util/CommandButton', 'Promise', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder', 'nmodule/schedule/rc/baja/fe/DayOfMonthScheduleEnumSetEditor', 'nmodule/schedule/rc/baja/fe/MonthScheduleEnumSetEditor', 'nmodule/schedule/rc/baja/fe/WeekdayScheduleEnumSetEditor', 'nmodule/schedule/rc/baja/fe/YearScheduleEditor', 'nmodule/schedule/rc/util/scheduleUtils', 'nmodule/webEditors/rc/wb/commands/DatePickerPopupCommand'], function (CommandButton, Promise, CompositeEditor, CompositeBuilder, DayOfMonthScheduleEnumSetEditor, MonthScheduleEnumSetEditor, WeekdayScheduleEnumSetEditor, YearScheduleEditor, scheduleUtils, DatePickerPopupCommand) {
  'use strict';

  var dateToDateSchedule = scheduleUtils.dateToDateSchedule;
  var tpl = function tpl(ed) {
    return "\n    <div class=\"composite-schedule-editor simple-schedule-editor\">\n      <div class=\"weekdaySchedule-schedule-editor\"></div>\n      <div class=\"daySchedule-schedule-editor\"></div>\n      <div class=\"monthSchedule-schedule-editor\"></div>\n      <div class=\"yearSchedule-schedule-editor\"></div>\n      <button type=\"button\" class=\"datePicker-schedule-editor ux-btn-tb\"></button>\n    </div>";
  };

  /**
   * @inner
   * @class
   * @param {module:nmodule/schedule/rc/baja/fe/DateScheduleEditor} ed
   * @extends {module:nmodule/webEditors/rc/wb/commands/DatePickerPopupCommand}
   */
  var SetDateScheduleCommand = function SetDateScheduleCommand(ed) {
    DatePickerPopupCommand.call(this);
    var func = this.getFunction();
    this.setFunction(function () {
      return func().then(function (date) {
        return date && ed.$applyDate(date);
      });
    });
  };
  SetDateScheduleCommand.prototype = Object.create(DatePickerPopupCommand.prototype);
  SetDateScheduleCommand.prototype.constructor = SetDateScheduleCommand;

  /**
   * Editor for a `schedule:DateSchedule`.
   *
   * @class
   * @alias module:nmodule/schedule/rc/baja/fe/DateScheduleEditor
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   */
  var DateScheduleEditor = function DateScheduleEditor() {
    CompositeEditor.apply(this, arguments);
    this.getCommandGroup().add(new SetDateScheduleCommand(this));
  };
  DateScheduleEditor.prototype = Object.create(CompositeEditor.prototype);
  DateScheduleEditor.prototype.constructor = DateScheduleEditor;

  /**
   * @private
   * @returns {module:bajaux/commands/Command} a Command that will pop up a
   * date picker and update the DateSchedule fields accordingly
   */
  DateScheduleEditor.prototype.$getDatePickerCommand = function () {
    return this.getCommandGroup().get(0);
  };

  /**
   * Update the child editors to match the fields on the input date, and mark
   * them all modified.
   * @private
   * @param {Date} date
   */
  DateScheduleEditor.prototype.$applyDate = function (date) {
    var _this = this;
    var dateSchedule = dateToDateSchedule(date),
      builder = this.getBuilder(),
      editableKeys = ['yearSchedule', 'monthSchedule', 'daySchedule', 'weekdaySchedule'];
    return this.value().get('yearSchedule').set({
      slot: 'alwaysEffective',
      value: false
    }).then(function () {
      return Promise.all(editableKeys.map(function (key) {
        var kid = builder.getEditorFor(key);
        return kid && kid.load(getValueForKey(dateSchedule, key)).then(function () {
          //we have to set modified on kids because only modified kids save,
          //and we have to do it "silently" because applyDate should result
          //in only one modify event
          kid.$modified = true;
        });
      }));
    }).then(function () {
      return _this.setModified(true);
    });
  };
  DateScheduleEditor.prototype.doInitialize = function (dom) {
    dom.html(tpl(this));
    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };
  DateScheduleEditor.prototype.makeBuilder = function () {
    var that = this,
      includeWeekdays = that.$isIncludeWeekdays(),
      builder = new CompositeBuilder();
    builder.getKeys = function () {
      var keys = [];
      if (includeWeekdays) {
        keys.push('weekdaySchedule');
      }
      return keys.concat('daySchedule', 'monthSchedule', 'yearSchedule', 'datePicker');
    };
    builder.getDomFor = function (key) {
      return that.jq().find('.' + key + '-schedule-editor');
    };
    builder.getValueFor = function (key) {
      switch (key) {
        case 'datePicker':
          return that.$getDatePickerCommand();
        default:
          return getValueForKey(this.getDataSource(), key);
      }
    };
    var editorTypeForKey = {
      'weekdaySchedule': WeekdayScheduleEnumSetEditor,
      'daySchedule': DayOfMonthScheduleEnumSetEditor,
      'monthSchedule': MonthScheduleEnumSetEditor
    };
    builder.getConfigFor = function (key) {
      var dateSchedule = this.getDataSource(),
        formFactor = that.getFormFactor();
      if (key === 'yearSchedule') {
        return {
          complex: dateSchedule.get(key),
          slot: 'year',
          type: YearScheduleEditor,
          formFactor: formFactor
        };
      } else if (key === 'datePicker') {
        return {
          type: CommandButton,
          formFactor: 'mini'
        };
      }
      return {
        complex: dateSchedule.get(key),
        slot: 'set',
        type: editorTypeForKey[key],
        formFactor: formFactor,
        properties: {
          singleSelection: true
        }
      };
    };
    return builder;
  };
  DateScheduleEditor.prototype.$isIncludeWeekdays = function () {
    return !!this.properties().getValue('includeWeekdays');
  };
  function getValueForKey(dateSchedule, key) {
    return dateSchedule.get(key).get(key === 'yearSchedule' ? 'year' : 'set');
  }
  DateScheduleEditor.prototype.doEnabled = function (enabled) {
    this.$getDatePickerCommand().setEnabled(!this.isReadonly() && enabled);
    return CompositeEditor.prototype.doEnabled.apply(this, arguments);
  };
  DateScheduleEditor.prototype.doReadonly = function (readonly) {
    this.$getDatePickerCommand().setEnabled(!readonly && this.isEnabled());
    return CompositeEditor.prototype.doReadonly.apply(this, arguments);
  };
  return DateScheduleEditor;
});
