/**
 * @copyright 2023 Tridium, Inc. All Rights Reserved.
 */

define([], function () {
  'use strict';

  /**
   * API Status: **Private**
   * @exports nmodule/weather/rc/util/airQualityUtil
   */
  var exports = {};

  /**
   * @param {number} concentration ozone concentration
   * @returns {number} AQI
   */
  exports.ozoneConcentrationToAqi = function (concentration) {
    return concentrationToAqi(O3, concentration);
  };

  /**
   * @param {number} concentration particulate matter concentration
   * @returns {number} AQI
   */
  exports.particulateConcentrationToAqi = function (concentration) {
    return concentrationToAqi(PM25, concentration);
  };

  /**
   * @param {baja.Component} conditions a `weather:CurrentConditions`, `weather:Forecast`, or
   * `weather:WeatherReport`
   * @returns {number} the greater of the ozone and particulate AQI, if they are not set to null.
   * Returns -1 if no AQI could be determined.
   */
  exports.getAqiForConditions = function (conditions) {
    if (conditions.getType().is('weather:WeatherReport')) {
      conditions = conditions.get('current');
    }
    var ozone = conditions.get('ozone');
    var particulateMatter = conditions.get('particulateMatter');
    var ozoneAqi = ozone.get('status').isNull() ? -1 : exports.ozoneConcentrationToAqi(ozone.get('value'));
    var particulateAqi = particulateMatter.get('status').isNull() ? -1 : exports.particulateConcentrationToAqi(particulateMatter.get('value'));
    return Math.max(ozoneAqi, particulateAqi);
  };

  /**
   * @since Niagara 4.14u3
   * @private
   * @param {JQuery} dom
   * @param {String} selector
   * @returns {JQuery}
   */
  exports.$getDomFromHxPx = function (dom, selector) {
    return dom.closest('#hxPx ' + selector);
  };
  function concentrationToAqi(conversions, concentration) {
    for (var i = 0, len = conversions.length; i < len; ++i) {
      var _conversions$i = conversions[i],
        cLow = _conversions$i.cLow,
        cHigh = _conversions$i.cHigh,
        iLow = _conversions$i.iLow,
        iHigh = _conversions$i.iHigh;
      if (concentration >= cLow && concentration <= cHigh) {
        return Math.ceil((iHigh - iLow) * ((concentration - cLow) / (cHigh - cLow)) + iLow);
      }
    }
    throw new Error('Invalid concentration value: ' + concentration);
  }

  /*
   ozone to AQI, as per (updated) EPA formula:
   https://www.airnow.gov/sites/default/files/2020-05/aqi-technical-assistance-document-sept2018.pdf
   */
  var O3 = [{
    cLow: 0,
    cHigh: 0.054,
    iLow: 0,
    iHigh: 50
  }, {
    cLow: 0.055,
    cHigh: 0.070,
    iLow: 51,
    iHigh: 100
  }, {
    cLow: 0.071,
    cHigh: 0.085,
    iLow: 101,
    iHigh: 150
  }, {
    cLow: 0.086,
    cHigh: 0.105,
    iLow: 151,
    iHigh: 200
  }, {
    cLow: 0.106,
    cHigh: 0.200,
    iLow: 201,
    iHigh: 300
  }, {
    cLow: 0.405,
    cHigh: 0.504,
    iLow: 301,
    iHigh: 400
  }, {
    cLow: 0.505,
    cHigh: 0.604,
    iLow: 401,
    iHigh: 500
  }];

  /*
    particulate to AQI, as per (updated) EPA formula:
    https://www.airnow.gov/sites/default/files/2020-05/aqi-technical-assistance-document-sept2018.pdf
   */
  var PM25 = [{
    cLow: 0,
    cHigh: 12.0,
    iLow: 0,
    iHigh: 50
  }, {
    cLow: 12.1,
    cHigh: 35.4,
    iLow: 51,
    iHigh: 100
  }, {
    cLow: 35.5,
    cHigh: 55.4,
    iLow: 101,
    iHigh: 150
  }, {
    cLow: 55.5,
    cHigh: 150.4,
    iLow: 151,
    iHigh: 200
  }, {
    cLow: 150.5,
    cHigh: 250.4,
    iLow: 201,
    iHigh: 300
  }, {
    cLow: 250.5,
    cHigh: 350.4,
    iLow: 301,
    iHigh: 400
  }, {
    cLow: 350.5,
    cHigh: 500.4,
    iLow: 401,
    iHigh: 500
  }];
  return exports;
});
