/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module transfer
 * @private
 */
define(['bajaScript/comm'], function (baja) {
  'use strict';

  /**
   * @namespace baja.transfer
   */
  var transfer = {
    /**
     * Perform a station-side copy using the Niagara Transfer API.
     * 
     * @param {Object} config
     * @param {Array.<String|baja.Ord>} config.sourceOrds ORDs of the source
     * nodes to be copied
     * @param {Array.<String|baja.Ord>} [config.sourceBases] Base ORDs used to
     * resolve the source ORDs; any missing base ORDs will use local:
     * @param {baja.NavNode} config.target a mounted target nav node to receive
     * the copied nodes
     * @param {Array.<String>} [config.names] desired names for the copied
     * nodes. If omitted, auto-generated names will be used.
     * @param {boolean} [config.keepLinks=false] set to true to have links
     * copied over from the source nodes
     * @param {boolean} [config.keepRelations=false] set to true to have
     * relations copied over from the source nodes
     * @param {string} [config.origin] set to `compute` to compute the origin
     * based off the position of the source nodes (as Paste Special does), or
     * set it to a `WsAnnotation` encoding such as `2,2,8` to specify exactly
     * what point on the wiresheet the copied nodes should go. If omitted, no
     * annotation will be added to the copied nodes.
     * @param {number} [config.numCopies=1] set how many times the nodes should
     * be duplicated
     * @returns {Promise.<{ undoKey: string, insertNames: string[] }>} promise to
     * be resolved after the copy operation completes; if the target is a
     * Component, its component space will also be synced. Promise will be
     * resolved with an array of the node names of the newly inserted nodes and a
     * key for a future call to `undo`.
     */
    copy: function copy(config) {
      var cb = new baja.comm.Callback();
      return baja.comm.copy(config, cb);
    },
    /**
     * Perform a station-side move using the Niagara Transfer API.
     * auto-generated names will be used if there is a conflict with existing names.
     *
     * @param {Object} config
     * @param {Array.<String|baja.Ord>} config.sourceOrds ORDs of the source
     * nodes to be moved
     * @param {Array.<String|baja.Ord>} [config.sourceBases] Base ORDs used to
     * resolve the source ORDs; any missing base ORDs will use local:
     * @param {baja.NavNode} config.target a mounted target nav node to receive
     * the copied nodes
     * @returns {Promise.<{ undoKey: string, insertNames: string[] }>} promise to
     * be resolved after the copy operation completes; if the target is a
     * Component, its component space will also be synced. Promise will be
     * resolved with an array of the node names of the newly inserted nodes and a
     * key for a future call to `undo`.
     * @since Niagara 4.11
     */
    move: function move(config) {
      var cb = new baja.comm.Callback();
      return baja.comm.move(config, cb);
    },
    /**
     * Perform a station-side delete using the Niagara Transfer API.
     *
     * @param {Object} config
     * @param {Array.<String|baja.Ord>} config.deleteOrds ORDs of the nodes to be
     * deleted
     * @param {Array.<baja.NavNode>} [config.deleteBases] Base objects used to
     * resolve the delete ORDs; any missing bases will use localhost
     * @returns {Promise.<string|undefined>} promise to be resolved after the
     * delete operation completes. Promise will be resolved with a string key
     * that may be used for a future undelete operation, or undefined if
     * undelete is not possible.
     * @since Niagara 4.11
     */
    "delete": function _delete(config) {
      var cb = new baja.comm.Callback();
      return baja.comm["delete"](config, cb);
    },
    /**
     * Perform a station-side undo using the Niagara Transfer API.
     *
     * @param {Object} config
     * @param {string} config.undoKey an undo key from a previous transfer
     * @returns {Promise} promise to be resolved after the
     * undo operation completes.
     * @since Niagara 4.11
     */
    undo: function undo(config) {
      var cb = new baja.comm.Callback();
      return baja.comm.undo(config, cb);
    }
  };
  baja.transfer = transfer;
  return baja;
});
