/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * Entry point for using BajaScript inside node. Require this with
 * with the CommonJS `require`. It will configure RequireJS for use
 * with BajaScript and add the necessary node-specific functionality for
 * BajaScript itself.
 *
 * This is a Node module, so require it the usual CommonJS way. After
 * loading it, start it and use it using RequireJS.
 *
 * Currently only works when the station is configured with Cookie
 * authentication.
 *
 * @module baja/env/node
 * @private
 *
 * @example

var getBaja = require('bajascript'),
    requirejs = require('requirejs'),
    connectParams = {
      hostname: 'localhost',
      port: 8088,
      method: 'post',
      path: '/box/',
      auth: 'admin:asdf1234'
    };

getBaja({ connectParams: connectParams }, function (err, baja) {
  if (err) throw err;

  //RequireJS is now configured for BajaScript, but it is not started.
  //start it by requiring it with the baja plugin.
  requirejs(['baja!',
             'baja!myModule:TypeA,myModule:TypeB], function (baja, types) {

    //BajaScript is now started and ready to use.

  });
});

// or, if you don't want to use RequireJS, call it this way. it will still
// use RequireJS internally.

var bajascript = require('bajascript');
bajascript.start(connectParams, function (err, baja) {
  // BajaScript is now started and ready to use.
  // note that if you want to load types you must use baja.importTypes().
});

*/

/* eslint-env node */

//TODO: support Cookie Digest. needs to do the SJCL PBKDF2 processing on server

'use strict';

var nodeConfig = require('./require-node-config'),
  requirejs = require('requirejs');

/**
 * Configures BajaScript for RequireJS, but does not start it. Start it with
 * the <code>baja</code> RequireJS plugin.
 * 
 * @param {Object} config
 * @param {Object} config.connectParams parameters BajaScript will use to
 * connect to the station
 * @param {String} config.connectParams.hostname
 * @param {Number} config.connectParams.port
 * @param {String} [config.connectParams.method="post"]
 * @param {String} [config.connectParams.path="/box/"]
 * @param {String} config.connectParams.auth basic auth "username:password"
 * @param {Function} callback
 */
function getBaja(config, callback) {
  var requirejs = nodeConfig(config);
  requirejs(['bajaScript/env/define-node'], function (baja) {
    callback(null, baja);
  }, function (err) {
    callback(err);
  });
}

/**
 * Requires and starts BajaScript. Any types you need to load must be done
 * afterwards with <code>baja.importTypes</code>. Still uses RequireJS
 * internally.
 */
getBaja.start = function (config, callback) {
  getBaja(config, function (err, baja) {
    if (err) {
      return callback(err);
    }
    requirejs(['baja!'], function (baja) {
      callback(null, baja);
    });
  });
};
module.exports = getBaja;
