/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/* eslint-env browser */

define([], function () {
  'use strict';

  var ENCODER = new TextEncoder("UTF-8");
  var DECODER = new TextDecoder("UTF-8");

  /**
   * API Status: **Private**
   * @exports baja/env/mux/muxUtils
   */
  var exports = {};

  /**
   * @param {string|Uint8Array} str
   * @returns {Uint8Array}
   */
  exports.toBytes = function (str) {
    if (typeof str === 'string') {
      return ENCODER.encode(str);
    }
    return str;
  };

  /**
   * @param {Uint8Array|string} arr
   * @returns {string}
   */
  exports.fromBytes = function (arr) {
    if (typeof arr === 'string') {
      return arr;
    }
    return DECODER.decode(arr);
  };

  /**
   * @param {Uint8Array|string} arr
   * @returns {module:baja/env/mux/BoxEnvelope~BoxFragment}
   */
  exports.decodeFragment = function (arr) {
    var isString = typeof arr === 'string';
    var delimiter = isString ? ';' : ';'.charCodeAt(0);
    var i = 0;
    function token() {
      var index = i;
      while (arr[index] !== delimiter) {
        index++;
      }
      var str = toToken(arr, i, index);
      i = index + 1;
      return str;
    }
    var fragmentMarker = token();
    if (fragmentMarker !== 'F') {
      throw new Error('unrecognized fragment marker "' + fragmentMarker + '"');
    }
    var boxVersion = token();
    if (boxVersion !== '2.3') {
      // shrug
    }
    var sessionId = token();
    var envelopeId = +token();
    var fragmentCount = +token();
    var fragmentIndex = +token();
    var unsolicited = token() === 'u';
    var payload = exports.toBytes(toToken(arr, i, arr.length));
    return {
      sessionId: sessionId,
      envelopeId: envelopeId,
      fragmentCount: fragmentCount,
      fragmentIndex: fragmentIndex,
      unsolicited: unsolicited,
      payload: payload
    };
  };

  /**
   * @param {Uint8Array|string} arr
   * @param {number} start
   * @param {number} end
   * @returns {string}
   */
  function toToken(arr, start, end) {
    if (typeof arr === 'string') {
      return arr.substring(start, end);
    } else {
      return exports.fromBytes(arr.slice(start, end));
    }
  }
  return exports;

  /**
   * Configuration settings required for setting up BOX muxing.
   * @private
   * @typedef baja/env/mux/muxUtils~MuxSettings
   * @property {number} minDelay - "debounce" time - minimum amount of time to delay a message from
   * sending so that more messages can be coalesced into a single network call
   * @property {number} maxDelay - the maximum amount of time to hold onto pending messages before'
   * sending. This would prevent a slow, steady trickle of BoxFrames from just queueing forever.
   * @property {number} maxEnvelopeSize - maximum amount of data to have pending at any one time
   * @property {number} maxMessageSize - maximum amount of data to send in a single message (e.g.
   * WebSocket size limit)
   * @property {function} sendToServer a function to be called when data is ready to go out over the
   * wire. It receives a Uint8Array and should send it up to the BOX service.
   * @property {function} onIncomingFrame a function that receives a BOX frame from the station
   * (either unsolicited, or as a response to an outgoing frame)
   * @property {function} setFragmentHandler BOX fragments are received at the Connection level and
   * passed back into the muxing system. This would be an event emitter if we had tinyevents, but
   * instead muxing will explicitly set the handler for when a new fragment is received from the
   * station.
   *
   */
});
