/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/* eslint-env browser */
/**
 * API Status: **Private**
 * @module baja/env/mux/BoxEnvelope
 */
define(['bajaScript/env/mux/muxUtils'], function (muxUtils) {
  'use strict';

  var toBytes = muxUtils.toBytes;

  /**
   * A `BoxEnvelope` contains a payload consisting of zero or more
   * string fragments. These fragments can be sent and received individually.
   * In this way, large payloads which may consist of many or large BoxFrames
   * can be sent without exceeding the maximum WebSocket message size.
   *
   * Think of this class as a transport-layer mechanism. It doesn't make any
   * decisions regarding which BoxFrames batch together into individual network
   * calls. It only handles the work of transmitting large payloads over a
   * protocol (i.e. WebSocket) that restricts the size of individual messages.
   *
   * @class
   * @alias module:baja/env/mux/BoxEnvelope
   * @param {object} params
   * @param {string} params.sessionId - server session ID
   * @param {number} params.envelopeId - envelope ID
   * @param {number} params.fragmentCount - number of fragments in this envelope (for an incoming
   * envelope, this is the number of _expected_ fragments required to complete the envelope; for an
   * outgoing envelope, this matches the number of fragments as they are appended)
   * @param {number} params.birthDate - envelope birthdate as a long timestamp
   * @param {number} params.maxMessageSize - the maximum allowable size of a
   * fragment (configurable by system property)
   * @param {number} params.maxEnvelopeSize - the maximum total allowable size
   * of this envelope (configurable by system property)
   * @param {boolean} params.unsolicited - true if this is an unsolicited
   * envelope sent down to us from the server
   * @param {boolean} [params.outgoing] - true if this envelope is going from client to server
   * @param {Array.<Uint8Array>} [params.fragments] if the envelope is fully formed
   * at the time of creation (i.e. on the client side), populate it with its
   * fragment payloads
   *
   */
  var BoxEnvelope = function BoxEnvelope(params) {
    params = params || {};
    var sessionId = this.$sessionId = params.sessionId;
    var envelopeId = this.$envelopeId = params.envelopeId;
    this.$birthDate = params.birthDate || +new Date();
    this.$maxMessageSize = params.maxMessageSize || Number.MAX_SAFE_INTEGER;
    this.$maxEnvelopeSize = params.maxEnvelopeSize || Number.MAX_SAFE_INTEGER;
    var unsolicited = this.$unsolicited = !!params.unsolicited;
    var outgoing = this.$outgoing = !!params.outgoing;
    if (outgoing) {
      this.$buffer = new Uint8Array(1024);
      this.$len = 0;
    } else {
      this.$fragments = params.fragments || [];
    }
    this.$fragmentCount = params.fragmentCount || 0;
    this.$prefixOverhead = toBytes(['F', '2.3', sessionId, envelopeId, 99, 99, unsolicited ? 'u' : 's', ''].join(';')).length;
  };

  /**
   * Creates a new BoxEnvelope from a given payload. The payload will be split
   * into fragments, each one guaranteed smaller than the max message size.
   *
   * @param {object} params
   * @param {number} params.envelopeId
   * @param {string} params.sessionId server session ID
   * @param {string} [params.destination] BOX destination
   * @param {number} params.maxMessageSize maximum message size that can be
   * transmitted to the server. This is typically the maximum WebSocket message
   * size, determined by the `box.ws.maxTextMessageSize` system property.
   * @param {number} params.maxEnvelopeSize - the maximum total allowable size
   * of this envelope
   * @param {Uint8Array} [params.payload]
   * @returns {module:baja/env/mux/BoxEnvelope}
   */
  BoxEnvelope.makeOutgoing = function (params) {
    var maxMessageSize = params.maxMessageSize;
    var payload = params.payload;
    var envelope = new BoxEnvelope({
      envelopeId: params.envelopeId,
      sessionId: params.sessionId,
      destination: params.destination,
      maxMessageSize: maxMessageSize,
      maxEnvelopeSize: params.maxEnvelopeSize,
      outgoing: true
    });
    envelope.append(payload);
    return envelope;
  };

  /**
   * @returns {number} envelope ID
   */
  BoxEnvelope.prototype.getEnvelopeId = function () {
    return this.$envelopeId;
  };

  /**
   * @returns {number} the number of fragments this envelope should contain
   */
  BoxEnvelope.prototype.getFragmentCount = function () {
    return this.isOutgoing() ? this.getFragments().length : this.$fragmentCount;
  };

  /**
   * @returns {number} envelope birth date as a long timestamp
   */
  BoxEnvelope.prototype.getBirthDate = function () {
    return this.$birthDate;
  };

  /**
   * @returns {boolean} true if this envelope was received from the server
   * unsolicited
   */
  BoxEnvelope.prototype.isUnsolicited = function () {
    return this.$unsolicited;
  };

  /**
   * @returns {boolean} true if this envelope collecting fragments to go from client to server;
   * false if collecting fragments sent to client from the server
   */
  BoxEnvelope.prototype.isOutgoing = function () {
    return this.$outgoing;
  };

  /**
   * @returns {Array.<Uint8Array>} the fragments
   * currently contained in this envelope (may be sparse)
   */
  BoxEnvelope.prototype.getFragments = function () {
    var fragments;
    if (this.$outgoing) {
      var maxFragmentLength = this.getMaxFragmentLength();
      fragments = [];
      var data = this.getPayload();
      while (data.length) {
        fragments.push(data.slice(0, maxFragmentLength));
        data = data.slice(maxFragmentLength);
      }
      this.$fragmentCount = fragments.length;
    } else {
      fragments = this.$fragments.slice();
    }
    return fragments;
  };

  /**
   * @returns {number} the maximum amount of actual BOX data the client can fit into one fragment
   */
  BoxEnvelope.prototype.getMaxFragmentLength = function () {
    return this.$maxMessageSize - this.$prefixOverhead;
  };

  /**
   * Adds a newly-received fragment to an envelope in progress.
   * @param {Uint8Array} fragment
   * @param {number} index fragment index
   * @throws {Error} if the fragment has an invalid fragment index or this is an outgoing envelope
   */
  BoxEnvelope.prototype.receiveFragment = function (fragment, index) {
    if (this.isOutgoing()) {
      throw new Error('outgoing envelope cannot receive fragment');
    }
    var fragments = this.$fragments;
    if (index < 0 || index >= this.$fragmentCount) {
      throw new Error('invalid fragment index');
    }
    fragments[index] = fragment;
  };

  /**
   * Appends new data to a pending outgoing BoxEnvelope and updates
   * fragment count.
   *
   * @param {string|Uint8Array} data
   */
  BoxEnvelope.prototype.append = function (data) {
    if (!data) {
      return;
    }
    if (typeof data === 'string') {
      data = toBytes(data);
    }
    var buffer = this.$buffer;
    while (this.$len + data.length > buffer.length) {
      var newBuffer = new Uint8Array(buffer.length * 2);
      newBuffer.set(buffer, 0);
      buffer = this.$buffer = newBuffer;
    }
    buffer.set(data, this.$len);
    this.$len += data.length;
  };

  /**
   * @param {number} payloadSize the size of the payload you wish to append to
   * the envelope
   * @returns {boolean} true if the payload can fit
   * @throws {Error} if this is an incoming envelope as the contents of the envelope are set by the
   * server
   */
  BoxEnvelope.prototype.willFit = function (payloadSize) {
    if (!this.isOutgoing()) {
      throw new Error('cannot call willFit on incoming envelope');
    }
    return this.$len + payloadSize <= this.maxPayloadSize();
  };

  /**
   * @returns {boolean} true if the number of fragments currently in this
   * envelope is equal to the expected fragment count
   */
  BoxEnvelope.prototype.isComplete = function () {
    var count = this.$fragmentCount;
    var fragments = this.$fragments;
    for (var i = 0; i < count; i++) {
      if (!fragments[i]) {
        return false;
      }
    }
    return true;
  };

  /**
   * @returns {boolean} true if there is nothing in the envelope
   */
  BoxEnvelope.prototype.isEmpty = function () {
    return this.isOutgoing() ? !this.$len : !this.$fragments.length;
  };
  BoxEnvelope.prototype.maxPayloadSize = function () {
    var maxFragments = Math.ceil(this.$maxEnvelopeSize / this.$maxMessageSize);
    return this.$maxEnvelopeSize - maxFragments * this.$prefixOverhead;
  };

  /**
   * @returns {Uint8Array} the complete payload reconstituted from all fragments
   * @throws {Error} if the envelope is not yet complete
   */
  BoxEnvelope.prototype.getPayload = function () {
    if (!this.isOutgoing() && !this.isComplete()) {
      throw new Error('envelope not complete');
    }
    return this.$outgoing ? this.$buffer.slice(0, this.$len) : mergeArrays(this.$fragments);
  };

  /**
   * @returns {Uint8Array[]}
   */
  BoxEnvelope.prototype.toBoxFragments = function () {
    var _this = this;
    return this.getFragments().map(function (fragment, i) {
      return toBoxFragment(_this, fragment, i);
    });
  };

  /**
   * @param {Uint8Array} arr
   * @returns {number} size of the payload in bytes
   */
  BoxEnvelope.byteLength = function (arr) {
    return arr.length;
  };
  BoxEnvelope.toBytes = function (str) {
    return toBytes(str);
  };

  /**
   * @param {module:baja/env/mux/BoxEnvelope} env
   * @param {number} fragmentIndex
   * @returns {Uint8Array}
   */
  function toBoxFragment(env, fragment, fragmentIndex) {
    return mergeArrays([toPrefix(env, fragmentIndex), fragment]);
  }

  /**
   * @param {Uint8Array[]} arrays
   */
  function mergeArrays(arrays) {
    return arrays.reduce(function (memo, array) {
      var result = new Uint8Array(memo.length + array.length);
      result.set(memo);
      result.set(array, memo.length);
      return result;
    }, new Uint8Array());
  }
  function toPrefix(env, fragmentIndex) {
    return toBytes(['F', '2.3', env.$sessionId, env.$envelopeId, env.$fragmentCount, fragmentIndex, env.$unsolicited ? 'u' : 's', ''].join(';'));
  }

  /**
   * @typedef module:baja/env/mux/BoxEnvelope~BoxFragment
   * @property {string} sessionId the server session id
   * @property {number} envelopeId the envelope id
   * @property {number} fragmentCount total fragment count of the envelope
   * @property {number} fragmentIndex fragment index within the envelope
   * @property {boolean} unsolicited true if this is an unsolicited envelope
   * sent down from the server
   * @property {Uint8Array} payload the fragment payload
   */

  return BoxEnvelope;
});
