/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam and Gareth Johnson
 */

/* eslint-env node */
'use strict';

/**
 * Hooks for running BajaScript inside of a browser.
 *
 * @module baja/env/define-node
 * @private
 */
define(["http", "bajaScript/sys", "bajaScript/comm", "bajaScript/ctypes", "bajaScript/obj", "bajaScript/nav", "bajaScript/comp", "bajaScript/bson", "bajaScript/boxcs", "bajaScript/ord", "bajaScript/coll", "bajaScript/virt", "bajaScript/tag", "bajaScript/hist", "bajaScript/alarm", "bajaScript/file"], function defineNode(http, baja) {
  // vars for async message sending
  var asyncQueue = [],
    sendAsync,
    processNext;

  /**
   * This namespace is for documentation purposes only and will not actually
   * be available to Bajascript apps. It details enhancements/decorations
   * applied to functions in <code>baja.comm</code> when Bajascript is deployed
   * to a web browser environment.
   * 
   * @namespace
   * @name baja.browser.comm
   */
  (function comm() {
    var oldStop = baja.comm.stop,
      oldStart = baja.comm.start,
      connectParams;
    baja.comm.start = function (obj) {
      if (!obj.connectParams) {
        throw new Error("connectParams required for baja.start()");
      }
      connectParams = obj.connectParams;
      oldStart(obj);
    };

    /**
     * In a browser, <code>baja.comm.reconnect</code> will simply reload the
     * browser page.
     *
     * @name baja.browser.comm.reconnect
     * @function
     * @private
     * @inner
     */
    baja.comm.reconnect = function reconnect() {
      //TODO: redo for node
    };
    baja.comm.stop = function (obj) {
      var doStop = obj.stopped || baja.ok,
        waitToShutdownTicket = setTimeout(function () {}, 5000); //give it 5 secs to shutdown

      //TODO: this is never called. is stop callback even called in a browser?
      obj.stopped = function () {
        clearInterval(waitToShutdownTicket);
        doStop();
      };
      oldStop(obj);
    };

    ////////////////////////////////////////////////////////////////
    // HTTP Comms
    //////////////////////////////////////////////////////////////// 

    /**
     * An Http Error.
     *
     * A HTTP Error happens as a result of problem with HTTP communication.
     * 
     * @class
     * @private
     * @param {XmlHttpRequest} x the object used for comms.
     */
    baja.comm.HttpError = function (x) {
      var t = x.responseText || "Session disconnected",
        status = x.status;
      this.name = "HttpError";
      this.message = t + " err: " + x.status;
      this.status = status;

      // Indicate to delay any reconnection if a 404 or no status is returned.
      this.delayReconnect = !status || status === 404;
    };
    baja.subclass(baja.comm.HttpError, baja.comm.ServerError);
    sendAsync = function sendAsync(data) {
      // Flag that we're sending the data
      data.$sending = true;
      var cb = data.cb,
        body = data.frame.toString(),
        req;

      //console.log('\n\n\nsending async');
      //console.log('request: ' + body);

      req = http.request(connectParams, function (res) {
        var body = '';
        res.on('data', function (chunk) {
          body += chunk;
        });
        res.on('end', function () {
          //console.log('\n\n\nresponse: ' + body);
          //console.log('cb.ok: ' + cb.ok);
          try {
            cb.ok(JSON.parse(body));
          } catch (e) {
            console.error(e);
            cb.fail(e);
          } finally {
            processNext();
          }
        });
      });
      req.write(body);
      req.end();
    };
    processNext = function processNext() {
      // Remove the first element the queue
      asyncQueue.shift();
      if (asyncQueue.length > 0) {
        // Remove and return the first element of the async queue
        sendAsync(asyncQueue[0]);
      }
    };

    /**
     * In a browser, <code>baja.comm.BoxFrame#send</code> will delegate to
     * <code>baja.browser.comm.sendHttp</code>.
     * 
     * @private
     * @inner
     * @ignore
     */
    baja.comm.BoxFrame.prototype.send = function (callback) {
      var that = this;
      if (!that.$sync) {
        // Push the frame onto our async queue (adds to end of array).
        asyncQueue.push({
          frame: that,
          cb: callback
        });

        // If there are no outstanding frames then simply process the queue (otherwise
        // rely on comms to make processNext request).
        if (asyncQueue.length === 1) {
          sendAsync(asyncQueue[0]);
        }
      } else {
        // For synchronous network calls, always make an HTTP call.
        sendAsync({
          frame: that,
          cb: callback
        });
      }
    };
  })();

  /**
   * This namespace is for documentation purposes only and will not actually
   * be available to Bajascript apps. It details enhancements/decorations
   * applied to functions in <code>baja</code> when Bajascript is deployed
   * to a web browser environment.
   * 
   * @namespace
   * @name baja.browser
   */
  (function sysUtil() {
    var _outln = baja.outln,
      _clearOut = baja.clearOut,
      _error = baja.error;

    /**
     * In a browser, <code>baja.outln</code> will (in addition to calling
     * <code>bajaJsPrint()</code>) look for a <code>&lt;pre&gt;</code> element 
     * with ID <code>bajaScriptOut</code> and append the message to that 
     * element as well.
     * 
     * @name baja.browser.outln
     * @function
     * @private
     * @inner
     */
    baja.outln = function (msg) {
      // If BajaScript has stopped then don't output anything else...
      if (baja.isStopping()) {
        return this;
      }
      console.log(msg);
      return _outln.call(this, msg);
    };

    /**
     * In a browser, <code>baja.clearOut</code> will look for a
     * <code>&lt;pre&gt;</code> element with ID <code>bajaScriptOut</code> and
     * wipe all text from it.
     * 
     * @name baja.browser.clearOut
     * @function
     * @private
     * @inner
     */
    baja.clearOut = function () {
      return _clearOut.apply(this, arguments);
    };

    /**
     * In a browser, <code>baja.error</code> will (in addition to calling
     * <code>baja.outln</code>) look for <code>window.console.error</code> and,
     * if it exists, pass the error message to it.
     * 
     * @name baja.browser.error
     * @function
     * @private
     * @inner
     */
    baja.error = function (msg) {
      if (baja.isStopping()) {
        return this;
      }
      console.error(msg);
      return _error.apply(this, arguments);
    };
  })();

  /**
   * This namespace is for documentation purposes only and will not actually
   * be available to Bajascript Apps. It details enhancements/decorations
   * applied to functions in <code>baja.registry</code> when Bajascript is 
   * deployed to a web browser environment.
   * 
   * @namespace
   * @name baja.browser.registry
   */
  (function sysRegistry() {

    //TODO: implement baja registry storage for node
  })();
  return baja;
});
