/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.OrdQuery}.
 * @module baja/ord/OrdQuery
 */
define(["bajaScript/sys"], function (baja) {
  "use strict";

  var subclass = baja.subclass,
    BaseBajaObj = baja.BaseBajaObj;

  /**
   * ORD Query.
   * 
   * The base class for all `OrdQuery` Objects.
   *
   * @class
   * @alias baja.OrdQuery
   * @extends baja.BaseBajaObj
   * @private
   * @inner
   */
  var OrdQuery = function OrdQuery(obj) {
    obj = obj || {};
    this.$scheme = obj.scheme;
    this.$schemeName = obj.schemeName;
    this.$body = obj.body;
    this.$isHost = obj.isHost || false;
    this.$isSession = obj.isSession || false;

    // Override any functions
    var p;
    for (p in obj) {
      if (obj.hasOwnProperty(p) && typeof obj[p] === "function") {
        this[p] = obj[p];
      }
    }
  };
  subclass(OrdQuery, BaseBajaObj);

  /**
   * Context instance passed as a parameter to the normalize function only when
   * it is called at ORD resolution time.
   *
   * @type {Object}
   * @since Niagara 4.13
   */
  OrdQuery.RESOLVING_ORD_CX = {
    resolvingOrd: true
  };

  /**
   * Return the ORD Scheme.
   *
   * @returns {baja.OrdScheme}
   */
  OrdQuery.prototype.getScheme = function () {
    return this.$scheme;
  };

  /**
   * Return the ORD Scheme name.
   *
   * @returns {String}
   */
  OrdQuery.prototype.getSchemeName = function () {
    return this.$schemeName;
  };

  /**
   * Return the body for the query.
   *
   * @returns {String}
   */
  OrdQuery.prototype.getBody = function () {
    return this.$body;
  };

  /**
   * Return a String representation of the query.
   *
   * @returns {String}
   */
  OrdQuery.prototype.toString = function () {
    return this.getSchemeName() + ":" + this.getBody();
  };

  /**
   * Return true if the Query is a Host.
   *
   * @returns {Boolean}
   */
  OrdQuery.prototype.isHost = function () {
    return this.$isHost;
  };

  /**
   * Return true if the Query is a Session.
   *
   * @returns {Boolean}
   */
  OrdQuery.prototype.isSession = function () {
    return this.$isSession;
  };

  /**
   * Normalize the query and return true if modified.
   *
   * @private
   *
   * @param {baja.OrdQueryList} list
   * @param {Number} index
   * @param {Object} [cx] As of Niagara 4.13, this optional parameter can be
   * specified to provide additional context information for this normalize
   * call. For example, if the RESOLVING_ORD_CX object is passed in, then this
   * normalize call is happening at ORD resolution time.
   *
   * @returns {Boolean}
   */
  OrdQuery.prototype.normalize = function (list, index, cx) {
    return false;
  };
  return OrdQuery;
});
