/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.Marker}.
 * @module baja/obj/Marker
 */
define(["bajaScript/sys", "bajaScript/baja/obj/Simple", "bajaScript/baja/obj/Boolean", "bajaScript/baja/obj/EnumRange"], function (baja, Simple, BBoolean, EnumRange) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper,
    def,
    encodingStr;

  /**
   * A marker value is used primarily with the tagging API in order to represent the
   * presence of a named tag that has no meaningful value. That is, the tag itself is
   * sufficient to convey meaning.
   *
   * @alias baja.Marker
   * @extends baja.Simple
   */
  var Marker = function Marker() {
    callSuper(Marker, this, arguments);
  };
  subclass(Marker, Simple);

  /**
   * Default `Marker` instance.
   *
   * @type {baja.Marker}
   */
  Marker.DEFAULT = def = new Marker();

  /**
   * Null `Marker` instance (same as `DEFAULT`).
   *
   * @type {baja.Marker}
   */
  Marker.NULL = Marker.DEFAULT;
  Marker.ENCODING_STR = encodingStr = "M";

  /**
   * Creates a new instance of `baja.Marker`.
   * 
   * @returns {baja.Marker}
   */
  Marker.make = function () {
    return def;
  };

  /**
   * @see baja.Marker.make
   * @returns {baja.Marker}
   */
  Marker.prototype.make = function () {
    return def;
  };

  /**
   * Returns the data type symbol (`m`) for `Marker` encoding.
   * 
   * @returns {String}
   */
  Marker.prototype.getDataTypeSymbol = function () {
    return 'm';
  };

  /**
   * Parse a `baja.Marker` from a `String`.
   * 
   * @method
   * @param {String} str
   * @returns {baja.Marker}
   * @throws if string is malformed.
   */
  Marker.prototype.decodeFromString = function (str) {
    if (str !== encodingStr) {
      throw new Error();
    }
    return def;
  };

  /**
   * Encode a `baja.Marker` to a `String`.
   * 
   * @method
   * @returns {String}
   */
  Marker.prototype.encodeToString = function () {
    return encodingStr;
  };

  /**
   * Equality test.
   *
   * @param obj
   * @returns {Boolean} 
   */
  Marker.prototype.equals = function (obj) {
    return obj === def;
  };

  /**
   * New Copy.
   *
   * @returns {Marker}
   */
  Marker.prototype.newCopy = function (exact) {
    return def;
  };

  /**
   * Return true if this enum's value is considered to be in an active state.
   *
   * @returns {Boolean} whether the enum is active.
   */
  Marker.prototype.isActive = function () {
    return true;
  };

  /**
   * Return the enum ordinal.
   *
   * @returns {Number} the enum ordinal.
   */
  Marker.prototype.getOrdinal = function () {
    return 1;
  };

  /**
   * Return the string identifier of this enum value.
   *
   * @returns {String} tag
   */
  Marker.prototype.getTag = function () {
    return "true";
  };

  /**
   * Asynchronously get the display tag of this enum value.
   * 
   * An Object Literal with a ok function callback must be specified.
   *
   * @param {Object} [obj] the Object Literal used to specify the method's
   * arguments.
   * 
   * @param {baja.Facets|Object} [obj.facets] optional facets used to specify 
   * the true and false text. For true text the facets key is `trueText` and
   * false is `falseText`. The argument can also be an Object Literal.
   * 
   * @param {Function} [obj.ok] (Deprecated: use Promise) ok callback called
   * with String value as argument.
   * 
   * @param {Function} [obj.fail] (Deprecated: use Promise) optional fail
   * callback.
   *
   * @returns {Promise.<String>}
   */
  Marker.prototype.getDisplayTag = function (obj) {
    return BBoolean.prototype.getDisplayTag.apply(this, arguments);
  };

  /**
   * Return the enum range.
   *
   * @returns {baja.EnumRange} the enum range.
   */
  Marker.prototype.getRange = function () {
    return EnumRange.BOOLEAN_RANGE;
  };

  /**
   * Return the Object's Icon.
   *
   * @returns {baja.Icon}
   */
  Marker.prototype.getIcon = function () {
    return BBoolean.prototype.getIcon.apply(this, arguments);
  };

  /**
   * Return a boolean (itself).
   *
   * @returns {Boolean} return the boolean (itself).
   */
  Marker.prototype.getBoolean = function () {
    return true;
  };

  /**
   * Return a boolean (itself).
   *
   * @function Marker#getEnum
   * @returns {Boolean} return the boolean (itself).
   */
  Marker.prototype.getEnum = Marker.prototype.getBoolean;

  /**
   * Returns a string representation of the `Marker`.
   *
   * If context is provided, returns a localized representation (bajaScript:Marker.displayName).
   * @see baja.Type#getDisplayName
   *
   * @param {Object} cx
   * @returns {Promise.<String>|String} returns a Promise if a cx is passed in.
   */
  Marker.prototype.toString = function (cx) {
    return this.getType().getDisplayName(cx);
  };
  return Marker;
});
