/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines a BajaScript implementation of `control:ControlPoint` (not exposed
 * on `baja` namespace).
 * @module baja/comp/ControlPoint
 */
define(["bajaScript/sys", "bajaScript/baja/comp/Component"], function (baja, Component) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper,
    slotRegex = /^(out|in([1-9]|1[0-6])|fallback|override|emergencyOverride|set)$/;

  /** the slots for which we swap out the `facets` slot for `getSlotFacets` */
  function matchesRegex(slot) {
    return slot && !!slot.getName().match(slotRegex);
  }

  /**
   * Represents a `control:ControlPoint` in BajaScript.
   *
   * @class
   * @alias module:baja/comp/ControlPoint
   * @extends baja.Component
   * @private
   */
  var ControlPoint = function ControlPoint() {
    var that = this;
    callSuper(ControlPoint, that, arguments);
    that.attach('changed', function (slot, cx) {
      if (slot.getName() === 'facets') {
        that.getSlots().filter(matchesRegex).each(function (slot) {
          that.fireHandlers('facetsChanged', baja.error, that, slot, cx);
        });
      }
    });
  };

  // Set up the prototype chain
  subclass(ControlPoint, Component);

  /**
   * Return the `Facets` for a Slot.
   * 
   * If no arguments are provided and the `Complex` has a parent, the 
   * facets for the parent's Property will be returned. 
   *
   * @private
   * @inner
   *
   * @param {baja.Slot|String} [slot]  the Slot or Slot name.
   * @returns {baja.Facets} the Facets for the Slot (or null if Slot not found) or
   *                        the parent's Property facets.
   */
  ControlPoint.prototype.getFacets = function (slot) {
    // Attempt to match Station Component's 'getSlotFacets' implementation...
    if (slot && matchesRegex(this.getSlot(slot))) {
      return this.get("facets");
    }

    // Call base class 'getFacets'
    return callSuper("getFacets", ControlPoint, this, arguments);
  };

  /**
   * Returns a promise that resolves with the string value of the out `StatusValue`
   *
   * @param {Object} [cx] a context object
   * @returns {Promise.<string>|string}
   */
  ControlPoint.prototype.toString = function (cx) {
    if (!cx) {
      return Component.prototype.toString.apply(this, arguments);
    }
    return this.propertyValueToString('out', cx);
  };

  /**
   * @private
   * @since Niagara 4.13
   * @returns {baja.StatusValue}
   */
  ControlPoint.prototype.getOutStatusValue = function () {
    return this.get('out');
  };

  /**
   * @private
   * @since Niagara 4.13
   * @returns {baja.StatusValue}
   */
  ControlPoint.prototype.getStatusValue = function () {
    return this.getOutStatusValue();
  };

  /**
   * @private
   * @since Niagara 4.13
   * @param {Object} [cx] a context object
   * @returns {Promise.<string>}
   */
  ControlPoint.prototype.getValueWithFacets = function (cx) {
    cx = Object.assign(this.get('facets').toObject(), cx);
    return this.getStatusValue().getValue().toString(cx);
  };

  /**
   * @private
   * @since Niagara 4.13
   * @returns {baja.Status}
   */
  ControlPoint.prototype.getStatus = function () {
    return this.getOutStatusValue().getStatus();
  };
  return ControlPoint;
});
