/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/boxcs/syncUtil
 * @private
 */
define([], function syncUtil() {
  "use strict";

  var syncComp,
    syncSlots,
    syncVal,
    emptyCommitCx = {
      commit: true,
      serverDecode: true,
      syncStructVals: true,
      fromLoad: true
    };

  /**
   * Private fw method used by LoadOp for synchronizing two Components.
   *
   * @param from  the from Component.
   * @param to    the to Component.
   *
   * @private
   */
  syncComp = function syncComp(from, to) {
    // Sanity check - must be same Type
    if (from.getType().getTypeSpec() !== to.getType().getTypeSpec()) {
      throw new Error("LoadOp Types differ: " + from.getType() + " - " + to.getType());
    }

    // Sanity check - must have same handle
    if (to.getHandle() !== from.getHandle()) {
      throw new Error("LoadOp Handle Error: " + from.getHandle() + " - " + to.getHandle());
    }

    // Update display name
    if (from.getPropertyInParent() !== null) {
      to.getPropertyInParent().$setDisplayName(from.getPropertyInParent().$getDisplayName());
      to.getPropertyInParent().$setDisplay(from.getPropertyInParent().$getDisplay());
    }

    // Map over cached permissions
    to.$fw("setPermissions", from.$permissionsStr);

    // If this component isn't loaded then don't try to sync its slots
    if (!from.$bPropsLoaded) {
      // TODO: Sync icon
      return;
    }

    // Signal the broker properties are loaded
    to.$bPropsLoaded = true;

    // Note down Properties before synchronization
    var pb = {},
      // Properties before
      tslots = to.$map.$map,
      // Access internal OrderedMap instead of cursor for speed
      p;
    for (p in tslots) {
      if (tslots.hasOwnProperty(p) && tslots[p].isProperty()) {
        pb[p] = tslots[p];
      }
    }
    var fslots = from.$map.$map,
      // Access internal OrderedMap for speed
      name,
      fromSlot,
      toSlot,
      fromFlags,
      fromValue,
      cx,
      reorder = false,
      reorderSlots;
    for (p in fslots) {
      if (fslots.hasOwnProperty(p)) {
        fromSlot = fslots[p];
        name = fromSlot.getName();
        toSlot = to.getSlot(name);
        fromFlags = from.getFlags(fromSlot);
        if (!fromSlot.isFrozen()) {
          reorderSlots = reorderSlots || [];
          reorderSlots.push(name);
        }

        // If to slot is not present then we need to add
        if (toSlot === null) {
          // TODO: Handle display String
          cx = {
            commit: true,
            serverDecode: true,
            fromLoad: true,
            displayName: fromSlot.$getDisplayName(),
            display: fromSlot.$getDisplay()
          };
          fromValue = from.get(name);
          if (fromValue.$parent) {
            fromValue.$parent = null; // TODO: Hack to get around any parenting problems
          }
          to.add({
            "slot": name,
            "value": fromValue,
            "flags": fromFlags,
            "facets": fromSlot.getFacets(),
            "cx": cx
          });
          continue;
        }

        // If there's already a dynamic slot on the 'to' Component then attempt a reorder
        if (!fromSlot.isFrozen()) {
          reorder = true;
        }
        delete pb[name];

        // Set the flags if they differ
        if (fromFlags !== toSlot.getFlags()) {
          to.setFlags({
            "slot": toSlot,
            "flags": fromFlags,
            "cx": emptyCommitCx
          });
        }
        if (!fromSlot.isProperty()) {
          continue;
        }
        syncSlots(from, fromSlot, to, toSlot);
      }
    }

    // at this point if there were any props before that we didn't 
    // sync we need to remove them now; since they weren't found 
    // on "from" that means they have since been removed
    var removeName;
    for (removeName in pb) {
      if (pb.hasOwnProperty(removeName)) {
        to.remove({
          "slot": pb[removeName],
          "cx": emptyCommitCx
        });
      }
    }

    // Sync Knobs
    var fromKnobs = from.$knobs,
      toKnobs = to.$knobs,
      installKnobs,
      uninstallKnobs,
      i,
      x;
    if (fromKnobs) {
      installKnobs = [];

      // Install any knobs missing from the to Component
      for (p in fromKnobs) {
        if (fromKnobs.hasOwnProperty(p)) {
          if (!toKnobs) {
            installKnobs.push(fromKnobs[p]);
          } else if (!toKnobs.hasOwnProperty(p)) {
            installKnobs.push(fromKnobs[p]);
          }
        }
      }
    }
    if (toKnobs) {
      uninstallKnobs = [];

      // Install any knobs missing from the to Component
      for (p in toKnobs) {
        if (toKnobs.hasOwnProperty(p)) {
          if (!fromKnobs) {
            uninstallKnobs.push(toKnobs[p]);
          } else if (!fromKnobs.hasOwnProperty(p)) {
            uninstallKnobs.push(toKnobs[p]);
          }
        }
      }
    }
    if (installKnobs) {
      for (i = 0; i < installKnobs.length; ++i) {
        to.$fw("installKnob", installKnobs[i], emptyCommitCx);
      }
    }
    if (uninstallKnobs) {
      for (x = 0; x < uninstallKnobs.length; ++x) {
        to.$fw("uninstallKnob", uninstallKnobs[x].getId(), uninstallKnobs[x].getSourceSlotName(), emptyCommitCx);
      }
    }

    // Sync Relation Knobs
    var fromRelationKnobs = from.$rknobs,
      toRelationKnobs = to.$rknobs,
      installRelationKnobs,
      uninstallRelationKnobs;
    if (fromRelationKnobs) {
      installRelationKnobs = [];

      // Install any knobs missing from the to Component
      for (p in fromRelationKnobs) {
        if (fromRelationKnobs.hasOwnProperty(p)) {
          if (!toRelationKnobs) {
            installRelationKnobs.push(fromRelationKnobs[p]);
          } else if (!toRelationKnobs.hasOwnProperty(p)) {
            installRelationKnobs.push(fromRelationKnobs[p]);
          }
        }
      }
    }
    if (toRelationKnobs) {
      uninstallRelationKnobs = [];

      // Install any knobs missing from the to Component
      for (p in toRelationKnobs) {
        if (toRelationKnobs.hasOwnProperty(p)) {
          if (!fromRelationKnobs) {
            uninstallRelationKnobs.push(toRelationKnobs[p]);
          } else if (!fromRelationKnobs.hasOwnProperty(p)) {
            uninstallRelationKnobs.push(toRelationKnobs[p]);
          }
        }
      }
    }
    if (installRelationKnobs) {
      for (i = 0; i < installRelationKnobs.length; ++i) {
        to.$fw("installRelationKnob", installRelationKnobs[i], emptyCommitCx);
      }
    }
    if (uninstallRelationKnobs) {
      for (x = 0; x < uninstallRelationKnobs.length; ++x) {
        to.$fw("uninstallRelationKnob", uninstallRelationKnobs[x].getId(), emptyCommitCx);
      }
    }

    // Attempt reorder
    if (reorder && reorderSlots) {
      to.reorder({
        dynamicProperties: reorderSlots,
        cx: emptyCommitCx
      });
    }
  };

  /**
   * Private fw method used by LoadOp for synchronizing Slots between two Components.
   *
   * @param from      the from Complex.
   * @param fromSlot  the from Complex Slot.
   * @param to        the to Complex.
   * @param toSlot    the to Complex Slot.
   *
   * @private
   */
  syncSlots = function syncSlots(from, fromSlot, to, toSlot) {
    // Sync the display names between the slots
    toSlot.$setDisplayName(fromSlot.$getDisplayName());

    // Don't sync any further if the Slot isn't a Property
    if (!fromSlot.isProperty()) {
      return;
    }

    // If neither of these Slots have been decoded then skip trying to sync!
    if (fromSlot.isFrozen() && !fromSlot.$isValueDecoded() && !toSlot.$isValueDecoded() && fromSlot.getType().getTypeSpec() === toSlot.getType().getTypeSpec()) {
      return;
    }

    // TODO: Update display string
    syncVal(from.get(fromSlot), to, toSlot, fromSlot.$getDisplay());
  }; // function syncSlots(

  /**
   * Private fw method used to synchronize a Property value in a Component.
   *
   * @param fromValue  the from Value.
   * @param to        the to Component.
   * @param toProp    the to Component Property.
   * @param display  the from display.
   *
   * @private
   */
  syncVal = function syncVal(fromValue, to, toProp, display) {
    var toValue = to.get(toProp),
      isFromValComp = fromValue.getType().isComponent();
    toProp.$setDisplay(display);
    // TODO: Update display string

    // we need to do a full replace if:
    //  - the type has changed 
    //  - the value is a simple
    //  - we need to assign component handle 
    if (fromValue.getType().getTypeSpec() !== toValue.getType().getTypeSpec() || fromValue.getType().isSimple() || toValue.getType().isComponent() && toValue.getHandle() === null) {
      if (fromValue.$parent) {
        fromValue.$parent = null; // TODO: Hack to get around the fact that we haven't implemented newCopy yet
      }

      // TODO: What about display name and display getting passed through here?
      to.set({
        "slot": toProp,
        "value": fromValue,
        "cx": emptyCommitCx
      });
      return;
    }

    // Do a full sync if this is a Component
    if (isFromValComp) {
      syncComp(fromValue, toValue);
      return;
    }

    // Otherwise this is a Struct so do a sync in place
    var fslots = fromValue.$map.$map,
      // Use internal OrderedMap instead of Cursors for speed
      nextFromProp,
      // From Property
      p;
    for (p in fslots) {
      if (fslots.hasOwnProperty(p)) {
        nextFromProp = fslots[p];
        syncSlots(fromValue, nextFromProp, toValue, toValue.getSlot(nextFromProp.getName()));
      }
    }
  }; // function syncVal(

  return {
    syncComp: syncComp,
    syncSlots: syncSlots,
    syncVal: syncVal
  };
});
