/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

define(['baja!', 'Promise', 'nmodule/tagdictionary/rc/taginfo/TagInfoMixin', 'nmodule/tagdictionary/rc/taginfo/TagGroupInfoMixin'], function (baja, Promise, TagInfoMixin, TagGroupInfoMixin) {
  'use strict';

  var Id = require('bajaScript/baja/tag/Id');

  /**
   * A utility class for making and dealing with tagging related rpc calls
   *
   * API Status: **Private**
   * @exports nmodule/tagdictionary/rc/util/rpcUtil
   */
  var exports = {};
  var TAG_NAME_KEY = 'n',
    DICTIONARY_KEY = 'd',
    TAG_TYPE_KEY = 't',
    TAG_B_TYPE_KEY = 'b',
    DEFAULT_VALUE_KEY = 'v',
    IS_IDEAL_KEY = 'ii',
    IS_VALID_KEY = 'iv',
    DICTIONARY_NAME_KEY = 'dn';
  exports.DICTIONARY_KEY = DICTIONARY_KEY;
  exports.DICTIONARY_NAME_KEY = DICTIONARY_NAME_KEY;
  var taggingRpc = function taggingRpc(methodName, argList, batch) {
    return baja.rpc({
      typeSpec: 'tagdictionary:TaggingRpcUtil',
      methodName: methodName,
      args: argList,
      batch: batch
    });
  };

  /**
   * Returns the ord for the Tag Dictionary service, returns an empty string if there isn't one.
   * @returns {Promise.<String>}
   */
  exports.getTagDictionaryServiceOrd = function () {
    return taggingRpc('getTagDictionaryServiceOrd', []);
  };

  /**
   * Retrieve the tagInfos for the given tagDictionary.
   * @param {String} tagDictionary
   * @param {String} targetComponentOrdStr
   * @returns {Promise.<Array.<TagInfoMixin>>}
   */
  exports.getTagInfos = function (tagDictionary, targetComponentOrdStr) {
    return taggingRpc('getTagInfos', [tagDictionary, targetComponentOrdStr]).then(function (result) {
      var tagInfos = [];
      result.forEach(function (tagInfoJson) {
        tagInfos.push(convertToTagInfoMixin(tagInfoJson));
      });
      return Promise.all(tagInfos);
    });
  };

  /**
   * Retrieve the tagGroupInfos for the given tagDictionary.
   * @param {String} tagDictionary
   * @param {String} targetComponentOrdStr
   * @returns {Promise.<Array.<TagGroupInfoMixin>>}
   */
  exports.getTagGroupInfos = function (tagDictionary, targetComponentOrdStr) {
    return taggingRpc('getTagGroupInfos', [tagDictionary, targetComponentOrdStr]).then(function (result) {
      var tagGroupInfos = [];
      result.forEach(function (tagGroupInfoJson) {
        tagGroupInfos.push(convertToTagGroupInfoMixin(tagGroupInfoJson));
      });
      return Promise.all(tagGroupInfos);
    });
  };

  /**
   * Retrieve the tagInfos for the tagGroup.
   * @param {String} tagGroupOrdStr
   * @param {String} targetComponentOrdStr
   * @returns {Promise.<Array.<TagInfoMixin>>}
   */
  exports.getTagInfosForTagGroup = function (tagGroupOrdStr, targetComponentOrdStr) {
    return taggingRpc('getTagInfosForTagGroup', [tagGroupOrdStr, targetComponentOrdStr]).then(function (result) {
      var tagInfos = [];
      result.forEach(function (tagInfoJson) {
        tagInfos.push(convertToTagInfoMixin(tagInfoJson));
      });
      return Promise.all(tagInfos);
    });
  };

  /**
   * Retrieve all dictionaries which we have access to.
   *
   * @returns {Promise.<Array.<Object>>} returns a Promise that resolves to
   * a list of objects, each object containing information about a tagdictionary.
   *
   */
  exports.getTagDictionariesInfo = function () {
    return taggingRpc('getTagDictionariesInfo', []);
  };

  /**
   * Retrieve all relation ids for all dictionaries which we have access to.
   * @returns {Promise.<Array.<String>>}
   */
  exports.getRelationIds = function () {
    return taggingRpc('getRelationIds', []);
  };

  /*
   * @returns {Promise.<TagInfoMixin>}
   */
  function convertToTagInfoMixin(tagInfoJson) {
    return baja.bson.decodeAsync(JSON.parse(tagInfoJson[DEFAULT_VALUE_KEY])).then(function (defaultValue) {
      return new TagInfoMixin(new Id(tagInfoJson[DICTIONARY_KEY], tagInfoJson[TAG_NAME_KEY]), defaultValue, tagInfoJson[TAG_B_TYPE_KEY], tagInfoJson[IS_IDEAL_KEY], tagInfoJson[IS_VALID_KEY], tagInfoJson[TAG_TYPE_KEY]);
    });
  }

  /*
   * @returns {Promise.<TagInfoMixin>}
   */
  function convertToTagGroupInfoMixin(tagGroupInfoJson) {
    return baja.bson.decodeAsync(JSON.parse(tagGroupInfoJson[DEFAULT_VALUE_KEY])).then(function (defaultValue) {
      return new TagGroupInfoMixin(new Id(tagGroupInfoJson[DICTIONARY_KEY], tagGroupInfoJson[TAG_NAME_KEY]), defaultValue, tagGroupInfoJson[TAG_B_TYPE_KEY], tagGroupInfoJson[IS_IDEAL_KEY], tagGroupInfoJson[IS_VALID_KEY]);
    });
  }
  return exports;
});
