var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

/**
 * @file Functions relating to navbar management in Niagara apps.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module mobile/util/nav
 */
define(['baja!', 'bajaux/Widget', 'Promise', 'underscore', 'mobile/util/views/views', 'mobile/util/ord', 'mobile/util/mobile/mobile'], function (baja, Widget, Promise, _, views, ordUtil, mobileUtil) {

  "use strict";

  //imports

  var getComponentName = views.getComponentName;

  ////////////////////////////////////////////////////////////////
  //NavModel
  ////////////////////////////////////////////////////////////////


  /**
   * Wraps an object in another object that provides extra functionality needed
   * for navigation. Attempts to derive navigation information from the given
   * object like nav ORD, icon, and description.
   * 
   * @class
   * @memberOf {module:mobile/util/nav}
   * @param {String|baja.Ord|baja.Complex|module:mobile/util/mobile/views/PageView} obj an
   * object to be used for navigation
   */
  var NavModelNode = function NavModelNode(obj) {
    this.$navChildren = [];
    this.$value = obj;
  };

  /**
   * @returns {String} this node's nav ORD
   */
  NavModelNode.prototype.getNavOrd = function () {
    return this.$navOrd;
  };

  /**
   * @param {String} navOrd the nav ORD to set
   */
  NavModelNode.prototype.setNavOrd = function (navOrd) {
    this.$navOrd = navOrd;
  };

  /**
   * @returns {String} this node's nav description
   */
  NavModelNode.prototype.getNavDescription = function () {
    return this.$navDescription;
  };

  /**
   * @param {String} navDescription the nav description to set
   */
  NavModelNode.prototype.setNavDescription = function (navDescription) {
    this.$navDescription = navDescription;
  };

  /**
   * @returns {baja.Icon} this node's nav icon
   */
  NavModelNode.prototype.getNavIcon = function () {
    return this.$navIcon;
  };

  /**
   * @param {baja.Icon} navIcon the nav icon to set
   */
  NavModelNode.prototype.setNavIcon = function (navIcon) {
    this.$navIcon = navIcon;
  };

  /**
   * @returns {Array} an array of this node's nav children (also NavModelNode)
   */
  NavModelNode.prototype.getNavChildren = function () {
    return this.$navChildren;
  };

  /**
   * @private
   * @param {Array} navChildren the nav children to set (must be NavModelNode)
   */
  NavModelNode.prototype.setNavChildren = function (navChildren) {
    this.$navChildren = navChildren;
  };

  /**
   * @returns {niagara.util.nav.NavModelNode} this node's nav parent
   */
  NavModelNode.prototype.getNavParent = function () {
    return this.$navParent;
  };

  /**
   * @private
   * @param {niagara.util.nav.NavModelNode} navParent the parent to set
   */
  NavModelNode.prototype.setNavParent = function (navParent) {
    this.$navParent = navParent;
  };

  /**
   * @returns {Object} this nav node's current value (ORD, component, view, etc)
   */
  NavModelNode.prototype.getValue = function () {
    return this.$value;
  };

  /**
   * @private
   * @param {Object} value the value to set
   */
  NavModelNode.prototype.setValue = function (value) {
    this.$value = value;
  };

  /**n
   * @returns a String representation of this node (just the nav ORD)
   */
  NavModelNode.prototype.toString = function () {
    return String(this.$navOrd);
  };

  NavModelNode.prototype.toDisplayName = function () {
    return Promise.resolve(this.$navDescription);
  };

  var OrdNavModelNode = function OrdNavModelNode(ord) {
    NavModelNode.apply(this, arguments);

    var $navOrd = baja.Ord.make(ord),
        $navDescription = String($navOrd),
        split = $navDescription.split('/');

    this.$navOrd = $navOrd;
    this.$navDescription = split[split.length - 1];
  };
  OrdNavModelNode.prototype = Object.create(NavModelNode.prototype);
  OrdNavModelNode.prototype.constructor = OrdNavModelNode;

  var ComplexNavModelNode = function ComplexNavModelNode(complex) {
    NavModelNode.apply(this, arguments);

    this.$navDescription = getComponentName(complex);
    this.$navOrd = ordUtil.deriveOrd(complex);
    this.$navIcon = complex.getIcon();
  };
  ComplexNavModelNode.prototype = Object.create(NavModelNode.prototype);
  ComplexNavModelNode.prototype.constructor = OrdNavModelNode;

  var ViewNavModelNode = function ViewNavModelNode(view) {
    NavModelNode.apply(this, arguments);

    var $navOrd,
        $navIcon,
        value = view.value();

    this.$view = view;

    if (value instanceof baja.Complex) {
      $navOrd = ordUtil.deriveOrd(value);
      $navIcon = value.getIcon();
    }

    if (!$navOrd && view.pageId) {
      $navOrd = mobileUtil.decodePageId(view.pageId);
      if ($navOrd.indexOf('ord?') === 0) {
        $navOrd = $navOrd.substring(4);
      }
    }

    this.$navOrd = $navOrd;
    this.$navIcon = $navIcon;
  };
  ViewNavModelNode.prototype = Object.create(NavModelNode.prototype);
  ViewNavModelNode.prototype.constructor = ViewNavModelNode;

  ViewNavModelNode.prototype.toDisplayName = function () {
    var that = this,
        displayName = that.$displayName;

    if (displayName) {
      return Promise.resolve(displayName);
    } else {
      return that.$view.toDisplayName().then(function (displayName) {
        that.$displayName = displayName;
        return displayName;
      });
    }
  };

  var ObjectNavModelNode = function ObjectNavModelNode(obj) {
    this.$navOrd = obj.$navOrd;
    this.$navDescription = obj.$navDescription;
    this.$navIcon = obj.$navIcon;
    this.$navChildren = obj.$navChildren || [];
    this.$navParent = obj.$navParent;
    this.$value = obj.$value;
  };
  ObjectNavModelNode.prototype = Object.create(NavModelNode.prototype);

  function makeNavModelNode(obj) {
    if (obj instanceof NavModelNode) {
      return obj;
    } else if (obj instanceof baja.Ord || typeof obj === 'string') {
      return new OrdNavModelNode(obj);
    } else if (obj instanceof Widget) {
      return new ViewNavModelNode(obj);
    } else if (obj instanceof baja.Complex) {
      return new ComplexNavModelNode(obj);
    } else if (obj && (typeof obj === 'undefined' ? 'undefined' : _typeof(obj)) === 'object') {
      return new ObjectNavModelNode(obj);
    }
  }

  ////////////////////////////////////////////////////////////////
  //NavModel
  ////////////////////////////////////////////////////////////////

  /**
   * Models a stack of navigation nodes, intended to back a navbar at the
   * top of the screen. A NavModel is backed by a simple array that is treated
   * like a stack using `push()` and `pop()`. It will typically represent a
   * single path through a tree, like a component space or a nav file.
   * 
   * @class
   * @memberOf module:mobile/util/nav
   */
  var NavModel = function NavModel() {
    this.$selectedNode = null;
    this.$nodes = [];
    this.$changeListeners = [];
    this.$cutListeners = [];
  };

  /*
   * Give NavModel event capabilities
   */
  baja.event.mixin(NavModel.prototype);

  /**
   * Adds a new navigation node. Default behavior is simply to push the new
   * node onto the end of the stack.
   * 
   * NavModel also gets a BajaScript event mixin, so you can can bind to the
   * following events:
   * 
   * - `change` - a nav node was added or removed, or the selected node was
   *   changed
   * - `cut` - a nav node was removed (the removed node is passed as first
   *   argument to the event handler
   * 
   * @param {Object} obj the node to add
   * @returns {module:mobile/util/nav.NavModel} this
   */
  NavModel.prototype.add = function add(obj) {
    var node = makeNavModelNode(obj);

    this.pruneOtherBranch(node);
    this.$nodes.push(node);

    return this;
  };

  /**
   * Removes a node from the stack, along with all descendant nodes.
   * 
   * @param {Object} node the node to cut
   * @returns {module:mobile/util/nav.NavModel} this
   */
  NavModel.prototype.cut = function cut(node) {
    if (!node || this.indexOf(node) < 0) {
      return this;
    }

    node = makeNavModelNode(node);

    var that = this,
        next = that.next(node);

    if (next) {
      that.cut(next);
    }

    that.$nodes.pop();

    that.fireHandlers('cut', baja.error, this, node);
    return that;
  };

  /**
   * Returns the currently selected node.
   *
   * @returns {Object} the currently selected node
   */
  NavModel.prototype.getSelectedNode = function getSelectedNode() {
    return this.$selectedNode;
  };

  /**
   * Returns the index of the currently selected node.
   *
   * @returns {Number} the currently selected index
   */
  NavModel.prototype.getSelectedIndex = function getSelectedIndex() {
    return this.indexOf(this.getSelectedNode());
  };

  /**
   * Sets the currently selected node.
   *
   * @param {Object} node the node to set as selected
   * @throws {Error} if the given node does not already exist in this NavModel
   */
  NavModel.prototype.setSelectedNode = function setSelectedNode(node) {
    node = makeNavModelNode(node);

    if (this.indexOf(node) < 0) {
      this.add(node);
    }

    this.$selectedNode = node;
    this.fireHandlers('change', baja.error, this);
  };

  /**
   * Returns all the nodes currently contained in this NavModel.
   *
   * @returns {Array} an array of nodes
   */
  NavModel.prototype.getNodes = function getNodes() {
    return this.$nodes;
  };

  /**
   * Returns an array of the ORDs corresponding to all the nodes contained
   * in this NavModel.
   *
   * @returns {Array} an array of ORDs
   * @see {module:mobile/util/nav.NavModelNode#getNavOrd}
   */
  NavModel.prototype.getOrds = function getOrds() {
    var ords = [],
        that = this;
    that.each(function (node) {
      ords.push(node.getNavOrd());
    });
    return ords;
  };

  /**
   * Iterate over the nodes in this NavModel, executing a function on each one.
   *
   * @param {Function} func a function to execute on each node - the node will
   * passed as the first argument and the node's index as the second
   * @returns the first value returned from `func`
   */
  NavModel.prototype.each = function each(func) {
    return baja.iterate(this.$nodes, func);
  };

  /**
   * Check to see if this NavModel contains no nodes.
   *
   * @returns {Boolean} true if the NavModel is empty
   */
  NavModel.prototype.isEmpty = function isEmpty() {
    return this.$nodes.length === 0;
  };

  /**
   * Returns the first node in this NavModel - often, this might correspond to
   * the station root.
   *
   * @returns {Object} the first (root) node
   */
  NavModel.prototype.first = function first() {
    return this.get(0);
  };

  /**
   * Returns the last node in this NavModel.
   *
   * @returns {Object} the last node
   */
  NavModel.prototype.last = function last() {
    return this.get(this.size() - 1);
  };

  /**
   * Returns the index of the given node. The index is determined by examining
   * the node's ORD (using `getNodeOrd`) and comparing it to the ORDs of the
   * other nodes in this NavModel.
   *
   * @param {Object} node the node whose index to find
   * @returns {Number} the index of the given node, or -1 if not found 
   */
  NavModel.prototype.indexOf = function indexOf(node) {
    if (!node) {
      return -1;
    }

    node = makeNavModelNode(node);

    var that = this,
        ord = node.getNavOrd(),
        index;

    index = that.each(function (node, i) {
      var otherOrd = node.getNavOrd();
      if (ordUtil.equivalent(ord, otherOrd)) {
        return i;
      }
    });

    return index === undefined ? -1 : index;
  };

  /**
   * Returns the node at the specified index.
   *
   * @param {Number} index the desired node index
   * @returns {Object} the node at the specified index
   */
  NavModel.prototype.get = function get(index) {
    return this.$nodes[index];
  };

  /**
   * Returns the node next in line (to the right of) the given node. If the
   * given node does not exist in the NavModel, or if there are no nodes to
   * the right of it, undefined will be returned.
   *
   * @param {Object} node find the node to the right of this node
   * @returns {Object} the next node in line, or undefined if not found
   */
  NavModel.prototype.next = function next(node) {
    var index = this.indexOf(node);
    return index >= 0 ? this.get(index + 1) : undefined;
  };

  /**
   * Returns the node previous (to the left of) the given node. If the given
   * node does not exist in the NavModel, or if there are no nodes to the left
   * of it, undefined will be returned.
   *
   * @param {Object} node find the node to the left of this node
   * @returns {Object} the previous node in line, or undefined if not found
   */
  NavModel.prototype.prev = function prev(node) {
    var index = this.indexOf(node);
    return index >= 0 ? this.get(index - 1) : undefined;
  };

  /**
   * Returns the number of nodes in this NavModel.
   *
   * @returns {Number} the number of nodes in this NavModel
   */
  NavModel.prototype.size = function size() {
    return this.$nodes.length;
  };

  /**
   * Prunes out any open views from other branches.
   * 
   * When navigating a component hierarchy, only one branch of the tree should
   * have components subscribed at any one time - otherwise, given enough
   * time and clicks, we could have dozens or hundreds of components with
   * subscribed views, eating up memory and CPU cycles. This function can prune
   * away other nodes.
   * 
   * For instance, if we start at `station:|slot:` and navigate down
   * to `station:|slot:/Services`, we will have two views open and subscribed.
   * If we go back up to the root and then navigate down to
   * `station:|slot:/Drivers`, the view on `Services` will be stopped and its
   * corresponding page will be removed from the DOM, since it is on a
   * different branch from the one currently being viewed.
   *
   * @param {module:mobile/util/nav.NavModelNode} node the view we wish to add
   * to the stack
   */
  NavModel.prototype.pruneOtherBranch = function pruneOtherBranch(node) {
    var that = this;

    if (that.isEmpty()) {
      return;
    }

    if (that.sameBranch(node, that.last())) {
      return;
    }

    baja.iterate(1, this.size(), function (i) {
      var otherNode = that.get(i);
      if (!that.sameBranch(node, otherNode)) {
        return that.cut(otherNode);
      }
    });
  };

  /**
   * Used in `pruneOtherBranch`, this function checks to see if two nodes are
   * in the same branch of whatever tree we're navigating. This allows us to
   * prune away any other nodes that are not in the same branch of our nav
   * tree each time we attempt to add a new node to our NavModel.
   * 
   * Checks to see if the two nodes exist in the same branch of the component
   * tree. It simply converts the two nodes to ORDs (using  `getNodeOrd`), and
   * checks to see if one is a substring of the other, starting at string index 0.
   * 
   * @param {Object} node1
   * @param {Object} node2
   * @returns {Boolean} true if the two nodes are on the same branch
   */
  NavModel.prototype.sameBranch = function sameBranch(node1, node2) {
    var ord1 = decodeURIComponent(String(node1.getNavOrd())),
        ord2 = decodeURIComponent(String(node2.getNavOrd()));

    ord1 = ord1.replace(/\/+/g, '/');
    ord2 = ord2.replace(/\/+/g, '/');

    return ord1.indexOf(ord2) === 0 || ord2.indexOf(ord1) === 0;
  };

  ////////////////////////////////////////////////////////////////
  //ComponentSpaceNavModel
  ////////////////////////////////////////////////////////////////

  /**
   * A NavModel that supports navigation up and down a simple component
   * tree.
   * 
   * @class
   * @memberOf module:mobile/util/nav
   * @extends module:mobile/util/nav.NavModel
   */
  var ComponentSpaceNavModel = function ComponentSpaceNavModel() {
    NavModel.call(this);
  };
  ComponentSpaceNavModel.prototype = Object.create(NavModel.prototype);
  ComponentSpaceNavModel.prototype.constructor = ComponentSpaceNavModel;

  /**
   * Override of the `add` function that ensures that the length of the node
   * stack is always the same as our depth into the component tree.
   * 
   * If we initialize our NavModel with a component at  `station:|slot:/` (the
   * station root), then our NavModel length will be 1.
   * 
   * However if we initialize with, say, the component at 
   * `station:|slot:/Services/UserService`, then our length will start out at
   * 3 - the initialization process goes like this:
   * 
   * # Set our mounted component at the proper position (position is dictated
   *   by depth into the component tree), and populate all nodes previous to it
   *   with simple string ORDs (in our example, `get(0)` will return
   *   `'station:|slot:/'` and `get(1)` will return `'station:|slot:/Services'`.
   *   This will be state of the NavModel as soon as `add` returns. (A `change`
   *   event will also be triggered before the function returns.)
   * 
   * # For any spots in our NavModel that don't have mounted components (here,
   *   0 and 1), use a `baja.BatchResolve` to resolve all the string ORDs to
   *   actual mounted components.
   * 
   * # Once these mounted components return, set them directly as navigation
   *   nodes. Trigger a `change` event to allow any navbars that use this
   *   NavModel to update themselves with the proper display names / icons that
   *   we now have. 
   * 
   * @param {Object} obj the node to add
   * @returns {module:mobile/util/nav.ComponentSpaceNavModel} this
   */
  ComponentSpaceNavModel.prototype.add = function (obj) {
    var that = this,
        node = makeNavModelNode(obj),
        nodes = that.$nodes,
        toResolve = {},
        toResolveArray = [],
        ord = node.getNavOrd(),
        ordHistory = ordUtil.makeOrdHistory(ord);

    that.pruneOtherBranch(node);

    /*
     * When we first load a page, unless we're only looking at the station root
     * we will need to retrieve the components to obtain their display names
     * and icons for the navbar. But this is an asynchronous action and we
     * have to show *something* while it's taking place, so we start by simply
     * storing an ORD in the view list, then replace with the component once
     * it is retrieved. (We don't do this for the slot we're adding a view for
     * - there's no need.)
     */
    baja.iterate(0, ordHistory.length - 1, function (i) {
      if (!nodes[i]) {
        var ord = ordHistory[i],
            len = toResolveArray.length;

        //spot in our node list is empty - set it to the ORD first...
        nodes[i] = makeNavModelNode(ord);

        //then make a record saying that we need to include this ORD in our
        //batch resolve
        toResolve['ord' + len] = {
          ord: ord,
          origIndex: i
        };
        toResolveArray.push(ord);
      }
    });

    if (toResolveArray.length) {
      //we have ORD sitting in our view stack - so let's replace them with
      //live components
      new baja.BatchResolve(toResolveArray).resolve({
        ok: function ok() {
          var objs = this.getTargetObjects();

          baja.iterate(objs, function (obj, i) {
            //replace the ORD in our view stack with the actual component
            var thing = toResolve['ord' + i];
            nodes[thing.origIndex] = makeNavModelNode(obj);
          });

          /*
           * Once the stack is fully populated with placeholders, trigger a 
           * change event so the nav bar (if we have one) can redraw itself 
           * with proper display names and icons.
           */
          that.fireHandlers('change', baja.error, this);
        },
        fail: baja.error
      });
    }

    nodes[ordHistory.length - 1] = node;
    that.fireHandlers('change', baja.error, this);
    return this;
  };

  /**
   * Converts a nav file root node into a flat hierarchy of 
   * `NavModelNode`s. Calls itself recursively to flatten the
   * node's children as well.
   *
   * @param {baja.NavContainer} node the nav file root node to flatten
   */
  function flattenNavFileNode(node) {}
  //TODO: if we ever bring back nav file support, revisit
  /* callbacks = callbackify(callbacks);
  
  var flattened = makeNavModelNode({
    $navIcon: node.getNavIcon(),
    $navDescription:  node.getNavDescription(),
    $navOrd: String(node.getNavOrd()).replace('local:|', '')
  });
  flattened.$fromNavFile = true;
  
  node.getNavChildren(function (kids) {
    if (kids && kids.length) {
      var invocs = [];
      
      baja.iterate(kids, function (kid) {
        invocs.push(function (parent) {
          var that = this;
          flattenNavFileNode(kid, function (flattenedKid) {
            parent.$navChildren.push(flattenedKid);
            flattenedKid.$navParent = parent;
            that.ok(parent);
          });
        });
      });
      util.flow.sequential.apply(this, invocs).invoke(flattened, callbacks);
    } else {
      callbacks.ok(flattened);
    }
  }); */


  /**
   * Examines a nav node, looking for a particular ORD. If the given node has
   * the given ORD, return that node. Otherwise, recursively examine the node's
   * children.
   *
   * @param {module:mobile/util/nav.NavModelNode} node the node to check for an ORD
   * @param {String|baja.Ord} ord the ORD to look for
   * @returns {module:mobile/util/nav.NavModelNode} the node matching the given ORD,
   * or null if not found
   */
  function getNodeMatchingOrd(node, ord) {
    if (!node) {
      return null;
    }

    if (ordUtil.equivalent(node.getNavOrd(), ord)) {
      return node;
    }

    return _.find(node.getNavChildren(), function (kid) {
      return getNodeMatchingOrd(kid, ord);
    });
  }

  /**
   * A navigation model that takes into account the user's nav file. If any
   * of the nodes in our node stack appear in the user's nav file, then the
   * stack will reflect that nav file structure up until the last node that
   * exists in the nav file. Nodes after that will follow a plain component
   * hierarchy structure.
   * 
   * @class
   * @memberOf module:mobile/util/nav
   * @extends module:mobile/util/nav.NavModel
   */
  var NavFileNavModel = function NavFileNavModel() {
    var that = this;

    NavModel.call(that);

    /**
     * A copy of the user's nav file, processed and stored as a plain 
     * NavModelNode. This allows us to query the nav file for ORDs 
     * synchronously (as opposed to using the methods off NavContainer which 
     * are asynchronous).
     * 
     * @field
     * @name niagara.util.nav.NavFileNavModel$flattenedNavFile
     * @private
     */
    that.$flattenedNavFile = undefined;

    //teeeeeeechnically this should all be a synchronous operation
    flattenNavFileNode(baja.nav.navfile.getRootNode(), function (flattened) {
      that.$flattenedNavFile = flattened;
    });
  };
  NavFileNavModel.prototype = Object.create(NavModel.prototype);
  NavFileNavModel.prototype.constructor = NavFileNavModel;

  /**
   * Processes an ORD into an ORD history that takes into account the user's
   * nav file. If any of the ORDs exist in the nav file, then the first ORD
   * in the resulting array will match the nav file root, and so on down
   * as far as we can go until we no longer match the nav file structure. ORDs
   * after leaving the nav file will be displayed as a simple component
   * hierarchy.
   *
   * @param {String|baja.Ord} ord the ord to make into an ord history
   * @returns {Array} an array of ORDs as strings
   */
  NavFileNavModel.prototype.makeNavFileOrdHistory = function makeNavFileOrdHistory(ord) {
    var that = this,
        ordHistory = ordUtil.makeOrdHistory(ord),
        len = ordHistory.length,
        backwardsHistory = [],
        forwardsHistory = [];

    //iterate backwards through ord
    baja.iterate(0, len, function (i) {
      var ordChunk = ordHistory[len - 1 - i],
          navFileNode = getNodeMatchingOrd(that.$flattenedNavFile, ordChunk);

      //this ord exists in the nav file - so everything from here on up the nav
      //file tree should come directly from the nav file
      if (navFileNode) {
        while (navFileNode) {
          backwardsHistory.push(navFileNode);
          navFileNode = navFileNode.getNavParent();
        }
        return true; //stop iteration - we don't care about the rest of our ORD
      } else {
        //still in component tree - continue on to next iteration, stepping up
        //one level in the hierarchy to see if we match anything in the nav
        //file yet
        backwardsHistory.push(makeNavModelNode(ordChunk));
      }
    });

    //put the ord history in forwards order
    len = backwardsHistory.length;
    baja.iterate(0, len, function (i) {
      forwardsHistory.push(backwardsHistory[len - 1 - i]);
    });

    return forwardsHistory;
  };

  /**
   * Adds an object, as a node, to our nav model. When adding a new node, the
   * entire nav model may be wiped clean and rebuilt, depending on whether the
   * new node exists in the user's nav file or not.
   *
   * @param {String|baja.Ord|baja.Complex|module:mobile/util/mobile/views/PageView} obj an
   * object to convert to a node and add
   * @returns {module:mobile/util/nav.NavFileNavModel} this
   */
  NavFileNavModel.prototype.add = function add(obj) {
    var node = makeNavModelNode(obj),
        that = this,
        toResolve = {},
        toResolveArray = [],
        ord = node.getNavOrd(),
        nodes = that.makeNavFileOrdHistory(ord),
        existingNode;

    that.pruneOtherBranch(node);

    existingNode = that.$nodes[nodes.length - 1] || nodes[nodes.length - 1];

    //be sure not to overwrite the icon/description from a nav file
    //node with the icon/description from a component or view - just reset the
    //value we're looking at
    if (existingNode && existingNode.$fromNavFile && ordUtil.equivalent(existingNode.getNavOrd(), node.getNavOrd())) {
      node = existingNode;
      node.setValue(obj);
    }

    nodes[nodes.length - 1] = node;
    that.$nodes.splice(0, nodes.length);
    that.$nodes = nodes.concat(that.$nodes);

    /*
     * When we first load a page, unless we're only looking at the station root
     * we will need to retrieve the components to obtain their display names
     * and icons for the navbar. But this is an asynchronous action and we
     * have to show *something* while it's taking place, so we start by simply
     * storing an ORD in the view list, then replace with the component once
     * it is retrieved. (We don't do this for the slot we're adding a view for
     * - there's no need.)
     */
    baja.iterate(0, that.$nodes.length, function (i) {
      var nodeValue = that.$nodes[i].getValue(),
          len = toResolveArray.length;

      if (nodeValue instanceof baja.Ord || typeof nodeValue === 'string') {
        //then make a record saying that we need to include this ORD in our
        //batch resolve
        toResolve['ord' + len] = {
          ord: nodeValue,
          origIndex: i
        };
        toResolveArray.push(nodeValue);
      }
    });

    if (toResolveArray.length) {
      //we have ORD sitting in our view stack - so let's replace them with
      //live components
      new baja.BatchResolve(toResolveArray).resolve({
        ok: function ok() {
          var objs = this.getTargetObjects();

          baja.iterate(objs, function (obj, i) {
            //replace the ORD in our view stack with the actual component
            var thing = toResolve['ord' + i];
            that.$nodes[thing.origIndex] = makeNavModelNode(obj);
          });

          /*
           * Once the stack is fully populated with placeholders, trigger a 
           * change event so the nav bar (if we have one) can redraw itself 
           * with proper display names and icons.
           */
          that.fireHandlers('change', baja.error, this);
        },
        fail: baja.error
      });
    }

    that.fireHandlers('change', baja.error, this);
    return this;
  };

  /**
   * Determines if two nav nodess are in the same branch of the tree, taking 
   * into account the structure of the user's nav file.
   * 
   * @param {module:mobile/util/nav.NavModelNode} node1
   * @param {module:mobile/util/nav.NavModelNode} node2
   * @returns {Boolean} true if the two nodes are in the same branch of
   * the tree
   */
  NavFileNavModel.prototype.sameBranch = function sameBranch(node1, node2) {
    var that = this,
        ord1 = node1.getNavOrd(),
        ord2 = node2.getNavOrd(),
        ordHistory1 = that.makeNavFileOrdHistory(ord1),
        ordHistory2 = that.makeNavFileOrdHistory(ord2),
        len1 = ordHistory1.length,
        len2 = ordHistory2.length,
        different;

    different = baja.iterate(0, Math.min(len1, len2), function (i) {
      var different = !ordUtil.equivalent(ordHistory1[i], ordHistory2[i]);
      if (different) {
        return true;
      }
    });

    return !different;
  };

  return {
    NavModel: NavModel,
    ComponentSpaceNavModel: ComponentSpaceNavModel,
    NavFileNavModel: NavFileNavModel
  };
});
