/**
 * @file A component view showing a list of radio buttons.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'jquery', 'jquerymobile', 'Promise', 'mobile/util/mobile/mobile', 'mobile/util/mobile/views/MobileView'], function (baja, $, jqm, Promise, mobileUtil, MobileView) {

  "use strict";

  var encodePageId = mobileUtil.encodePageId;

  /**
   * A View that displays the slots on a Complex in the form of a set of
   * JQM radio buttons.
   * 
   * @class
   * @name niagara.util.mobile.RadioButtonView
   * @extends niagara.util.mobile.MobileView
   */
  var RadioButtonView = baja.subclass(function () {
    baja.callSuper(RadioButtonView, this, arguments);
  }, MobileView);

  /**
   * Builds up the set of radio buttons. Iterates over the slots on the
   * component (filtered by `shouldIncludeSlot`) and appends the results of
   * `makeInput` and `makeLabel` for each one.
   *  
   * @name niagara.util.mobile.RadioButtonView#doLoad
   * @function
   * @param {baja.Complex} value a Complex to assemble into radio buttons
   */
  RadioButtonView.prototype.doLoad = function doLoad(value) {
    baja.strictArg(value, baja.Complex);

    var that = this,
        fieldset = $('<fieldset data-role="controlgroup" />'),
        filteredSlots = value.getSlots(function (slot) {
      return that.shouldIncludeSlot(slot);
    });

    baja.iterate(filteredSlots, function (slot) {
      fieldset.append(that.makeInput(value, slot)).append(that.makeLabel(value, slot));
    });

    this.jq().empty().append(fieldset).trigger('create');

    return Promise.resolve();
  };

  /**
   * Arms an event listener to call `selectionChanged` whenever the user
   * selects a different radio button.
   *  
   * @name niagara.util.mobile.RadioButtonView#doInitialize
   * @function
   */
  RadioButtonView.prototype.doInitialize = function (element) {
    var that = this;

    return MobileView.prototype.doInitialize.call(this, element).then(function () {
      element.on('change', 'input[type=radio]', function () {
        that.selectionChanged($(this).jqmData('slot'));
      });
    });
  };

  /**
   * Called whenever the user selects a different radio button.
   * 
   * @name niagara.util.mobile.RadioButtonView#selectionChanged
   * @function
   * @param {baja.Slot} slot the slot corresponding to the selected radio
   * button
   */
  RadioButtonView.prototype.selectionChanged = function selectionChanged(slot) {
    this.$selectedSlot = slot;
  };

  /**
   * Programmatically set the selected radio button. If the given slot does
   * not correspond to an available radio button, all radio buttons will be
   * unchecked.
   * 
   * @name niagara.util.mobile.RadioButtonView#setSelectedSlot
   * @function
   * @param {baja.Slot|String} slot the slot to select
   */
  RadioButtonView.prototype.setSelectedSlot = function setSelectedSlot(slot) {
    slot = String(slot);

    var pageId = encodePageId(slot),
        input = this.jq().find('input#' + pageId);

    slot = this.value().getSlot(slot);

    if (slot && input.length) {
      this.$selectedSlot = slot;
      input.prop('checked', true).checkboxradio('refresh');
    } else {
      delete this.$selectedSlot;
      this.jq().find('input[type=radio]').prop('checked', false);
    }

    this.selectionChanged(slot);
  };

  /**
   * Returns the currently selected slot.
   * 
   * @name niagara.util.mobile.RadioButtonView#getSelectedSlot
   * @function
   * @returns {baja.Slot} slot the currently selected slot
   */
  RadioButtonView.prototype.getSelectedSlot = function getSelectedSlot() {
    return this.$selectedSlot || this.jq().find('input:checked').jqmData('slot');
  };

  /**
   * Returns the currently selected value (equivalent to
   * `this.value().get(this.getSelectedSlot())`).
   * 
   * @name niagara.util.mobile.RadioButtonView#getSelectedValue
   * @function
   * @returns {baja.Value} the currently selected value
   */
  RadioButtonView.prototype.getSelectedValue = function getSelectedValue() {
    var slot = this.getSelectedSlot();
    if (slot) {
      return this.value().get(slot);
    }
  };

  /**
   * Creates the `<input>` element to be appended for each slot on the
   * `Complex` to be displayed.
   * 
   * @name niagara.util.mobile.RadioButtonView#makeInput
   * @function
   * @param {baja.Complex} complex the Complex whose slots are to be assembled
   * into radio buttons
   * @param {baja.Slot} slot the slot to make into a radio button input
   * @returns {jQuery} the assembled `<input>` element
   */
  RadioButtonView.prototype.makeInput = function (complex, slot) {
    slot = String(slot);
    var pageId = encodePageId(slot),
        nameId = encodePageId(complex.getName()),
        input = $('<input type="radio" />').attr('name', nameId).attr('id', pageId).attr('value', pageId);
    if (slot === String(this.$selectedSlot)) {
      input.prop('checked', true);
    }
    input.jqmData('slot', slot);
    return input;
  };

  /**
   * Creates the `<label>` element to be used for display after a certain
   * `<input>` radio button.
   * 
   * @name niagara.util.mobile.RadioButtonView#makeLabel
   * @function
   * @param {baja.Complex} complex the Complex whose slots are to be assembled
   * into radio buttons
   * @param {baja.Slot} slot the slot to make into a label
   * @returns {jQuery} the assembled `<label>` element
   */
  RadioButtonView.prototype.makeLabel = function (complex, slot) {
    var pageId = encodePageId(String(slot)),
        display = complex.getDisplay(slot),
        displayName = baja.SlotPath.unescape(String(slot)),
        label = $('<label data-theme="c"/>').attr('for', pageId);

    label.append($('<span/>').text(displayName)).append($('<br/>')).append($('<span/>').text(display));

    return label;
  };

  /**
   * Decides whether or not a certain slot should be included in the set of
   * radio buttons. By default, only slots that are Properties are included.
   * 
   * @name niagara.util.mobile.RadioButtonView#shouldIncludeSlot
   * @function
   * @param {baja.Slot} slot a slot to test for inclusion
   * @returns {Boolean} true if the slot should be displayed
   */
  RadioButtonView.prototype.shouldIncludeSlot = function shouldIncludeSlot(slot) {
    return slot.isProperty() && !slot.isAction() && !slot.isTopic();
  };

  return RadioButtonView;
});
