/**
 * @file A view to take up an entire JQM page.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module mobile/util/mobile/views/PageView
 */
define(['baja!', 'bajaux/Widget', 'jquery', 'jquerymobile', 'Promise', 'css!mobile/util/mobile/views/PageView'], function (baja, Widget, $, jqm, Promise) {

  "use strict";

  /**
   * A view designed to take up an entire jQuery mobile page. Its job is
   * twofold: 1) to create the HTML necessary to create a new JQM page, and
   * 2) to populate that page's content div with another View (e.g.
   * ListView).
   * 
   * In most cases there will be no need to call any functions on this class
   * directly - if subclassing, simply override them as needed and
   * `PageViewManager` should do the work for you.
   * 
   * @private
   * @class
   * @alias module:mobile/util/mobile/views/PageView
   * @extends module:bajaux/Widget
   * 
   * @param {Object} [options] options to customize this particular view
   * @param {Function} [options.contentViewConstructor] the constructor for
   * the View subclass to display in the page's content div
   * @param {String} [options.title] a title to display in the page's header
   * div - if omitted will fall back to `View.getName()`.
   */

  var PageView = function PageView(options) {
    Widget.call(this, 'mobile', 'PageView');

    options = baja.objectify(options, 'contentViewConstructor');
    this.contentViewConstructor = options.contentViewConstructor;
    this.$title = options.title;
    this.$ignoreProfileClasses = "";
  };
  PageView.prototype = Object.create(Widget.prototype);
  PageView.prototype.constructor = PageView;

  ////////////////////////////////////////////////////////////////
  // PageView basic behavior
  ////////////////////////////////////////////////////////////////

  /**
   * Ignores profile-related CSS classes in the page view's header div. This
   * essentially turns off the functionality whereby you can disable the
   * header / back button / etc using the user's web profile - by ignoring
   * one of the following CSS classes, the corresponding header element will
   * _always_ be shown. You may include any or all of the applicable
   * CSS classes in a single string separated by spaces.
   * 
   * - `profileHeader` - always show the page header
   * - `profileHeaderBack` - always show the back button whenever the page
   *   header is visible (hiding the entire header will still hide the back
   *   button)
   * - `profileHeaderCommands` - always show the commands button whenever the
   *   page header is visible (hiding the entire header will still hide the
   *   commands button)
   * 
   * If subclassing PageView, `this.$ignoreProfileClasses` will
   * be available as an instance variable after calling this method.
   * 
   * @param {String} classString a string list of CSS classes to ignore, 
   * separated by spaces
   */
  PageView.prototype.ignoreProfileClasses = function ignoreProfileClasses(classString) {
    this.$ignoreProfileClasses = classString || "";
  };

  /**
   * Creates a jQuery mobile page to hold a component view. When instantiating
   * a new view, this page's `div[data-role="content"]` element will be used as
   * its target div.
   * 
   * Profile-related CSS classes `profileHeader` and `profileHeaderBack` will
   * be applied to the appropriate elements in the JQM page's header div
   * (unless ignored via `ignoreProfileClasses`).
   * 
   * @see niagara.util.mobile.PageViewManager#initializeView
   * @see #ignoreProfileClasses
   * @returns {jQuery} an initialized JQM page
   */
  PageView.prototype.createPage = function createPage() {
    var pageDiv = $('<div class="ux-fullscreen-support" data-role="page" data-theme="c" />'),
        headerDiv = $('<div data-role="header" data-position="fixed" data-tap-toggle="false" data-theme="a" />').appendTo(pageDiv),
        backButton = $('<a href="#" class="profileHeader profileHeaderBack" data-rel="back" data-icon="arrow-l" data-iconpos="notext"></a>').appendTo(headerDiv),
        headerTitle = $('<h1 class="profileHeader viewName"/>').appendTo(headerDiv);

    $('<div data-role="content" class="ux-root ux-fullscreen"/>').appendTo(pageDiv);

    backButton.removeClass(this.$ignoreProfileClasses);
    headerTitle.removeClass(this.$ignoreProfileClasses);

    return pageDiv;
  };

  /**
   * Returns the header title for this page. If passed in as the `title`
   * parameter to the PageView's constructor, will use that - otherwise falls
   * back to `View.getName()`.
   * 
   * @returns {Promise} will be resolved with the view's display name
   */
  PageView.prototype.toDisplayName = function () {
    return Promise.resolve(this.$title || this.contentView && this.contentView.toDisplayName() || undefined);
  };

  /**
   * Finds and returns this view's enclosing JQM page.
   * 
   * @return {jQuery} the enclosing JQM page
   */
  PageView.prototype.getPage = function getPage() {
    return this.jq();
  };

  /**
   * Returns the header div for this JQM page.
   * 
   * @return {jQuery} the JQM page's header div
   */
  PageView.prototype.getHeaderDiv = function getHeaderDiv() {
    return this.jq().children(':jqmData(role=header)');
  };

  /**
   * Returns the content div for this JQM page.
   * 
   * @return {jQuery} the JQM page's content div
   */
  PageView.prototype.getContentDiv = function getContentDiv() {
    return this.jq().children(':jqmData(role=content)');
  };

  /**
   * Returns the footer div for this JQM page.
   * 
   * @return {jQuery} the JQM page's footer div
   */
  PageView.prototype.getFooterDiv = function getFooterDiv() {
    return this.jq().children(':jqmData(role=footer)');
  };

  /**
   * Returns any view parameters associated with the PageView's containing
   * JQM page.
   * 
   * @return {baja.ViewQuery} an object containing the view params
   */
  PageView.prototype.getViewQuery = function getViewQuery() {
    return this.viewQuery;
  };

  ////////////////////////////////////////////////////////////////
  // PageView override points
  ////////////////////////////////////////////////////////////////

  /**
   * This is where the PageView instantiates the sub-View that will display
   * inside its page's content div. A ListPageView would return a ListView,
   * a TablePageView would return a TableView, etc. Note that your subclass
   * might display a different type of content view depending on the type of
   * the value being displayed.
   * 
   * A content view constructor can be passed in directly to `PageView`'s
   * constructor as `options.contentViewConstructor`. If so, by default, that
   * constructor will be used to instantiate the content view. If no
   * constructor was provided, then this function _must_ be implemented
   * by a subclass, or an error will be thrown.
   * 
   * @param {baja.Value} value the value to be displayed in the page's content
   * div
   * @return {module:bajaux/Widget} the content view to be displayed
   * @throws {Error} if not implemented in a subclass
   */
  PageView.prototype.instantiateContentView = function instantiateContentView(value) {
    var Ctor = this.contentViewConstructor;
    if (Ctor) {
      return new Ctor();
    } else {
      throw new Error("instantiateContentView not implemented");
    }
  };

  /**
   * Returns true if the page view's content view is modified.
   * 
   * @return {Boolean} true if the content view is modified
   */
  PageView.prototype.isModified = function () {
    var contentView = this.contentView;

    return contentView && contentView.isModified();
  };

  /**
   * Readies a JQM page div for loading in a value. Since a PageView consists 
   * of a full JQM page (which JQM will enhance on its own), it just retrieves 
   * the `viewQuery` (if any) and the `pageId` from the JQM page being
   * displayed. No alterations will be made to the page structure.
   * 
   * Binds to any `modified.fieldeditor` events (triggered by field editors
   * when the user enters data) and calls `setModified()` accordingly.
   * 
   * @param {jQuery} element the div in which to insert this component
   * view's display div
   * @throws {Error} if attempting to build inside of a div that is not a
   * JQM page (data-role=page)
   */
  PageView.prototype.doInitialize = function doInitialize(element) {
    var that = this,
        pageData;

    if (element.jqmData('role') !== 'page') {
      throw new Error("PageView requires a JQM page (data-role='page')");
    }

    pageData = element.data('pageData');
    that.viewQuery = pageData && pageData.viewQuery;
    that.pageId = element.attr('id');
  };

  /**
   * When loading a new value, the PageView will instantiate its content view,
   * build content View's HTML into the PageView's content div, and then load
   * the content View with the desired value.
   * 
   * @param {baja.Value} initialValue The value initially loaded.
   */
  PageView.prototype.doLoad = function doLoad(initialValue) {
    //instantiate the content view, and build its html in our content div
    var that = this,
        contentView = that.contentView;

    return Promise.resolve(contentView && contentView.destroy()).then(function () {
      contentView = that.contentView = that.instantiateContentView(initialValue);

      var ord = that.getPage().data("ord");

      return Promise.all([ord ? contentView.resolve(ord) : Promise.resolve(initialValue), contentView.initialize(that.getContentDiv())]);
    }).then(function (params) {
      var value = params[0];

      // TOTAL HACK: if the resolved value's type is different from the
      // initially resolved value's type, use the initial value. This will 
      // keep the history app happy. Yes it's a complete kludge
      // but as it's Christmas...
      if (!value.getType().is(initialValue.getType())) {
        value = initialValue;
      }

      return contentView.load(value);
    }).then(function () {
      return contentView.toDisplayName();
    }).then(function (displayName) {
      that.getHeaderDiv().find('h1').text(displayName);
    });
  };

  PageView.prototype.destroy = function () {
    var that = this,
        contentView = that.contentView,
        args = arguments;

    return Promise.resolve(contentView && contentView.destroy()).then(function () {
      return Widget.prototype.destroy.apply(that, args);
    });
  };

  /**
   * Saves the content view. After the content view saves successfully, 
   * `this.isModified` will be reset to false as well.
   * 
   * @returns {Promise}
   */
  PageView.prototype.save = function save(params) {
    var that = this;

    return that.contentView.save(params).then(function (val) {
      return Promise.resolve(that.setModified(false)).then(function () {
        return val;
      });
    });
  };

  PageView.prototype.getCommandGroup = function () {
    return this.contentView.getCommandGroup();
  };

  PageView.prototype.doLayout = function () {
    return Promise.resolve(this.contentView && this.contentView.layout());
  };

  return PageView;
});
