/**
 * @file Functions relating to navbar management in Niagara mobile apps.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module mobile/util/mobile/nav
 */
define(['baja!', 'bajaux/icon/iconUtils', 'jquery', 'Promise', 'underscore'], function (baja, iconUtils, $, Promise, _) {

  "use strict";

  ////////////////////////////////////////////////////////////////
  //Utility functions
  ////////////////////////////////////////////////////////////////

  var toHtml = iconUtils.toHtml;

  ////////////////////////////////////////////////////////////////
  //NavBar
  ////////////////////////////////////////////////////////////////

  /**
   * This class represents a JQM navbar for navigating through a Bajascript
   * app. The buttons will be labeled with different stops in our nav model
   * so that we can quickly navigate back up/down to any point in our 
   * tree we already have open.
   * 
   * This class does not just have a single div - since we navigate back
   * and forth between property sheets using `$.mobile.changePage`, the
   * necessary navbar div will need to be dynamically generated and inserted
   * upon every new page view (at least until JQM implement persistent header
   * bars).
   * 
   * When viewing a property sheet in a long parent-child component chain,
   * the first and last navbar buttons will be replaced with "..." buttons.
   * Clicking these buttons will scroll the navbar back and forth (no page
   * reloads or property sheet traversal is involved) so that you can jump
   * back and forth quickly to your desired spot.
   * 
   * If a node has nav children (as in the case of a
   * `niagara.util.nav.NavFileNavModel`), the button in the nav bar will have a
   * down arrow indicating so. The node itself will be passed to the `click`
   * handler, so it is up to the app using this nav bar to examine the node for
   * children and handle the situation appropriately.
   * 
   * NavBar gets a Bajascript event mixin, so you can can attach to the
   * following events:
   * 
   * - `click` - a button in the nav bar was clicked. The argument to the
   *   handler will be an object literal containing the following properties:
   *   
   * -- `a`: the <a> link element that was clicked
   * -- `linkActive`: a boolean value indicating whether or not the link
   *     clicked was already in the highlighted state or not
   * -- `node`: the `niagara.util.nav.NavModelNode` backing the particular nav
   *    button that was clicked
   * 
   * @class
   * @memberOf module:mobile/util/mobile/nav
   * @param {niagara.util.nav.NavModel} navModel the nav model that will back
   * this nav bar
   */

  function NavBar(navModel) {
    var that = this;

    navModel.attach('change', function () {
      that.refresh(navModel.getSelectedNode());
    });

    this.navModel = navModel;
    this.currentIndices = {};
    this.LEFT_TEXT = "...";
    this.RIGHT_TEXT = "...";
  }

  /*
   * Give NavBar event capabilities
   */
  baja.event.mixin(NavBar.prototype);

  /**
   * Makes a nav bar div with `data-role="navbar"`, ready to be appended to a
   * JQM header div.
   *
   * @private
   * @returns {Promise} promise to be resolved with a nav bar div
   */
  NavBar.prototype.makeDiv = function makeDiv() {
    var that = this,
        currentIndices = that.currentIndices,
        startIndex = currentIndices.startIndex,
        endIndex = currentIndices.endIndex,
        navModel = that.navModel,
        range = _.range(startIndex, endIndex);

    var makeListItems = _.map(range, function (i) {
      var li = $('<li/>'),
          a = $('<a data-theme="b" />').appendTo(li),
          navNode,
          navChildren;

      if (i === startIndex && i > 0) {
        //scroll left button
        a.addClass("scrollLeft");
        a.text(that.LEFT_TEXT);
        return li;
      } else if (i === endIndex - 1 && i < navModel.size() - 1) {
        //scroll right button
        a.addClass("scrollRight");
        a.text(that.RIGHT_TEXT);
        return li;
      } else {
        a.addClass("componentLink");
        navNode = navModel.get(i);
        navChildren = navNode.getNavChildren();

        a.data('node', navNode);

        //if we have nav children, show a little down-arrow to indicate so
        if (navChildren.length) {
          a.data('icon', 'arrow-d');
          a.data('iconpos', 'right');
          a.data('children', navChildren);
        }

        return navNode.toDisplayName().then(function (displayName) {
          a.text(baja.SlotPath.unescape(displayName));
          return li;
        });
      }
    });

    return Promise.all(makeListItems).then(function (lis) {
      var div = $('<div class="nav" data-role="navbar" data-theme="b" />');

      $('<ul/>').html(lis).appendTo(div);

      return div;
    });
  };

  /**
   * Calculates the necessary indices for the property sheet we are currently
   * viewing. Returns an object literal with the following properties:
   * 
   * - `nodeIndex`: the index of the nav node viewed, in the context of the
   *   complete nav model (i.e. 0 = station root and increases the further you
   *   go to the right)
   * - `selectedIndex`: the index, in the context of the currently visible
   *   navbar buttons, of the highlighted button. A value of 0 means we are
   *   viewing the node referenced by the button on the far left, so highlight
   *   that one. Likewise, a value of 4 for the button on the far right.
   * - `startIndex`: the node index of the node to be displayed on the far left
   *   of the nav bar. If we are viewing the station root, this will likely be
   *   0.
   * - `endIndex`: the node index + 1 of the node to be displayed on the far
   *   right of the nav bar. If we have navigated to the furthest point down
   *   into the nav model, this will be `navModel.size()`.
   *
   * @private
   * 
   * @param {Number} nodeIndex the index of the property sheet to be viewed
   * @param {Number} [numberOfSheets] the number of property sheets 
   * currently loaded. If omitted, `this.navModel.size()` will be used.
   * @returns {Object} an object literal with index information
   */
  NavBar.prototype.getNavBarIndices = function getNavBarIndices(nodeIndex, numberOfSheets) {
    numberOfSheets = numberOfSheets || this.navModel.size();

    var startIndex = nodeIndex - 2,
        endIndex = nodeIndex + 3,
        delta = 0;

    if (endIndex > numberOfSheets) {
      delta = endIndex - numberOfSheets;
    } else if (startIndex < 0) {
      delta = startIndex;
    }
    endIndex -= delta;
    startIndex -= delta;

    startIndex = Math.max(startIndex, 0);
    endIndex = Math.min(endIndex, numberOfSheets);

    return {
      nodeIndex: nodeIndex,
      selectedIndex: nodeIndex - startIndex,
      startIndex: startIndex,
      endIndex: endIndex
    };
  };

  /**
   * Set the nav bar scrolling to the given node index.
   *
   * @private
   * 
   * @param {Number} index the index of the property sheet to scroll to
   */
  NavBar.prototype.scrollTo = function scrollTo(index) {
    this.currentIndices = this.getNavBarIndices(index);
  };

  /**
   * To be called when the left-side "..." scroll button is clicked. Scrolls
   * our nav bar two spaces to the left.
   * 
   * @private
   */
  NavBar.prototype.scrollLeft = function scrollLeft() {
    var index = Math.max(this.currentIndices.startIndex, 0);
    this.scrollTo(index);
  };

  /**
   * To be called when the right-side "..." scroll button is clicked. Scrolls
   * our nav bar two spaces to the right.
   *
   * @private
   */
  NavBar.prototype.scrollRight = function scrollRight() {
    var index = Math.min(this.currentIndices.endIndex - 1, this.navModel.size());
    this.scrollTo(index);
  };

  /**
   * Enables click handlers on our nav bar buttons for scrolling left and
   * right and for linking to other nav nodes.
   *
   * @private
   * 
   * @param {jQuery} navDiv the nav bar div (output from `makeDiv` - must
   * already have been appended to a JQM page)
   */
  NavBar.prototype.armHandlers = function armHandlers(navDiv) {
    var that = this;

    navDiv.on('click', 'a.componentLink', function () {
      var $this = $(this),
          node = $this.data('node');

      that.fireHandlers('click', baja.error, that, {
        a: $this,
        linkActive: $this.hasClass('ui-btn-active'),
        node: node
      });
    });

    navDiv.on('click', 'a.scrollLeft', function () {
      that.scrollLeft();
      that.replaceNavBar(that.navModel.getSelectedNode().getValue());
    });

    navDiv.on('click', 'a.scrollRight', function () {
      that.scrollRight();
      that.replaceNavBar(that.navModel.getSelectedNode().getValue());
    });
  };

  /**
   * Builds the nav bar div and appends it to the header of the current
   * JQM page. After this method returns, the nav bar will be fully
   * constructed, visible, and clickable.
   *
   * @private
   * 
   * @param {jQuery} page the JQM page in which to insert the navbar
   * @returns {Promise} promise to be resolved with the nav bar div, already
   * appended to the current active JQM page
   */
  NavBar.prototype.replaceNavBar = function replaceNavBar(page) {
    if (typeof page.getPage === 'function') {
      page = page.getPage();
    }

    var that = this,
        header = page.children('div:jqmData(role=header)'),
        oldNav = header.children('div:jqmData(role=navbar)');

    return that.makeDiv().then(function (nav) {
      if (oldNav.length) {
        oldNav.replaceWith(nav);
      } else {
        header.append(nav);
      }
      nav.navbar();
      that.armHandlers(nav);

      //set the icons - can only do this after navbar() is called unfortunately
      return Promise.all(_.map(nav.find('a.componentLink'), function (el) {
        var a = $(el),
            node = a.data('node'),
            icon = node.getNavIcon(),
            div = $('<span class="navbar-icon-container ui-btn-text"></span>').prependTo(a);
        return Promise.resolve(icon ? toHtml(icon) : '').then(function (html) {
          div.append($('<span class="bajaImage badgeSupport"/>').html(html));
        });
      })).then(function () {
        return nav;
      });
    });
  };

  /**
   * Refreshes the nav bar to reflect which node is currently being viewed.
   * This is `NavBar`'s one public method which should be called just after
   * navigating to a different page.
   *
   * @param {jQuery} page the JQM page in which the navbar lives
   * @returns {Promise} promise to be resolved after the navbar is up to date
   */
  NavBar.prototype.refresh = function (page) {
    //keep repeated calls all in order
    var that = this;
    return Promise.resolve(that.$refreshPromise).then(function () {
      var prom = that.doRefresh(page);
      that.$refreshPromise = prom;
      return prom;
    }).finally(function () {
      delete that.$refreshPromise;
    });
  };

  NavBar.prototype.doRefresh = function refresh(page) {
    var that = this;

    page = page && page.length ? page : that.$page;

    if (!page || !page.length) {
      return;
    }

    that.$page = page;

    that.scrollTo(that.navModel.getSelectedIndex());
    return that.replaceNavBar(page).then(function (nav) {
      var links = nav.find('ul li a'),
          selectedLink;

      if (links.length > 1) {
        //selected button should have its 'i have nav children' icon enabled -
        //all others should have it grayed out
        selectedLink = links.eq(that.currentIndices.selectedIndex);
        selectedLink.addClass('ui-btn-active');
        selectedLink.find('.ui-icon').removeClass('ui-disabled');
      }
    });
  };

  return {
    NavBar: NavBar
  };
});
