/**
 * @file
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global baja, niagara, describe, it, expect, beforeEach */

(function testScheduleBlock() {
  "use strict";

  var util = niagara.util,
      mock = niagara.test.mock.mock,
      makeDay = util.test.schedule.makeDay,
      hourMillis = util.time.MILLIS_IN_HOUR;

  describe("ScheduleBlock", function testScheduleBlock() {
    var day, block;
    beforeEach(function () {
      day = makeDay({
        start: hourMillis * 1,
        finish: hourMillis * 3,
        value: mock('baja:StatusString').addProp('value', 'block1to3')
      }, {
        start: hourMillis * 5,
        finish: hourMillis * 8,
        value: mock('baja:StatusString').addProp('value', 'block5to8')
      }, {
        start: hourMillis * 20,
        finish: hourMillis * 0,
        value: mock('baja:StatusString').addProp('value', 'block20toMid')
      });

      block = day.getBlocks()[1]; //block5to8
    });

    describe("ScheduleBlock object", function scheduleBlockObject() {
      it("can be created", function canBeCreated() {
        expect(block).toBeDefined();
      });
    });

    describe("equivalent", function equivalent() {
      var otherDay, block1, block2;

      beforeEach(function () {
        day = makeDay({
          start: hourMillis * 1,
          finish: hourMillis * 3,
          value: mock('baja:StatusString').addProp('value', 'block1to3')
        });

        otherDay = makeDay({
          start: hourMillis * 1,
          finish: hourMillis * 3,
          value: mock('baja:StatusString').addProp('value', 'block1to3')
        });
        block1 = day.getBlocks()[0];
        block2 = otherDay.getBlocks()[0];
      });

      it("checks start finish and value", function startFinishValue() {
        expect(block1.equivalent(block2)).toBeTruthy();
      });

      it("is false if start is different", function differentStart() {
        block2.setStart(baja.Time.make(12345));
        expect(block1.equivalent(block2)).toBeFalsy();
      });

      it("is false if finish is different", function differentFinish() {
        block2.setFinish(baja.Time.make(hourMillis * 23));
        expect(block1.equivalent(block2)).toBeFalsy();
      });

      it("is false if value is different", function differentValue() {
        block2.setValue(mock('baja:Thingy'));
        expect(block1.equivalent(block2)).toBeFalsy();
      });
    });

    describe("getFinish", function getFinish() {
      it("returns the TimeSchedule's finish time", function returnsFinish() {
        expect(block.getFinish().getTimeOfDayMillis()).toEqual(hourMillis * 8);
      });
    });

    describe("getStart", function getStart() {
      it("returns the TimeSchedule's start time", function returnsStart() {
        expect(block.getStart().getTimeOfDayMillis()).toEqual(hourMillis * 5);
      });
    });

    describe("getTimeSchedule", function getTimeSchedule() {
      it("returns the TimeSchedule object", function returnsTimeSchedule() {
        expect(block.getTimeSchedule()).toBe(day.value.getTime1());
      });
    });

    describe("getValue", function getValue() {
      it("returns the TimeSchedule's effective value", function returnsValue() {
        expect(block.getValue().getValue()).toEqual('block5to8');
      });
    });

    describe("overlaps", function overlaps() {
      it("checks for time overlap", function checksForOverlap() {
        expect(block.overlaps(baja.Time.make(hourMillis * 0), baja.Time.make(hourMillis * 5))).toBeFalsy();
        expect(block.overlaps(baja.Time.make(hourMillis * 0), baja.Time.make(hourMillis * 5 + 1))).toBeTruthy();
        expect(block.overlaps(baja.Time.make(hourMillis * 1), baja.Time.make(hourMillis * 6))).toBeTruthy();
        expect(block.overlaps(baja.Time.make(hourMillis * 6), baja.Time.make(hourMillis * 7))).toBeTruthy();
        expect(block.overlaps(baja.Time.make(hourMillis * 8 - 1), baja.Time.make(hourMillis * 0))).toBeTruthy();
        expect(block.overlaps(baja.Time.make(hourMillis * 8), baja.Time.make(hourMillis * 0))).toBeFalsy();
        expect(block.overlaps(baja.Time.make(hourMillis * 0), baja.Time.make(hourMillis * 0))).toBeTruthy();
      });
    });

    describe("setFinish", function setFinish() {
      it("sets the TimeSchedule's finish time", function setsFinish() {
        block.setFinish(baja.Time.make(hourMillis * 12));
        expect(block.getFinish().getTimeOfDayMillis()).toEqual(hourMillis * 12);
        expect(day.value.getTime1().getFinish().getTimeOfDayMillis()).toEqual(hourMillis * 12);
      });

      it("dies if start after finish", function startAfterFinish() {
        var start = block.getStart(),
            newFinish = baja.Time.make(hourMillis * 1);
        expect(function () {
          block.setFinish(newFinish);
        }).toThrow("finish time (" + newFinish + ") was before start time (" + start + ")");
      });

      it("sets sucessfully to 0", function setsTo0() {
        day.removeBlock(day.getBlocks()[2]); //avoid overlap with block20ToMid
        block.setFinish(baja.Time.make(0));
        expect(block.getFinish().getTimeOfDayMillis()).toEqual(0);
        expect(day.value.getTime1().getFinish().getTimeOfDayMillis()).toEqual(0);
      });

      it("dies in case of overlap", function overlaps() {
        var start = block.getStart(),
            newFinish = baja.Time.make(hourMillis * 22);
        expect(function () {
          block.setFinish(newFinish);
        }).toThrow("(" + start + "), (" + newFinish + ") overlaps with another block");
      });
    });

    describe("setStart", function setStart() {
      it("sets the TimeSchedule's start time", function setsStart() {
        block.setStart(baja.Time.make(hourMillis * 4));
        expect(block.getStart().getTimeOfDayMillis()).toEqual(hourMillis * 4);
        expect(day.value.getTime1().getStart().getTimeOfDayMillis()).toEqual(hourMillis * 4);
      });

      it("dies if start after finish", function startAfterFinish() {
        var newStart = baja.Time.make(hourMillis * 9),
            finish = block.getFinish();
        expect(function () {
          block.setStart(newStart);
        }).toThrow("finish time (" + finish + ") was before start time (" + newStart + ")");
      });

      it("sets sucessfully to 0", function setsTo0() {
        day.removeBlock(day.getBlocks()[0]); //avoid overlap with block1to3
        block.setStart(baja.Time.make(0));
        expect(block.getStart().getTimeOfDayMillis()).toEqual(0);
        expect(day.value.getTime1().getStart().getTimeOfDayMillis()).toEqual(0);
      });

      it("dies in case of overlap", function overlaps() {
        var newStart = baja.Time.make(hourMillis * 2),
            finish = block.getFinish();
        expect(function () {
          block.setStart(newStart);
        }).toThrow("(" + newStart + "), (" + finish + ") overlaps with another block");
      });
    });

    describe("setValue", function setValue() {
      it("sets the TimeSchedule's value", function setsValue() {
        var newValue = mock('baja:StatusString').addProp('value', 'superfly');
        block.setValue(newValue);
        expect(block.getValue().getValue()).toEqual('superfly');
        expect(day.value.getTime1().getEffectiveValue().getValue()).toEqual('superfly');
      });
    });
  });
})();
