/**
 * @file
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global baja, $, niagara, describe, it, expect, beforeEach */

(function testDay() {
  "use strict";

  var util = niagara.util,
      mock = niagara.test.mock.mock,
      makeDay = util.test.schedule.makeDay,
      hourMillis = util.time.MILLIS_IN_HOUR;

  describe("Day", function testDay() {
    var day, dayDiv;

    beforeEach(function () {
      day = makeDay({
        start: hourMillis * 1,
        finish: hourMillis * 3,
        value: mock('baja:StatusString').addProp('value', 'block1to3')
      }, {
        start: hourMillis * 5,
        finish: hourMillis * 8,
        value: mock('baja:StatusString').addProp('value', 'block5to8')
      }, {
        start: hourMillis * 20,
        finish: hourMillis * 0,
        value: mock('baja:StatusString').addProp('value', 'block20toMid')
      });
      dayDiv = $('<div/>');
    });

    describe("Day object", function dayObject() {
      it("can be created", function canBeCreated() {
        expect(makeDay()).toBeDefined();
      });
    });

    describe("addTimeSchedule", function addTimeSchedule() {
      beforeEach(function () {
        day.empty();
      });

      it("adds a new block", function addsNewBlock() {
        var start = baja.Time.make(0),
            finish = baja.Time.make(3600000),
            value = mock('baja:Thingy'),
            newTimeSchedule;

        expect(day.getBlocks().length).toEqual(0);
        day.addTimeSchedule(start, finish, value);
        expect(day.getBlocks().length).toEqual(1);

        newTimeSchedule = day.getBlocks()[0].getTimeSchedule();
        expect(newTimeSchedule.getStart()).toEqual(start);
        expect(newTimeSchedule.getFinish()).toEqual(finish);
        expect(newTimeSchedule.getEffectiveValue()).toEqual(value);

        expect(day.value.get('time')).toEqual(newTimeSchedule);
      });

      it("adds multiple", function addsMultiple() {
        day.addTimeSchedule(baja.Time.make(0), baja.Time.make(999), mock('baja:Thingy'));
        day.addTimeSchedule(baja.Time.make(1000), baja.Time.make(1999), mock('baja:Thingy'));
        day.addTimeSchedule(baja.Time.make(2000), baja.Time.make(2999), mock('baja:Thingy'));

        var ts1 = day.getBlocks()[0].getTimeSchedule(),
            ts2 = day.getBlocks()[1].getTimeSchedule(),
            ts3 = day.getBlocks()[2].getTimeSchedule();

        expect(ts1.getStart().getTimeOfDayMillis()).toEqual(0);
        expect(ts1.getFinish().getTimeOfDayMillis()).toEqual(999);
        expect(ts2.getStart().getTimeOfDayMillis()).toEqual(1000);
        expect(ts2.getFinish().getTimeOfDayMillis()).toEqual(1999);
        expect(ts3.getStart().getTimeOfDayMillis()).toEqual(2000);
        expect(ts3.getFinish().getTimeOfDayMillis()).toEqual(2999);

        expect(day.value.get('time')).toEqual(ts1);
        expect(day.value.get('time1')).toEqual(ts2);
        expect(day.value.get('time2')).toEqual(ts3);
      });
    });

    describe("allDayEvent", function allDayEvent() {
      it("fills up the whole day", function fillsUpDay() {
        var blocks = day.getBlocks();
        expect(blocks.length).toEqual(3);
        expect(blocks[0].getValue().getValue()).toEqual('block1to3');
        expect(blocks[1].getValue().getValue()).toEqual('block5to8');
        expect(blocks[2].getValue().getValue()).toEqual('block20toMid');

        day.allDayEvent(blocks[1].getValue());
        blocks = day.getBlocks();

        expect(blocks.length).toEqual(1);
        expect(blocks[0].getStart().getTimeOfDayMillis()).toEqual(0);
        expect(blocks[0].getFinish().getTimeOfDayMillis()).toEqual(0);
        expect(blocks[0].getValue().getValue()).toEqual('block5to8');
      });
    });

    describe("canMoveBlockTo", function canMoveBlockTo() {
      it("won't allow overlap", function wontAllowOverlap() {
        var block1to3 = day.getBlocks()[0];
        expect(day.canMoveBlockTo(block1to3, baja.Time.make(hourMillis * 4), baja.Time.make(hourMillis * 5))).toBeTruthy();
        expect(day.canMoveBlockTo(block1to3, baja.Time.make(hourMillis * 4), baja.Time.make(hourMillis * 5 + 1))).toBeFalsy();
        expect(day.canMoveBlockTo(block1to3, baja.Time.make(hourMillis * 8), baja.Time.make(hourMillis * 9))).toBeTruthy();
        expect(day.canMoveBlockTo(block1to3, baja.Time.make(hourMillis * 8 - 1), baja.Time.make(hourMillis * 9))).toBeFalsy();
      });

      it("doesn't care if block overlaps itself", function blockOverlapsItself() {
        var block1to3 = day.getBlocks()[0],
            block5to8 = day.getBlocks()[1];
        expect(day.canMoveBlockTo(block1to3, baja.Time.make(hourMillis * 2), baja.Time.make(hourMillis * 4))).toBeTruthy();
        expect(day.canMoveBlockTo(block5to8, baja.Time.make(hourMillis * 2), baja.Time.make(hourMillis * 4))).toBeFalsy();
      });
    });

    describe("copy", function copy() {
      it("creates a new copy", function createsACopy() {
        var daySchedule = mock('schedule:DaySchedule'),
            copied = day.copy(daySchedule),
            blocks = day.getBlocks(),
            copiedBlocks = copied.getBlocks();

        expect(copiedBlocks.length).toEqual(blocks.length);
        util.iterate(blocks, function (block, i) {
          expect(block.equivalent(copiedBlocks[i]));
        });
      });
    });

    describe("empty", function empty() {
      it("empties out the day", function empties() {
        expect(day.getBlocks().length).toEqual(3);
        day.empty();
        expect(day.getBlocks().length).toEqual(0);
      });
    });

    describe("equivalent", function equivalent() {
      it("checks all schedule blocks", function checksBlocks() {
        var daySchedule = mock('schedule:DaySchedule'),
            copied = day.copy(daySchedule);
        expect(day.equivalent(copied)).toBeTruthy();
        copied.getBlocks()[2].setFinish(baja.Time.make(hourMillis * 23));
        expect(day.equivalent(copied)).toBeFalsy();
      });
    });

    describe("removeBlock", function removeBlock() {
      it("removes a block", function removesABlock() {
        var blocks = day.getBlocks(),
            block5to8 = blocks[1];

        expect(blocks.length).toEqual(3);
        expect(block5to8.getValue().getValue()).toEqual('block5to8');

        day.removeBlock(block5to8);
        blocks = day.getBlocks();

        expect(blocks.length).toEqual(2);
        expect(blocks[0].getValue().getValue()).toEqual('block1to3');
        expect(blocks[1].getValue().getValue()).toEqual('block20toMid');
      });
    });
  });
})();
