var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

/**
 * @file Utilities for managing color gradients in the schedule app.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module mobile/schedule/ScheduleColors
 */
define(['baja!', 'jquery', 'underscore', 'mobile/util/color', 'mobile/util/gradient', 'mobile/schedule/util.schedule', 'lex!schedule'], function statusGradients(baja, $, _, colorUtil, gradientUtil, scheduleUtil, lexs) {

  "use strict";

  var applyLinearGradientBackground = gradientUtil.applyLinearGradientBackground,
      createLinearGradientCss = gradientUtil.createLinearGradientCss,
      hsvToRgb = colorUtil.hsvToRgb,
      parseRgba = colorUtil.parseRgba,
      _lexs = _slicedToArray(lexs, 1),
      scheduleLex = _lexs[0],
      stopCache = {},
      VERTICAL_ANGLE = 270,
      LUMA_THRESHOLD = 150,
      BOX_SHADOW_PREFS = ['-webkit-', '-moz-', '-ms-', '-o-', ''],
      TRUE_BACKGROUND_COLOR = parseRgba("#f065cb65"),
      FALSE_BACKGROUND_COLOR = parseRgba("#f0e17171"),
      NULL_BACKGROUND_COLOR = parseRgba("#f8cccccc"),
      DEFAULT_BACKGROUND_COLOR = parseRgba("#f88fbc8f"),
      DEFAULT_ENUM_COLORS = ["#f08c0000", //dark red
  "#f0387038", //dark green
  "#f0703870", //dark purple
  "#f025255a", //dark blue
  "#f01f6f70", //dark indigo
  "#f0704308", //dark orange
  "#f0606060", //gray

  "#f0e17171", //red
  "#f065cb65", //green
  "#f0cb7fcb", //purple
  "#f06c6ccb", //blue
  "#f054b7b8", //indigo
  "#f0c88435", //orange
  "#f0e1e171" //yellow
  ];

  ////////////////////////////////////////////////////////////////
  // Color parsing / gradients
  ////////////////////////////////////////////////////////////////

  /**
   * Get the background color of the schedule display, to use for alpha
   * blending calculations against the schedule block.
   *
   * @returns {niagara.util.color.Color}
   */


  function getScheduleBackgroundColor() {
    return parseRgba($('#main').css('background-color'));
  }

  /**
   * Transforms the given color into a pastelized gradient.
   *
   * @param {niagara.util.color.Color} color
   * @returns {Array} an array of stops to pass to niagara.util.gradient
   */
  function toGradientStops(color) {
    if (stopCache[color.css]) {
      return stopCache[color.css];
    }

    var h = color.h,
        s = color.s,
        v = color.v,
        a = color.alpha * 255,
        rgba1,
        rgba2,
        rgba3,
        rgba4,
        stops;

    //light/pastel at the top...
    rgba1 = hsvToRgb(h, s * 0.50, v * 1.35);
    rgba2 = hsvToRgb(h, s * 0.60, v * 1.25);

    //...darker/more vibrant at bottom
    rgba3 = hsvToRgb(h, s * 0.90, v * 1.05);
    rgba4 = hsvToRgb(h, s, v);

    //keep alpha
    if (typeof a === 'number') {
      rgba1 = parseRgba([rgba1.red, rgba1.green, rgba1.blue, a]);
      rgba2 = parseRgba([rgba2.red, rgba2.green, rgba2.blue, a]);
      rgba3 = parseRgba([rgba3.red, rgba3.green, rgba3.blue, a]);
      rgba4 = parseRgba([rgba4.red, rgba4.green, rgba4.blue, a]);
    }

    stops = stopCache[color.css] = [["0%", rgba1], ["30%", rgba2], ["90%", rgba3], ["97%", rgba4], ["100%", rgba3]];

    return stops;
  }

  /**
   * Calculates the luma value of the color. Does alpha blending to take into
   * account the background color and the fact that it might be partially
   * transparent.
   *
   * @param {niagara.util.color.Color} color
   * @return {Number}
   */
  function luma(color) {
    if (color.alpha < 255) {
      color = colorUtil.alphaBlend(color, getScheduleBackgroundColor());
    }

    return colorUtil.luma(color.red, color.green, color.blue);
  }

  /**
   * Get either black or white depending on which contrasts best with the
   * color's luma value.
   *
   * @param {niagara.util.color.Color} color
   * @returns {Object} a css object with `color` and `text-shadow` properties
   */
  function getContrastingTextCss(color) {
    if (luma(color) > LUMA_THRESHOLD) {
      return {
        color: 'black',
        'text-shadow': '0 1px 1px rgba(255, 255, 255, 0.7)'
      };
    }

    return {
      color: 'white',
      'text-shadow': '0 1px 1px rgba(0, 0, 0, 0.7)'
    };
  }

  /**
   * Calculates the gradient color stops and contrasting text color and
   * applies the CSS to the given div.
   *
   * @param {JQuery} div
   * @param {niagara.util.color.Color} blockColor
   */
  function applyGradient(div, blockColor) {
    var gradientStops = toGradientStops(blockColor),
        foregroundCss = getContrastingTextCss(gradientStops[0][1]);

    div.css(foregroundCss);

    if (div.parents('.schedule').length) {
      //main weekly schedule view = fancy and shiny
      applyLinearGradientBackground(div, VERTICAL_ANGLE, gradientStops);
    } else {
      //editable day sliders = background color only for speed
      div.css('background-color', createLinearGradientCss(VERTICAL_ANGLE, gradientStops).background);
    }
  }

  /**
   * Applies box shadow to schedule block div. This is a white inset box shadow
   * 50% of the block's height to give the shiny effect.
   *
   * @param {JQuery} blockDiv
   */
  function applyBoxShadow(blockDiv) {
    //only apply to main schedule div. too slow on interactive sliders
    if (!blockDiv.parents('.schedule').length) {
      return;
    }

    var height = blockDiv.height(),
        highlightHeight = height / 2,
        css = '0 2px 6px rgba(0,0,0,0.3), ' + 'inset 0 0px 0px 1px rgba(255,255,255,0.1)',
        highlightAlpha,
        prefs = BOX_SHADOW_PREFS,
        i;

    if (highlightHeight < 160) {
      //begin fading in the glass highlight effect at 160px, for a max of
      //0.06 alpha at 80px and less.
      highlightAlpha = Math.min(160 - highlightHeight, 80) / 80 * 0.07;
      css += ', inset 0 ' + highlightHeight + 'px ' + 'rgba(255,255,255,' + highlightAlpha + ')';
    }

    for (i = 0; i < prefs.length; i++) {
      blockDiv.css(prefs[i] + 'box-shadow', css);
    }
  }

  ////////////////////////////////////////////////////////////////
  // WeeklySchedule
  ////////////////////////////////////////////////////////////////

  /**
   * Given a schedule block value, walk up the parent tree until you find
   * a WeeklySchedule.
   *
   * @returns {baja.Component} `schedule:WeeklySchedule` or undefined
   */
  function getWeeklySchedule() {
    return scheduleUtil.getCurrentSchedule().value();
  }

  /**
   * Gets an enum schedule color configured in the lexicon, if any. Configured
   * as lexicon entries `EnumSchedule.colors.0` through however many.
   *
   * @param {String} key
   * @returns {String} color string from lexicon, or undefined
   */
  function getLexiconColor(key) {
    var value = scheduleLex.get(key);

    if (value) {
      try {
        return parseRgba(value);
      } catch (e) {
        baja.error('Invalid color in lexicon: ' + key + ' = ' + value);
      }
    }
  }

  /**
   * Retrieve the color configured directly on the schedule (via a `colors`
   * facet) for the given ordinal.
   *
   * @param {baja.Complex} weeklySchedule a `schedule:WeeklySchedule`
   * @param {Number} ordinal For an EnumSchedule, use the ordinal from the
   * DynamicEnum value. For a BooleanSchedule, use 0 = false, 1 = true.
   * @returns {niagara.util.color.Color|null}
   */
  function getConfiguredColor(weeklySchedule, ordinal) {
    var colors;

    try {
      if (weeklySchedule) {
        colors = weeklySchedule.get('facets').get('colors');
      }

      if (colors && colors.getType().is('baja:EnumRange')) {
        var tag = colors.get(ordinal).getTag();
        return parseRgba(baja.SlotPath.unescape(tag));
      }
    } catch (e) {
      return null;
    }
  }

  ////////////////////////////////////////////////////////////////
  // BooleanSchedule
  ////////////////////////////////////////////////////////////////

  function getLexiconColorFromBoolean(b) {
    return getLexiconColor('BooleanSchedule.colors.' + (b ? 1 : 0));
  }

  function getDefaultColorFromBoolean(b) {
    return b ? TRUE_BACKGROUND_COLOR : FALSE_BACKGROUND_COLOR;
  }

  /**
   * Red for false, green for true.
   *
   * @param {baja.Complex} statusBoolean `baja:StatusBoolean`
   * @returns {Object} color
   */
  function getColorFromStatusBoolean(statusBoolean) {
    var schedule = getWeeklySchedule(),
        b = statusBoolean.get('value');

    return getConfiguredColor(schedule, b ? 1 : 0) || getLexiconColorFromBoolean(b) || getDefaultColorFromBoolean(b);
  }

  ////////////////////////////////////////////////////////////////
  // EnumSchedule
  ////////////////////////////////////////////////////////////////

  /**
   * Get the enum's ordinal's index within all ordinals in the enum's range.
   * Protects against gaps in ordinal sequence when picking a color.
   *
   * @param {baja.DynamicEnum} en
   * @returns {Number}
   */
  function getOrdinalIndex(en) {
    if (en === baja.DynamicEnum.DEFAULT) {
      //default value was not encoded, so no range. however, must always be 0.
      return 0;
    }

    var ordinal = en.getOrdinal(),
        ordinals = en.getRange().getOrdinals(),
        index = _.indexOf(ordinals, ordinal);

    if (index === -1) {
      return ordinal;
    }

    return index;
  }

  /**
   * Get the default color for an enum schedule block. Pulled from
   * DEFAULT_ENUM_COLORS using the enum's ordinal. For ordinals 0 to
   * 6, returns a vibrant, gem-like color. For 7 to 13, returns a lighter
   * pastel color to contrast with first 7. Loops back around at 14.
   *
   * @param {baja.Enum} en
   * @returns {niagara.util.color.Color} color
   */
  function getDefaultColorFromEnum(en) {
    var colors = DEFAULT_ENUM_COLORS,
        len = colors.length,
        index = getOrdinalIndex(en),
        ord = index % (len * 2),
        color = colors[ord % len];

    return parseRgba(color);
  }

  /**
   * Gets an enum schedule color configured in the lexicon, if any. Configured
   * as lexicon entries `EnumSchedule.colors.0` through however many.
   *
   * @param {baja.Enum} en
   * @returns {String} color string from lexicon, or undefined
   */
  function getLexiconColorFromEnum(en) {
    var index = getOrdinalIndex(en);

    if (index >= 0) {
      return getLexiconColor('EnumSchedule.colors.' + index);
    }
  }

  /**
   * Calculate the block color for the given StatusEnum. If colors are
   * configured directly on the schedule object, uses those. If not, checks
   * the lexicon for configured colors. If not in the lexicon, pulls
   * from the default set of colors.
   *
   * @param {baja:Complex} statusEnum `baja:StatusEnum`
   * @returns {Array} hsva
   */
  function getColorFromStatusEnum(statusEnum) {
    var schedule = getWeeklySchedule(),
        en = statusEnum.getValue(),
        range = en.getRange();

    /*
     * If you manually type all your ordinals, THEN choose a range, the
     * range does not get saved on your schedule blocks, so we must pull it
     * from the schedule instead.
     */
    if (range === baja.EnumRange.DEFAULT) {
      range = schedule.get('facets').get('range');
      if (range && range !== baja.EnumRange.DEFAULT) {
        en = baja.DynamicEnum.make({
          ordinal: en.getOrdinal(),
          range: range
        });
      }
    }

    return getConfiguredColor(schedule, en.getOrdinal()) || getLexiconColorFromEnum(en) || getDefaultColorFromEnum(en);
  }

  ////////////////////////////////////////////////////////////////
  // ScheduleColors
  ////////////////////////////////////////////////////////////////

  /**
   * Get the appropriate null color. Check for `EnumSchedule.colors.null` or
   * `BooleanSchedule.colors.null`, then fall back to gray.
   *
   * @param {baja.Struct} statusValue `baja:StatusValue`
   * @return {niagara.util.color.Color}
   */
  function getNullColor(statusValue) {
    var color,
        type = statusValue.getType();

    if (type.is('baja:StatusEnum')) {
      color = getLexiconColor('EnumSchedule.colors.null');
    } else if (type.is('baja:StatusBoolean')) {
      color = getLexiconColor('BooleanSchedule.colors.null');
    }

    if (!color) {
      color = NULL_BACKGROUND_COLOR;
    }

    return color;
  }

  /**
   * Default coloring, used for StringSchedule and NumericSchedule blocks.
   *
   * @param {JQuery} div
   */
  function applyDefault(div) {
    applyGradient(div, DEFAULT_BACKGROUND_COLOR);
    div.css(getContrastingTextCss(DEFAULT_BACKGROUND_COLOR));
    applyBoxShadow(div);
  }

  /**
   * Calculates color gradients for a mobile schedule block. This could be
   * calculated and reused for multiple blocks that share the same value.
   *
   * @private
   * @class
   * @alias module:mobile/schedule/ScheduleColors
   * @param {baja.Complex} statusValue a `baja:StatusValue` from a schedule
   * block
   */
  function ScheduleColors(statusValue) {
    var that = this,
        backgroundColor,
        foregroundCss,
        apply,
        type = statusValue && statusValue.getType();

    if (!type || !type.is('baja:StatusValue')) {

      apply = applyDefault;
    } else if (statusValue.get('status').isNull()) {

      backgroundColor = getNullColor(statusValue);
      foregroundCss = getContrastingTextCss(backgroundColor);
      apply = function apply(div) {
        div.css({
          'background-color': backgroundColor.css
        });
        div.css(foregroundCss);
      };
    } else if (type.is('baja:StatusEnum')) {

      backgroundColor = getColorFromStatusEnum(statusValue);
      apply = function apply(div) {
        applyGradient(div, backgroundColor);
        applyBoxShadow(div);
      };
    } else if (type.is('baja:StatusBoolean')) {

      backgroundColor = getColorFromStatusBoolean(statusValue);
      apply = function apply(div) {
        applyGradient(div, backgroundColor);
        applyBoxShadow(div);
      };
    } else {

      apply = applyDefault;
    }

    /**
     * Applies the calculated colors to a schedule block div.
     *
     * @name niagara.schedule.scheduleGradients.ScheduleColors#apply
     * @function
     * @param {jQuery} div
     */
    //    that.apply = function (div) {
    //      if (!div.data('$hasGradient')) {
    //        apply(div);
    //        div.data('$hasGradient', true);
    //      }
    //    };
    //TODO: come back to this - make ScheduleBlock more modular and separate out text update, color update, and layout
    that.apply = apply;
  }

  /**
   * @namespace niagara.schedule.scheduleGradients
   */
  return ScheduleColors;
});
