/**
 * @file The Data Table model takes an ORD as the data source for the model and
 * resolves the ORD using either a default limit and offset or a declared
 * limit and offset. The values are retrieved from the data source ORD using
 * the baja.Ord.get method, which accepts a limit and offset as parameters.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Jason Spangler
 */

define(['baja!'], function (baja) {
  "use strict";

  var ORDER_BY_REGEX = /\s+order\s+by[\s\w]*$/i;

  /**
   * This method creates a column sort state object. The sort state contains
   * information about which columns are sorted and the order that they
   * are sorted in (ASC, DESC, or NONE if undefined).
   * 
   * @returns {Object} container object with column sort information.
   */
  function makeSortState() {
    var order, column;
    return {
      getColumn: function getColumn() {
        return column;
      },
      getOrder: function getOrder() {
        return order;
      },
      isOrderAsc: function isOrderAsc() {
        return order === 'ASC';
      },
      isOrderDesc: function isOrderDesc() {
        return order === 'DESC';
      },
      setColumn: function setColumn(col) {
        column = col;
      },
      setOrderAsc: function setOrderAsc() {
        order = 'ASC';
      },
      setOrderDesc: function setOrderDesc() {
        order = 'DESC';
      },
      clearState: function clearState() {
        order = undefined;column = undefined;
      }
    };
  }

  function stripOrderBy(bql) {
    return bql.replace(ORDER_BY_REGEX, '');
  }

  function removeViewQuery(queryList) {
    var newList = new baja.OrdQueryList();
    queryList.getCursor().each(function (value) {
      var scheme = value.getSchemeName();

      if (scheme !== 'view') {
        newList.add(value);
      }
    });

    return newList;
  }

  function getBqlOrderedOrd(queryList, columnName, ascOrDesc) {
    var orderBy = ' ORDER BY ' + columnName + ' ' + ascOrDesc,
        hadBql,
        ordStr;

    //if we have a view scheme, remove and append our bql query
    queryList.getCursor().each(function (value) {
      var scheme = value.getSchemeName();

      if (scheme === 'bql') {
        value.$body = stripOrderBy(value.$body) + orderBy;
        hadBql = true;
      }
    });

    ordStr = queryList.toString();

    if (!hadBql) {
      ordStr += '|bql:select *' + orderBy;
    }

    return baja.Ord.make(ordStr);
  }

  /**
   * This method checks if the current ORD requires a BQL sort query based
   * on the sort state of the data table view. If the sort state is defined,
   * a BQL sort query is appended to the given ORD, removing the view
   * query in the process as the ORD is intended for data gathering purposes
   * only.
   *  
   * @param {baja.Ord} ord - baja.Ord to append query to.
   * @param {Object} colSortState - column sort state to use to create BQL query.  
   * @returns {baja.Ord}
   */
  function appendSortQuery(ord, colSortState) {
    var queryList,
        columnName = colSortState.getColumn(),
        ascOrDesc = colSortState.getOrder();

    //if sorting, parse our ORD and append BQL query
    if (columnName !== undefined) {
      //modify the ord to use a BQL query to order the data based on this
      //column
      queryList = removeViewQuery(ord.parse());
      ord = getBqlOrderedOrd(queryList, columnName, ascOrDesc);
    }

    return ord;
  }

  /**
   * Data Table Model constructor including default limit and offset. 
   * 
   * @memberOf niagara.mobile.table
   * 
   * @param {Object} obj 
   *          - argument parameter for the constructor of the model. 
   * @param {baja.Ord} obj.ord
   *          - ORD to resolve that contains the data for this data model.
   * @param {Number} [obj.offset]
   *          - The initial offset to use when resolving the data ORD. By
   *            default, the value is 0.
   * @param {Number} [obj.limit]
   *          - The number of records fetched from the resolved ORD. This
   *            value is used in relation to the offset to retrieve the Next
   *            and Previous set of records.
   */
  function DataTableModel(obj) {
    var options = baja.objectify(obj, 'ord');

    this.dataOrd = baja.strictArg(options.ord, baja.lt('baja:Ord'));
    this.offset = options.offset || 0;
    this.limit = options.limit || 20;
    this.sortChangeCallback = undefined;
    this.colSortState = makeSortState();
    this.registry = {};
  }

  /**
   * Set the model sort state for the given column. Order can be given as
   * ASC, DESC, or NONE. 
   * 
   * @param {String} column - name of the column to sort on.
   * @param {String} order - can be ASC, DESC, or NONE.
   */
  DataTableModel.prototype.setSortOrder = function (column, order) {

    //TODO: verify that column is part of current data ORDs Schema

    this.colSortState.setColumn(column);
    if (order === 'ASC') {
      this.colSortState.setOrderAsc();
    } else if (order === 'DESC') {
      this.colSortState.setOrderDesc();
    } else {
      this.colSortState.clearState();
    }
  };

  /**
   * Retrieves the current sort order of the data table model.
   * 
   * @returns {Object} container object with column sort information for all 
   *                  data columns belonging to this data table model.
   */
  DataTableModel.prototype.getSortOrder = function () {
    return this.colSortState;
  };

  /**
   * Advance the offset by the current limit.
   */
  DataTableModel.prototype.next = function () {
    this.offset += this.limit;
  };

  /**
   * Retreat the offset by the current limit.
   */
  DataTableModel.prototype.previous = function () {
    this.offset -= this.limit;
    if (this.offset < 0) {
      this.offset = 0;
    }
  };

  /**
   * Get the set of records based on the current offset and limit.
   * 
   * @param {Object} [params]
   * @param {Object} [params.cursor] cursor configuration for Ord#get
   */
  DataTableModel.prototype.get = function (params) {

    var ord = appendSortQuery(this.dataOrd, this.colSortState),
        offset = this.offset,
        limit = this.limit,
        cursor = params && params.cursor || {},
        _each = cursor.each;

    return ord.get().then(function (table) {
      table.cursor({
        offset: offset,
        limit: limit,
        before: cursor.before,
        after: cursor.after,
        each: function each() {
          return _each && _each.apply(this, arguments);
        }
      });
      return table;
    });
  };

  /**
   * A data model can be sorted if it is backed by a history query or BQL query.
   */
  DataTableModel.prototype.isSortable = function () {
    var ord = this.dataOrd,
        queryList = ord.parse();

    if (queryList.get('history')) {
      return true;
    }

    if (queryList.get('bql')) {
      return true;
    }

    return false;
  };

  /**
   * Return the current offset of this table model.
   * 
   * @returns {Number} The current offset from 0 to indicate the index of 
   *                  where to being retrieving values from the data table model 
   *                  collection.
   */
  DataTableModel.prototype.getOffset = function () {
    return this.offset;
  };

  /**
   * Return the current record limit for this table model.
   * 
   * @returns {Number} current limit set for the number of records to retrieve.
   */
  DataTableModel.prototype.getLimit = function () {
    return this.limit;
  };

  /**
   * Reset the state of the table model.
   */
  DataTableModel.prototype.reset = function () {
    this.offset = 0;
    this.colSortState.clearState();
  };

  return DataTableModel;
});
