/**
 * @file
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson and Jason Spangler
 */

/*global niagara*/

define(['baja!history:HistoryRecord,history:NumericTrendRecord,' + 'history:BooleanTrendRecord,history:EnumTrendRecord,' + 'history:StringTrendRecord,history:AuditRecord,' + 'history:LogRecord', 'css!mobile/history/HistoryApp', 'css!mobile/history/chart/HistoryChart', 'baja!', 'lex!mobile', 'jquery', 'jquerymobile', 'Promise', 'underscore', 'mobile/util/ord', 'mobile/util/mobile/mobile', 'mobile/util/mobile/commands', 'mobile/util/mobile/dialogs', 'mobile/util/mobile/pages', 'mobile/fieldeditors/fieldeditors', 'mobile/util/mobile/views/ListView', 'mobile/util/mobile/views/PageView', 'mobile/util/mobile/views/PageViewManager', 'mobile/history/util.mobile.history'], function (types, unusedCss1, unusedCss2, baja, lexs, $, jqm, Promise, _, ordUtil, mobileUtil, commands, dialogs, pages, fe, ListView, PageView, PageViewManager, historyUtil) {

  // Use ECMAScript 5 Strict Mode
  "use strict";

  //local vars

  var mobileLex = lexs[0],
      historyManager,
      HistoryNavListView,
      HistoryNavPageView,
      timeRangeOnQuery = {},
      navPageTitles = {},


  //constants
  TABLE_BTN_DISPLAY_KEY = 'history.table.select',
      CHART_BTN_DISPLAY_KEY = 'history.chart.select',
      TIME_RANGE_BTN_DISPLAY_KEY = 'history.timeRange.select',
      START_TIME_FACET_NAME = 'START_TIME',
      END_TIME_FACET_NAME = 'END_TIME',
      TIME_RANGE_TYPE = 'baja:AbsTimeRange',
      NUMERIC_TREND_RECORD_TYPE = 'history:NumericTrendRecord',
      ENUM_TREND_RECORD_TYPE = 'history:EnumTrendRecord',
      BOOLEAN_TREND_RECORD_TYPE = 'history:BooleanTrendRecord',
      MOBILE_HISTORY_APP_TYPE = 'mobile:MobileHistoryApp',
      CHART_VIEW_ID = 'mobile:MobileHistoryAppChartView',
      TABLE_VIEW_ID = 'mobile:MobileHistoryAppTableView',
      FOLDER_TYPE = 'baja:Folder',
      SSH_TYPE = 'mobile:HistoryServerSideHandler',
      MTD_HISTORY_NAV_LIST = 'getHistoryNavList',
      MTD_NAV_CHILDREN = 'getNavChildren',
      MTD_ORD_DISPLAY_NAME = 'getOrdDisplayName',
      MTD_LIST_DEVICE_HISTORIES = 'getDeviceHistories',
      HISTORY_BTN_HTML = "<a data-role='button' class='ui-btn-up-c " + "ui-btn-corner-all' data-inline='true' " + "data-icon='{icon}' data-iconpos='notext' " + "data-theme='c'>{id}</a>";

  /**
   * This method redirects the current page to the given view ord.
   * 
   * @param el - HTML element representing our history slot.
   * @param viewQueryId - String view parameter - history or table
   */
  function redirect(el, viewQueryId) {
    var editor,
        li = el.parents('li'),
        expandedDiv = li.next('.expanded'),
        queryOn = li.find('.timeButton.ui-btn-active').length > 0,
        historyId = el.jqmData('historyId'),
        historyGroupRegex = /\/\/.*\/\//,
        query = historyId.$ord.replace(historyGroupRegex, "/");

    function doLink() {
      mobileUtil.linkToOrd("/ord?" + query + "%7Cview:" + viewQueryId);
    }

    if (queryOn) {
      editor = expandedDiv.find('.editor.baja-String').data('editor');
      editor.validate().then(function () {
        return editor.read();
      }).then(function (paramString) {
        query += paramString.replace(/\+/g, "%2B");
        doLink();
      }).catch(dialogs.error);
    } else {
      doLink();
    }
  }

  ////////////////////////////////////////////////////////////////
  //  History Device List View
  ////////////////////////////////////////////////////////////////

  /**
   * Create a unordered list of the devices and history group folders
   * within the base history space of our station. Each item in the list
   * represents either a history device or a history grouping that the
   * user can navigate through in order to reach specific history objects.
   * 
   * @param component
   * @param targetElement
   * @returns {HistoryNavListView}
   */
  HistoryNavListView = baja.subclass(function HistoryNavListView() {
    baja.callSuper(HistoryNavListView, this, arguments);
  }, ListView);

  HistoryNavListView.prototype.toDisplayName = function () {
    return Promise.resolve(mobileLex.get('history.title'));
  };

  HistoryNavListView.prototype.shouldIncludeSlot = function shouldIncludeSlot(slot) {
    return slot.getType().is(FOLDER_TYPE);
  };

  /**
   * create the first level of list items for display. These are the 
   * history groups and devices objects.
   * 
   * @param deviceId - String name of the device to which the histories belong.
   */
  HistoryNavListView.prototype.makeListItemParams = function (row, slotName) {
    var displayName = row.get('displayName'),
        id = row.get('id'),
        ord = row.get('ord'),
        recordType = row.get('recordType');

    if (id !== null) {
      timeRangeOnQuery[id] = false;
    }

    return Promise.resolve(recordType && baja.importTypes([recordType])).then(function () {
      return {
        ord: ord !== null ? ord : "history:" + id,
        title: displayName !== null ? displayName : id,
        id: slotName,
        recordType: recordType,
        linkable: id === null
      };
    });
  };

  /**
   * This is used to populate our expanded div created by our
   * doExpand method call when the time query button is clicked. This will pull
   * the start/end time facets off the slot, and instantiate a field editor for
   * a history query param string (either with start/end or period params).
   * @returns {Promise} promise to be resolved after the div is populated
   */
  HistoryNavListView.prototype.populateExpandedDiv = function (prop, expandedDiv) {
    var paramDiv = $('<div class="property"/>'),
        facets = prop.getFacets(),
        start = facets.get(START_TIME_FACET_NAME) || baja.AbsTime.DEFAULT,
        end = facets.get(END_TIME_FACET_NAME) || baja.AbsTime.now(),
        range = baja.$(TIME_RANGE_TYPE, { startTime: start, endTime: end }),
        recordType = expandedDiv.prev().data('params').recordType;

    recordType = baja.lt(recordType.toString());

    return fe.makeFor({
      value: historyUtil.toParamString(range),
      showDelta: recordType.is(NUMERIC_TREND_RECORD_TYPE),
      key: 'historyQueryParams',
      element: paramDiv.appendTo(expandedDiv)
    });
  };

  /**
   * Create history list items that display graph, table, and time range
   * buttons. 
   */
  HistoryNavListView.prototype.makeListItem = function (row, slot) {
    var that = this;

    return that.makeListItemParams(row, slot).then(function (params) {
      var li = $('<li class="property"/>').attr('id', params.id || ''),
          recordType = params.recordType,
          btnContainer = $('<div data-role="controlgroup" data-type="horizontal" />').addClass('historyViewSelector'),
          historyId = params.ord,
          isHistoryItem = recordType !== null,
          a;

      li.data('params', params);

      $('<label class="title" />') //label with property name
      .text(params.title || '').addClass(isHistoryItem ? 'historyViewSelectorLabel' : '').appendTo(li);

      $('<div class="display" />').text(params.display || '').appendTo(li);

      if (params.linkable || params.expandable) {
        a = $('<a />');
        a.jqmData('ord', params.ord);
        a.jqmData('title', params.title);
        navPageTitles[params.ord] = params.title;

        if (params.expandable) {
          a.addClass('expandable');
          li.attr('data-icon', 'arrow-d');
        } else {
          a.addClass('link');
        }
        li.wrapInner(a);
      }

      // check if our list item represents a history
      if (isHistoryItem) {
        recordType = baja.lt(recordType.toString()); //ensure a Type object
        btnContainer.appendTo(li);
        li.css('overflow', 'auto');

        // add chart button if our history record type is a trend record
        if (recordType.is(NUMERIC_TREND_RECORD_TYPE) || recordType.is(ENUM_TREND_RECORD_TYPE) || recordType.is(BOOLEAN_TREND_RECORD_TYPE)) {

          $(HISTORY_BTN_HTML.patternReplace({
            id: mobileLex.getSafe(CHART_BTN_DISPLAY_KEY),
            icon: 'chart'
          })).addClass('chartButton').jqmData('historyId', historyId).appendTo(btnContainer).click(function () {
            redirect($(this), CHART_VIEW_ID);
          });
        }

        //create table button
        $(HISTORY_BTN_HTML.patternReplace({
          id: mobileLex.getSafe(TABLE_BTN_DISPLAY_KEY),
          icon: 'grid'
        })).addClass('tableButton').jqmData('historyId', historyId).appendTo(btnContainer).click(function () {
          redirect($(this), TABLE_VIEW_ID);
        });

        //create time query button
        $(HISTORY_BTN_HTML.patternReplace({
          id: mobileLex.getSafe(TIME_RANGE_BTN_DISPLAY_KEY),
          icon: 'gear'
        })).addClass('timeButton').appendTo(btnContainer).click(function () {
          that.doExpand(slot.getName());
          $(this).toggleClass('ui-btn-active');
          timeRangeOnQuery[slot.getName()] = $(this).hasClass('ui-btn-active');
        });
      }

      return li;
    });
  };

  /**
   * Create the history nav list view
   */
  HistoryNavListView.prototype.makeListView = function (value) {
    var type = value.getType(),
        that = this;

    function buildRow(row) {
      return that.makeListItem(row, row.getPropertyInParent());
    }

    function assembleRows(val) {
      //iterate through our box collection rows and create list items
      var folders = val.getSlots().is('baja:Folder').toValueArray(),
          buildRows = _.map(folders, buildRow);
      return Promise.all(buildRows).then(function (rows) {
        return $('<ul data-role="listview" data-theme="c"/>').html(rows);
      });
    }

    if (type.is(MOBILE_HISTORY_APP_TYPE)) {
      //use server side call to resolve our groups and history devices
      return value.serverSideCall({
        typeSpec: SSH_TYPE,
        methodName: MTD_HISTORY_NAV_LIST
      }).then(assembleRows);
    } else {
      return assembleRows(value);
    }
  };

  /**
   * Always return false for isModified - this contains field editors, but
   * is not a writable view.
   * we don't want to pop up confirmations when navigating away
   */
  HistoryNavListView.prototype.isModified = function () {
    return false;
  };

  ////////////////////////////////////////////////////////////////
  //  History Nav Page View
  ////////////////////////////////////////////////////////////////

  /**
   * The history nav page view is the page view implementation of the 
   * history navigation view. 
   * 
   * @param {baja.Ord} ord the ORD of the nav view.
   */
  HistoryNavPageView = baja.subclass(function HistoryNavPageView(ord) {
    var that = this;

    baja.callSuper(HistoryNavPageView, this, arguments);

    baja.strictArg(ord, 'baja:String');
    this.$title = navPageTitles[ord];
    this.ord = ord;

    if (this.$title === undefined) {
      ordUtil.get(niagara.view.app.appOrd, function (component) {
        component.serverSideCall({
          typeSpec: SSH_TYPE,
          methodName: MTD_ORD_DISPLAY_NAME,
          value: baja.Ord.make(ord),
          ok: function ok(val) {
            that.$title = val;
            if (that.jq()) {
              that.getHeaderDiv().find('h1').text(val);
            }
          }
        });
      });
    }
  }, PageView);

  HistoryNavPageView.prototype.instantiateContentView = function () {
    return new HistoryNavListView();
  };

  /**
   * This method initializes our commands button in the header bar and is called
   * after the DOM is initialized.
   */
  HistoryNavPageView.prototype.createPage = function () {
    var page = PageView.prototype.createPage.call(this),
        homeBtn = commands.getCommandsButton(),
        pageHeader = page.children(':jqmData(role=header)');

    pageHeader.append(homeBtn);

    return page;
  };

  ////////////////////////////////////////////////////////////////
  // History View Manager
  ////////////////////////////////////////////////////////////////

  /**
   * The component view manager for the History app.
   * @returns {HistoryPageViewManager}
   */
  function HistoryPageViewManager() {
    PageViewManager.call(this);
  }
  HistoryPageViewManager.prototype = new PageViewManager();

  /**
   * This method is used to instantiate the view objects for this app. 
   * 
   * The table and chart views are checked for to instantiate DataTablePageView
   * and DataChartView objects with the ORD values necessary to implement
   * them.
   */
  HistoryPageViewManager.prototype.instantiateView = function (component, pageData) {
    var type = component.getType();

    if (type.is(MOBILE_HISTORY_APP_TYPE) || component.get('resultType') === 'deviceList' || component.get('resultType') === 'navNodeList') {
      return Promise.resolve(new HistoryNavPageView(pageData.ord));
    } else {
      return Promise.reject("Could not instantiate view for type " + type);
    }
  };

  /**
   * The doResolveOrd method is used to intercept a given ORD at time of 
   * resolution and perform any necessary server side calls that are 
   * particular to our history app.
   */
  HistoryPageViewManager.prototype.doResolveOrd = function doResolveOrd(ord) {
    var historyListRegex = /history:\/[^\W\/]*/,
        tableViewRegex = /display=table/,
        navNodeRegex = /history:\/\//,
        historyRootRegex = /history:\/\/$/,
        historyRootMatch = ord.match(historyRootRegex),
        tableMatch = ord.match(tableViewRegex),
        navNodeMatch = ord.match(navNodeRegex),
        match = ord.match(historyListRegex),
        deviceId;

    //append final slash for special history root case
    if (historyRootMatch) {
      ord += "/";
    }

    if (match && match[0].length === ord.length) {
      //ord is a list of device histories for a particular device ID
      deviceId = ord.substring(ord.indexOf('/') + 1);
      return ordUtil.get(niagara.view.app.appOrd).then(function (component) {
        return component.serverSideCall({
          typeSpec: SSH_TYPE,
          methodName: MTD_LIST_DEVICE_HISTORIES,
          value: deviceId
        });
      }).then(function (result) {
        return result.add({ slot: 'resultType', value: 'deviceList' }).then(function () {
          return result;
        });
      });
    } else if (tableMatch) {
      return Promise.resolve(baja.Ord.make(ord));
    } else if (navNodeMatch) {
      return ordUtil.get(niagara.view.app.appOrd).then(function (component) {
        return component.serverSideCall({
          typeSpec: SSH_TYPE,
          methodName: MTD_NAV_CHILDREN,
          value: baja.Ord.make(ord)
        });
      }).then(function (result) {
        return result.add({ slot: 'resultType', value: 'navNodeList' }).then(function () {
          return result;
        });
      });
    } else {
      return baja.Ord.make(ord).get();
    }
  };

  ////////////////////////////////////////////////////////////////
  //  Main
  ////////////////////////////////////////////////////////////////

  function initializeUI() {
    pages.register({
      pageshow: function pageshow() {
        $.mobile.changePage(baja.Ord.make(niagara.view.ord).toUri(), {
          transition: "none",
          changeHash: false
        });
      }
    });

    historyManager = new HistoryPageViewManager();
    historyManager.registerPages();
  }

  return initializeUI;
});
